/**
 * 
 */
package com.bmc.arsys.plugin.sample;

import java.math.BigDecimal;
import java.sql.Connection;
import java.sql.DatabaseMetaData;
import java.sql.DriverManager;
import java.sql.ResultSet;
import java.sql.ResultSetMetaData;
import java.sql.SQLException;
import java.sql.Statement;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;

import com.bmc.arsys.api.ARException;
import com.bmc.arsys.api.ArithmeticOrRelationalOperand;
import com.bmc.arsys.api.AttachmentFieldLimit;
import com.bmc.arsys.api.AttachmentValue;
import com.bmc.arsys.api.CharacterFieldLimit;
import com.bmc.arsys.api.Constants;
import com.bmc.arsys.api.DataType;
import com.bmc.arsys.api.DateInfo;
import com.bmc.arsys.api.DateOnlyFieldLimit;
import com.bmc.arsys.api.DateTimeFieldLimit;
import com.bmc.arsys.api.DecimalFieldLimit;
import com.bmc.arsys.api.Entry;
import com.bmc.arsys.api.EntryListFieldInfo;
import com.bmc.arsys.api.FieldLimit;
import com.bmc.arsys.api.IntegerFieldLimit;
import com.bmc.arsys.api.OutputInteger;
import com.bmc.arsys.api.QualifierInfo;
import com.bmc.arsys.api.RealFieldLimit;
import com.bmc.arsys.api.SortInfo;
import com.bmc.arsys.api.StatisticsResultInfo;
import com.bmc.arsys.api.StatusInfo;
import com.bmc.arsys.api.TimeOnlyFieldLimit;
import com.bmc.arsys.api.Timestamp;
import com.bmc.arsys.api.Value;
import com.bmc.arsys.api.VendorForm;
import com.bmc.arsys.pluginsvr.plugins.ARDBCPlugin;
import com.bmc.arsys.pluginsvr.plugins.ARPluginContext;
import com.bmc.arsys.pluginsvr.plugins.ARPluginInfo;
import com.bmc.arsys.pluginsvr.plugins.ARVendorField;

/**
 * A sample ARDBC plugin implementation as an example of how to use the sdk.
 * 
 * This plugin extends ARDBCPlugin object. Alternatively, one could implement the interface ARDBCPluggable.
 * 
 * This implementation fetches the jdbc connection information as config items using the plugin context. 
 * Uses that information to establish a jdbc connection to a SQL Server database. Provides a way of creating
 * vendor forms using tables in the underlying database. 
 */
public class ARDBCSample extends ARDBCPlugin {

    /*
     * In this sample, we are using a private object as a connection to the database. 
     * Each thread will create an instance of this object and initialize it, meaning,
     * there will be multiple connections established to the same database. 
     * 
     * There are two alternatives to this if one wants to share resources, connections
     * in this case. 
     * 
     * 1. Use static variables to hold your resource/resources. in which case the plugin
     * developer is responsible for making sure that those resources are thread safe.
     * 2. Implement the createInstance method, create connection in any thread safe way, 
     * and return the resource. This call is made once per thread. The resource returned
     * will be passed to every other call after that for that thread.
     * 
     * Note: If you are using the same class as two different plugins identified by two 
     * different ids,  then these will not be able to share any resources as they are 
     * loaded by two different classloaders, unless they were grouped in a plugingroup
     * in the plugin configuration, in which case they will be able to share resources.
     */
    
    // JDBC connection object
    private Connection con;
    private ARPluginInfo pluginInfo = new ARPluginInfo(NAME, this);
    
    //defaule names for plugin name and DB name configured in odbc driver
    private static final String NAME = "SAMPLE_JDBC_PLUGIN";
    private static final String DBNAME = "SamplePlugin";

    /**
     * Default constructor. Could as well do whatever is done in initialize. 
     * But as a default constructor, this will not have access to ARPluginContext object. 
     * In cases where that information is needed, better to do any initializations in 
     * initialize() 
     */
    public ARDBCSample() {
    }

    /** get list of forms from db and return a list of vendor forms */
    synchronized ArrayList<VendorForm> getTables() throws SQLException {
        /* Initializes  a statement object to the connection*/

        Statement stmt = con.createStatement();
        ArrayList<VendorForm> list = new ArrayList<VendorForm>();
        /* Holds the Meta-Data */
        DatabaseMetaData dbmd;
        /* ResultSet objects to hold the database table names and database column names */
        ResultSet rsTables = null;
        /* Retrieve meta-data from the database through the connection established*/
        dbmd = con.getMetaData();

        /* Get the table names in the ResultSet object using the 
         getTables(String catalog, String schemaPattern, String tableNamePattern, String[] types) 
         method matching the catalog, schema, table name and type criteria specified. 
         In this case all the tables are returned */
        rsTables = dbmd.getTables(null, null, null, null);

        /* Iterate through the ResultSet object and store the tables and columns in the QueryBean */

        while (rsTables.next()) {
            String tableName = rsTables.getString("TABLE_NAME");
            /* checks if it's a user-defined table*/
            if (tableName.startsWith("ar")) {
                VendorForm form = new VendorForm(ARDBCSample.NAME, tableName);
                list.add(form);
            }
        }
        rsTables.close();
        stmt.close();

        return list;
    }

    /** get list of fields for a form from db and return a list of vendor fields*/
    synchronized ArrayList<ARVendorField> getColumns(String tableName) throws SQLException {
        /* Initializes  a statement object to the connection*/

        Statement stmt = con.createStatement();
        ArrayList<ARVendorField> list = new ArrayList<ARVendorField>();
        String query = "Select TOP 1 * from " + tableName;
        ResultSet set = stmt.executeQuery(query);
        ResultSetMetaData metadata = set.getMetaData();
        int numCols = metadata.getColumnCount();
        for (int i = 1; i <= numCols; i++) {
            String name = metadata.getColumnName(i);
            int type = metadata.getColumnType(i);
            FieldLimit limit = null;
            switch (type) {
            case java.sql.Types.CHAR:
                type = Constants.AR_DATA_TYPE_CHAR;
                limit = new CharacterFieldLimit(1, 0, 0, null, null, 0);
                break;
            case java.sql.Types.DECIMAL:
                type = Constants.AR_DATA_TYPE_DECIMAL;
                limit = new DecimalFieldLimit("-99999999999999999999999999.99", "99999999999999999999999999.99", 3);
                break;
            case java.sql.Types.DOUBLE:
            case java.sql.Types.FLOAT:
            case java.sql.Types.REAL:
                type = Constants.AR_DATA_TYPE_REAL;
                limit = new RealFieldLimit(-1.845e+019, 1.845e+019, 6);
                break;
            case java.sql.Types.INTEGER:
                type = Constants.AR_DATA_TYPE_INTEGER;
                new IntegerFieldLimit(-2147483647, 2147483647);
                break;
            case java.sql.Types.BLOB:
                type = Constants.AR_DATA_TYPE_ATTACH;
                limit = new AttachmentFieldLimit(200000, 0, 0);
                break;
            case java.sql.Types.LONGVARCHAR:
                type = Constants.AR_DATA_TYPE_CHAR;
                limit = new CharacterFieldLimit(0, 0, 0, null, null, 0);
                break;
            case java.sql.Types.VARCHAR:
                type = Constants.AR_DATA_TYPE_CHAR;
                limit = new CharacterFieldLimit(255, 0, 0, null, null, 0);
                break;
            case java.sql.Types.DATE:
                type = Constants.AR_DATA_TYPE_DATE;
                limit = new DateOnlyFieldLimit();
                break;
            case java.sql.Types.TIME:
                type = Constants.AR_DATA_TYPE_TIME_OF_DAY;
                limit = new TimeOnlyFieldLimit();
                break;
            case java.sql.Types.TIMESTAMP:
                type = Constants.AR_DATA_TYPE_TIME;
                limit = new DateTimeFieldLimit();
                break;
            default:
                continue;
            }
            ARVendorField field = new ARVendorField(name, 0, type, limit);
            list.add(field);
        }
        set.close();
        stmt.close();
        return list;
    }

    /*
     * 
     * Get the record for the entry
     */
    Entry getRecord(String tableName, List<ARVendorField> fields, String entryId) throws SQLException{
        /* Initializes  a statement object to the connection*/

        Statement stmt = con.createStatement();
        Entry entry = null;
        String query = "Select * from " + tableName;
        /*try{
         query +=  " where ROWNUM = " + Integer.parseInt(arg5.getEntryID()) ;
        }
        catch(Exception e){}
        */
        ResultSet set = stmt.executeQuery(query);
        HashMap<Integer, Value> values = null;
        int rowNum = 1;
        if(set != null && fields != null){
            while (set.next()) {
                String id = "";
                values = new HashMap<Integer, Value>();
                for(int i = 0; i < fields.size(); i ++){
                    ARVendorField field = fields.get(i);
                    if(field.getFieldId() == 1){
                        String value = Integer.toString(rowNum++);
                        id = value;
                        Value val = new Value(value, DataType.toDataType(field.getDataType()));
                        values.put(new Integer(field.getFieldId()), val);
                        continue;
                    }
                    String value = set.getString(field.getName());
                    Value val = new Value(value, DataType.toDataType(field.getDataType()));
                    values.put(new Integer(field.getFieldId()), val);                
                }
                if(id.equals(entryId))
                    break;
            }
            entry = new Entry(values);
            set.close();
        }
        stmt.close();
        return entry;
    }
    
    /*
     * 
     * Get records from the entry
     */
    
    List<Entry> getRecords(String tableName, List<ARVendorField> fields, QualifierInfo qual) throws SQLException{
        ArrayList<Entry> entries = new ArrayList<Entry>();
        /* Initializes  a statement object to the connection*/
        Statement stmt = con.createStatement();
        String query = "Select * from " + tableName; //to get where string from qual later
        ResultSet set = stmt.executeQuery(query);
        int rowNum = 1;
        if(set != null && fields != null){
            while (set.next()) {
            	HashMap<Integer, Value> values = new HashMap<Integer, Value>();
                for(int i = 0; i < fields.size(); i ++){
                    ARVendorField field = fields.get(i);
                    if(field.getFieldId() == 1){
                        String value = Integer.toString(rowNum++);
                        Value val = new Value(value, DataType.toDataType(field.getDataType()));
                        values.put(new Integer(field.getFieldId()), val);
                        continue;
                    }
                    String value = set.getString(field.getName());
                    Value val = new Value(value, DataType.toDataType(field.getDataType()));
                    values.put(new Integer(field.getFieldId()), val);
                }
                Entry entry = new Entry(values);
                entries.add(entry);
            }
            set.close();
        }
        stmt.close();
        return entries;
    }
    
    String createRecord(ARPluginContext context, String tableName, List<ARVendorField> fields, 
            Entry entry) throws SQLException {
        String key = null;
        Statement stmt = con.createStatement();
        if(tableName!= null && fields != null && entry != null){
            String query = "Insert into " + tableName + " ("; //to get where string from qual later
            for(int j = 0; j < fields.size(); j ++){
               ARVendorField field1 = fields.get(j);
               if(field1.getFieldId() == 1)
                   continue;
               query += field1.getName();
               if(j + 1 < fields.size())
                   query+=",";
            }
            query +=  ") Values (";
            for(int i = 0; i < fields.size(); i ++){
                ARVendorField field = fields.get(i);
                Value value = entry.get(field.getFieldId());
                if(field.getFieldId() == 1){
                    continue;
                }
                switch(field.getDataType()){
                case Constants.AR_DATA_TYPE_CHAR:
                    query += " '" + (String)value.getValue() + "' ";
                    break;
                case Constants.AR_DATA_TYPE_DECIMAL:
                    query += " " + ((BigDecimal)value.getValue()).doubleValue() + "  ";
                    break;
                case Constants.AR_DATA_TYPE_REAL:
                    query += " " + ((Double)value.getValue()).doubleValue() + "  ";
                    break;
                case Constants.AR_DATA_TYPE_INTEGER:
                    query += " " + ((Integer)value.getValue()).intValue() + " ";
                    break;
                case Constants.AR_DATA_TYPE_ATTACH:
                    query += " " + (value.getValue()) + " ";
                    break;
                case Constants.AR_DATA_TYPE_DATE:
                    query += " " + ((DateInfo)value.getValue()).getValue() + " ";
                    break;
                case Constants.AR_DATA_TYPE_TIME:
                    query += " " + ((Timestamp)value.getValue()).getValue() + " ";
                    break;
                case Constants.AR_DATA_TYPE_TIME_OF_DAY:
                    query += " " + ((com.bmc.arsys.api.Time)value.getValue()).getValue() + " ";
                    break;
                default:
                    continue;
                }
                if(i + 1 < fields.size())
                    query+=",";
            }
            query += ")";
            int row = stmt.executeUpdate(query);
            if(row > 0)
                key = Integer.toString(row);
        }
        stmt.close();
        return key;        
    }



    /* (non-Javadoc)
     * @see com.bmc.arsys.pluginsvr.plugins.ARDBCPlugin#cancel(com.bmc.arsys.pluginsvr.plugins.ARPluginContext, java.lang.Object, long)
     */
    @Override
    public void cancel(ARPluginContext context, long transid) throws ARException {
        context.logMessage(pluginInfo, ARPluginContext.PLUGIN_LOG_LEVEL_INFO, "cancel()");
        super.cancel(context, transid);
    }

    /* (non-Javadoc)
     * @see com.bmc.arsys.pluginsvr.plugins.ARDBCPlugin#commit(com.bmc.arsys.pluginsvr.plugins.ARPluginContext, java.lang.Object, long)
     */
    @Override
    public void commit(ARPluginContext context, long transid) throws ARException {
        context.logMessage(pluginInfo, ARPluginContext.PLUGIN_LOG_LEVEL_INFO, "commit()");
        super.commit(context, transid);
    }

    /* (non-Javadoc)
     * @see com.bmc.arsys.pluginsvr.plugins.ARDBCPlugin#createEntry(com.bmc.arsys.pluginsvr.plugins.ARPluginContext, java.lang.Object, java.lang.String, java.util.List, long, com.bmc.arsys.api.Entry)
     */
    @Override
    public String createEntry(ARPluginContext context, String tableName, List<ARVendorField> fields, long transid, Entry newEntry) throws ARException {
        context.logMessage(pluginInfo, ARPluginContext.PLUGIN_LOG_LEVEL_INFO, "createEntry()");
        String entry = null;
        try {
            entry = this.createRecord(context, tableName,fields,newEntry);
        } catch (SQLException e) {
            new ARException(Arrays.asList(new
        			StatusInfo(Constants.AR_RETURN_ERROR, 1007, "Error creating entry for table " + tableName, e.getMessage())));
        }
        return entry;

    }

    /* (non-Javadoc)
     * @see com.bmc.arsys.pluginsvr.plugins.ARDBCPlugin#deleteEntry(com.bmc.arsys.pluginsvr.plugins.ARPluginContext, java.lang.Object, java.lang.String, java.util.List, long, com.bmc.arsys.api.String)
     */
    @Override
    public void deleteEntry(ARPluginContext context, String tableName, List<ARVendorField> fields, long transid, String entryId) throws ARException {
        context.logMessage(pluginInfo, ARPluginContext.PLUGIN_LOG_LEVEL_INFO, "deleteEntry()");
    }

    /* (non-Javadoc)
     * @see com.bmc.arsys.pluginsvr.plugins.ARDBCPlugin#getBLOB(com.bmc.arsys.pluginsvr.plugins.ARPluginContext, java.lang.Object, java.lang.String, java.util.List, long, com.bmc.arsys.api.EntryKey, long)
     */
    @Override
    public AttachmentValue getBLOB(ARPluginContext context, String tableName, List<ARVendorField> fields, long transid, String entryId,
            int fieldId) throws ARException {
        context.logMessage(pluginInfo, ARPluginContext.PLUGIN_LOG_LEVEL_INFO, "getBlob()");
        return super.getBLOB(context, tableName, fields, transid, entryId, fieldId);
    }

    /* (non-Javadoc)
     * @see com.bmc.arsys.pluginsvr.plugins.ARDBCPlugin#getEntry(com.bmc.arsys.pluginsvr.plugins.ARPluginContext, java.lang.Object, java.lang.String, java.util.List, long, com.bmc.arsys.api.EntryKey, int[])
     */
    @Override
    public Entry getEntry(ARPluginContext context, String tableName, List<ARVendorField> fields, long transid, String entryId,
            int[] fieldIds) throws ARException {
        context.logMessage(pluginInfo, ARPluginContext.PLUGIN_LOG_LEVEL_INFO, "getEntry()");
        Entry entry = null;
        try {
            entry = this.getRecord(tableName, fields, entryId);
        } catch (SQLException e) {
            StatusInfo info = new StatusInfo(2, 1004, "Error fetching entry " + entryId + " from table " + tableName, e.getMessage());
            ArrayList<StatusInfo> status = new ArrayList<StatusInfo>();
            status.add(info);
            throw new ARException(status);
        }
        if(entry == null){
        	new ARException(Arrays.asList(new
        			StatusInfo(Constants.AR_RETURN_ERROR, 1006, "No such entry " + entryId + " for table " + tableName, "")));
        }
        return entry;
    }

    /* (non-Javadoc)
     * @see com.bmc.arsys.pluginsvr.plugins.ARDBCPlugin#getEntryStatistics(com.bmc.arsys.pluginsvr.plugins.ARPluginContext, java.lang.Object, java.lang.String, java.util.List, long, com.bmc.arsys.api.QualifierInfo, com.bmc.arsys.api.ArithmeticOrRelationalOperand, int, int[])
     */
    @Override
    public List<StatisticsResultInfo> getEntryStatistics(ARPluginContext context, String tableName, List<ARVendorField> fields,
            long transid, QualifierInfo qual, ArithmeticOrRelationalOperand arithOp, int statistic, int[] groupBy) throws ARException {
        context.logMessage(pluginInfo, ARPluginContext.PLUGIN_LOG_LEVEL_INFO, "getEntryStatistics()");
        return super.getEntryStatistics(context, tableName, fields, transid, qual, arithOp, statistic, groupBy);
    }

    /* (non-Javadoc)
     * @see com.bmc.arsys.pluginsvr.plugins.ARDBCPlugin#getListEntryWithFields(com.bmc.arsys.pluginsvr.plugins.ARPluginContext, java.lang.Object, java.lang.String, java.util.List, long, com.bmc.arsys.api.QualifierInfo, java.util.List, java.util.List, int, int, java.lang.Integer)
     */
    @Override
    public List<Entry> getListEntryWithFields(ARPluginContext context, String tableName, List<ARVendorField> fields, long transid,
            QualifierInfo qual, List<SortInfo> sortList, List<EntryListFieldInfo> fieldList, int startAt, int maxReturn, OutputInteger numMatches) throws ARException {
        context.logMessage(pluginInfo, ARPluginContext.PLUGIN_LOG_LEVEL_INFO, "getListEntryWithFields()");
        List<Entry> entries = new ArrayList<Entry>();
        try {
            entries = this.getRecords(tableName, fields, qual);
        } catch (SQLException e) {
        	new ARException(Arrays.asList(new
        			StatusInfo(Constants.AR_RETURN_ERROR, 1004, "Error fetching entries for given qualification from table " + tableName, e.getMessage())));

        }
        if(entries == null || entries.size() == 0){
            new ARException(Arrays.asList(new
        			StatusInfo(Constants.AR_RETURN_ERROR, 1005, "No entries for table " + tableName, "")));
        }
        return entries;
    }

    /* (non-Javadoc)
     * @see com.bmc.arsys.pluginsvr.plugins.ARDBCPlugin#getListForms(com.bmc.arsys.pluginsvr.plugins.ARPluginContext, java.lang.Object)
     */
    @Override
    public List<VendorForm> getListForms(ARPluginContext context) throws ARException {
        context.logMessage(pluginInfo, ARPluginContext.PLUGIN_LOG_LEVEL_INFO, "getListForms()");
        ArrayList<VendorForm> list = null;
        try {
            list = this.getTables();
        } catch (SQLException e) {
            new ARException(Arrays.asList(new
        			StatusInfo(Constants.AR_RETURN_ERROR, 1004, "Error fetching table names from the datasource", e.getMessage())));
        }
        return list;
    }

    /* (non-Javadoc)
     * @see com.bmc.arsys.pluginsvr.plugins.ARDBCPlugin#getMultipleFields(com.bmc.arsys.pluginsvr.plugins.ARPluginContext, java.lang.Object, com.bmc.arsys.api.VendorForm)
     */
    @Override
    public List<ARVendorField> getMultipleFields(ARPluginContext context, VendorForm form) throws ARException {
        context.logMessage(pluginInfo, ARPluginContext.PLUGIN_LOG_LEVEL_INFO, "getMultipleFields()");
        ArrayList<ARVendorField> list = null;
        try {
            list = this.getColumns(form.getTableName());
        } catch (SQLException e) {
            new ARException(Arrays.asList(new
        			StatusInfo(Constants.AR_RETURN_ERROR, 1003, "Error fetching Columnnames for the table " + form.getTableName(), e.getMessage())));
        }
        return list;
    }

    /* (non-Javadoc)
     * @see com.bmc.arsys.pluginsvr.plugins.ARDBCPlugin#rollback(com.bmc.arsys.pluginsvr.plugins.ARPluginContext, java.lang.Object, long)
     */
    @Override
    public void rollback(ARPluginContext context, long transid) throws ARException {
        context.logMessage(pluginInfo, ARPluginContext.PLUGIN_LOG_LEVEL_INFO, "rollBack()");
    }

    /* (non-Javadoc)
     * @see com.bmc.arsys.pluginsvr.plugins.ARDBCPlugin#setBLOB(com.bmc.arsys.pluginsvr.plugins.ARPluginContext, java.lang.Object, java.lang.String, java.util.List, long, com.bmc.arsys.api.EntryKey, long, com.bmc.arsys.api.AttachmentInfo)
     */
    @Override
    public void setBLOB(ARPluginContext context, String tableName, List<ARVendorField> fields, long transid, String entryId,
            int fieldId, AttachmentValue blob) throws ARException {
        context.logMessage(pluginInfo, ARPluginContext.PLUGIN_LOG_LEVEL_INFO, "setBlob()");
    }

    /* (non-Javadoc)
     * @see com.bmc.arsys.pluginsvr.plugins.ARDBCPlugin#setEntry(com.bmc.arsys.pluginsvr.plugins.ARPluginContext, java.lang.Object, java.lang.String, java.util.List, long, com.bmc.arsys.api.Entry, com.bmc.arsys.api.Timestamp)
     */
    @Override
    public void setEntry(ARPluginContext context, String tableName, List<ARVendorField> fields, long transid, String entryId,
            Entry entry, Timestamp lastModified) throws ARException {
        context.logMessage(pluginInfo, ARPluginContext.PLUGIN_LOG_LEVEL_INFO, "setEntry()");
    }

    /* (non-Javadoc)
     * @see com.bmc.arsys.pluginsvr.plugins.ARPlugin#initialize(com.bmc.arsys.pluginsvr.plugins.ARPluginContext, java.lang.String[])
     */
    @Override
    public void initialize(ARPluginContext context) throws ARException {
        context.logMessage(pluginInfo, ARPluginContext.PLUGIN_LOG_LEVEL_INFO, "initialize()");
        //default connection information
        String url = "jdbc:odbc:" + DBNAME;
        String driver = "sun.jdbc.odbc.JdbcOdbcDriver";
        String user = "sa";
        String password = "";    
        try {
            //now fetch the connection information provided in the configuration
            String item = context.getConfigItem(pluginInfo, "dburl");
            if(item != null && !item.equals(""))
                url = item;
            item = context.getConfigItem(pluginInfo, "driver");
            if(item != null && !item.equals(""))
                driver = item;
            item = context.getConfigItem(pluginInfo, "user");
            if(item != null && !item.equals(""))
                user = item;
            item = context.getConfigItem(pluginInfo, "password");
            if(item != null)
                password = item;
            
            /* Load the jdbc-odbc driver */
            Class.forName(driver);
            /* Open a connection to the odbc data source entered by the user */
            con = DriverManager.getConnection(url, user, password);
        } catch (Exception e) {
            new ARException(Arrays.asList(new
        			StatusInfo(Constants.AR_RETURN_ERROR, 1000, "Error establishing jdbc connection to the datasource " + url, e.getMessage())));
        }
    }

    /* (non-Javadoc)
     * @see com.bmc.arsys.pluginsvr.plugins.ARPlugin#onEvent(com.bmc.arsys.pluginsvr.plugins.ARPluginContext, int)
     */
    @Override
    public void onEvent(ARPluginContext contexxt, int eventId) throws ARException {
        contexxt.logMessage(pluginInfo, ARPluginContext.PLUGIN_LOG_LEVEL_INFO, "onEvent()");
        super.onEvent(contexxt, eventId);
    }


    /* (non-Javadoc)
     * @see com.bmc.arsys.pluginsvr.plugins.ARPlugin#terminate(com.bmc.arsys.pluginsvr.plugins.ARPluginContext)
     */
    @Override
    public void terminate(ARPluginContext context) throws ARException {
        try {
            context.logMessage(pluginInfo, ARPluginContext.PLUGIN_LOG_LEVEL_INFO, "terminate()");
            /*
             * Close the connection
             */
            con.close();
        } catch (Exception e) {
            new ARException(Arrays.asList(new
        			StatusInfo(Constants.AR_RETURN_ERROR, 1001, "Error closing jdbc connection to the datasource", DBNAME)));
        }
    }

}
