/* File: api.c */

#include <stdlib.h>
#ifdef _WIN32
#include <time.h>
#else /* _WIN32 */
#include <unistd.h>
#endif /* _WIN32 */
#include <stdio.h>
#include <string.h>

#include "ar.h"
#include "arextern.h"
#include "arfree.h"
#include "api.h"
#include "get.h"
#include "globals.h"
#include "print.h"
#include "util.h"

#ifndef OUTPUT_MODE
void BeginAPICall()
{
   /* perform any generic pre-call operations here */
}

void EndAPICall(status)
ARStatusList *status;  /* IN; status list */
{
   /* perform any generic post-call operations here */
}
#else
void BeginAPICall();
void EndAPICall();
#endif


/*****************************************************************************/
/*                                                                           */
/*                           APIPrintStatusHistory                           */
/*                                                                           */
/*****************************************************************************/

void APIPrintStatusHistory(header, value)
char      *header;            /* IN; header string for the value */
char      *value;             /* IN; value to decode and print */

{
   int                   result;
   ARStatusHistoryList   statHistList;
   ARStatusList          status={0, NULL};

   /* Call routine to decode the value */
   result = ARDecodeStatusHistory(GetControlStructPtr(), value, &statHistList,
                                  &status);

   /* Print results */
   if (result >= AR_RETURN_ERROR)
   {
      DriverPrintResult("%s   DECODE STATUS-HISTORY FAILURE\n", header);
      PrintARStatusList(&status);
   }
   else
      PrintARStatusHistoryList(header, &statHistList);

   /* Cleanup */
   if (result < AR_RETURN_ERROR)
      FreeARStatusHistoryList(&statHistList, FALSE);
   FreeARStatusList(&status, FALSE);
}


/*****************************************************************************/
/*                                                                           */
/*                              APIPrintDiary                                */
/*                                                                           */
/*****************************************************************************/

void APIPrintDiary(header, value)
char      *header;            /* IN; header string for the value */
char      *value;             /* IN; value to decode and print */

{
   ARDiaryList    diaryList;
   int            result;
   ARStatusList   status={0, NULL};

   /* Call routine to decode the value */
   result = ARDecodeDiary(GetControlStructPtr(), value, &diaryList, &status);

   /* Print results */
   if (result >= AR_RETURN_ERROR)
   {
      DriverPrintResult("%s   DECODE DIARY FAILURE\n", header);
      PrintARStatusList(&status);
   }
   else
      PrintARDiaryList(header, &diaryList);

   /* Cleanup */
   if (result < AR_RETURN_ERROR)
      FreeARDiaryList(&diaryList, FALSE);
   FreeARStatusList(&status, FALSE);
}


/*****************************************************************************/
/*                                                                           */
/*                            APIPrintDiaryList                              */
/*                                                                           */
/*****************************************************************************/

void APIPrintDiaryList(header, header2, value)
char             *header;     /* IN; header string for the value */
char             *header2;    /* IN; header string for 2nd+ item */
ARTextStringList *value;      /* IN; value to print */

{
   unsigned int      i;       /* working index */

   DriverPrintResult("%s : %u items\n", header, value->numItems);
   for (i = 0; i < value->numItems; i++)
      APIPrintDiary(header2, value->stringList[i]);
}


/*****************************************************************************/
/**********                                                         **********/
/**********                          ENTRY                          **********/
/**********                                                         **********/
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*                               APIARGetEntry                               */
/*                                                                           */
/*****************************************************************************/

void APIARGetEntry()

{
   AREntryIdList      entryId;
   ARFieldValueList   fieldList;
   ARInternalIdList   idList;
   int                numItems;
   int                result;
   ARNameType         schema;
   ARStatusList       status={0, NULL};

   /* Get Parameters */
   DriverPrintHeader("GET ENTRY");
   strcpy(schema, GetChar("Schema: ", ""));
   numItems = GetInt("Number of Items in EntryId ? (1): ", 1);
   entryId.numItems = numItems;
   entryId.entryIdList = (AREntryIdType *) 
                         malloc(numItems * sizeof(AREntryIdType));
   if (entryId.entryIdList == NULL)
   {
      DriverPrintError("**** malloc failure for ARGetEntry call ****\n");
      return;
   }

   for (numItems=0; numItems < (int) entryId.numItems; numItems++)
      strcpy(entryId.entryIdList[numItems], GetChar("Entry Id: ", ""));
   DriverPrintPrompt("Ids of fields to retrieve:\n");
   GetARInternalIdList(&idList);

   /* Call routine */
   BeginAPICall();
   result = ARGetEntry(GetControlStructPtr(), schema, &entryId, &idList,
                       &fieldList, &status);
   EndAPICall(&status);

   /* Print results */
   DriverPrintResultHeader("ARGetEntry");
   PrintReturnCode(result);
   if (result < AR_RETURN_ERROR)
      PrintARFieldValueList(&fieldList);
   PrintARStatusList(&status);

   /* Cleanup */
   FreeARInternalIdList(&idList, FALSE);
   if (result < AR_RETURN_ERROR)
      FreeARFieldValueList(&fieldList, FALSE);
   FreeARStatusList(&status, FALSE);
   FreeAREntryIdList(&entryId, FALSE);
}


/*****************************************************************************/
/*                                                                           */
/*                          APIARGetMultipleEntries                          */
/*                                                                           */
/*****************************************************************************/

void APIARGetMultipleEntries()

{
   AREntryIdListList      entryId;
   ARBooleanList          existList;
   ARFieldValueListList   fieldList;
   int                    i;
   unsigned int           j;
   ARInternalIdList       idList;
   int                    numEntries;
   int                    numItems;
   int                    result;
   ARNameType             schema;
   ARStatusList           status={0, NULL};

   /* Get Parameters */
   DriverPrintHeader("GET MULTIPLE ENTRIES");
   strcpy(schema, GetChar("Schema: ", ""));
   while ((numEntries = GetInt("Number of Entries in EntryId List ? (1): ", 1)) <= 0)
      DriverPrintWarning("Number of entries must be greater than 0\n");
   entryId.numItems = numEntries;
   entryId.entryIdList = (AREntryIdList *) 
                         malloc(numEntries * sizeof(AREntryIdList));
   if (entryId.entryIdList == NULL)
   {
      DriverPrintError("**** malloc failure for ARGetMultipleEntries call ****\n");
      return;
   }
   for (i = 0; i < numEntries; i++)
   {
      DriverPrintPrompt("Enter entry id number %d\n", i + 1);
      numItems = GetInt("Number of Items in EntryId ? (1): ", 1);
      entryId.entryIdList[i].numItems = numItems;
      entryId.entryIdList[i].entryIdList = (AREntryIdType *) 
                         malloc(numItems * sizeof(AREntryIdType));
      if (entryId.entryIdList[i].entryIdList == NULL)
      {
         DriverPrintError("**** malloc failure for ARGetMultipleEntries call ****\n");
         free(entryId.entryIdList);
         return;
      }
      for (j = 0; j < (int) entryId.entryIdList[i].numItems; j++)
         strcpy(entryId.entryIdList[i].entryIdList[j], GetChar("Entry Id: ", ""));
   }

   DriverPrintPrompt("Ids of fields to retrieve:\n");
   GetARInternalIdList(&idList);

   /* Call routine */
   BeginAPICall();
   result = ARGetMultipleEntries(GetControlStructPtr(), schema, &entryId,
                                 &idList, &existList, &fieldList, &status);
   EndAPICall(&status);

   /* Print results */
   DriverPrintResultHeader("ARGetMultipleEntries");
   PrintReturnCode(result);
   if (result < AR_RETURN_ERROR)
   {
      PrintARBooleanList("Entry exists list", &existList);
      PrintARFieldValueListList(&fieldList);
   }
   PrintARStatusList(&status);

   /* Cleanup */
   FreeARInternalIdList(&idList, FALSE);
   FreeARBooleanList(&existList, FALSE);
   FreeARFieldValueListList(&fieldList, FALSE);
   FreeARStatusList(&status, FALSE);
   FreeAREntryIdListList(&entryId, FALSE);
}


/*****************************************************************************/
/*                                                                           */
/*                               APIARSetEntry                               */
/*                                                                           */
/*****************************************************************************/

void APIARSetEntry()

{
   AREntryIdList      entryId;
   ARFieldValueList   fieldList;
   ARTimestamp        getTime;
   int                numItems;
   int                option;
   int                result;
   ARNameType         schema;
   ARStatusList       status={0, NULL};

   /* Get Parameters */
   DriverPrintHeader("SET ENTRY");
   strcpy(schema, GetChar("Schema: ", ""));
   numItems = GetInt("Number of Items in EntryId ? (1): ", 1);
   entryId.numItems = numItems;
   entryId.entryIdList = (AREntryIdType *) 
                         malloc(numItems * sizeof(AREntryIdType));
   if (entryId.entryIdList == NULL)
   {
      DriverPrintError("**** malloc failure for ARSetEntry call ****\n");
      return;
   }

   for (numItems = 0; numItems < (int) entryId.numItems; numItems++)
      strcpy(entryId.entryIdList[numItems], GetChar("Entry Id: ", ""));
   DriverPrintPrompt("Field/value pairs to set:\n");
   GetARFieldValueList(&fieldList);
   getTime = GetARTimestamp("Time of Get operation (0): ", (ARTimestamp) 0);
   option = GetInt("SetEntry option ? (0): ", 0);

   /* Call routine */
   BeginAPICall();
   result = ARSetEntry(GetControlStructPtr(), schema, &entryId, &fieldList,
                       getTime, option, &status);
   EndAPICall(&status);

   /* Print results */
   DriverPrintResultHeader("ARSetEntry");
   PrintReturnCode(result);
   PrintARStatusList(&status);

   /* Cleanup */
   FreeARFieldValueList(&fieldList, FALSE);
   FreeARStatusList(&status, FALSE);
   FreeAREntryIdList(&entryId, FALSE);
}


/*****************************************************************************/
/*                                                                           */
/*                              APIARCreateEntry                             */
/*                                                                           */
/*****************************************************************************/

void APIARCreateEntry()

{
   AREntryIdType      entryId;
   ARFieldValueList   fieldList;
   int                result;
   ARNameType         schema;
   ARStatusList       status={0, NULL};

   /* Get Parameters */
   DriverPrintHeader("CREATE ENTRY");
   strcpy(schema, GetChar("Schema: ", ""));
   DriverPrintPrompt("Field/value pairs to create:\n");
   GetARFieldValueList(&fieldList);

   /* Call routine */
   BeginAPICall();
   result = ARCreateEntry(GetControlStructPtr(), schema, &fieldList, entryId,
                          &status);
   EndAPICall(&status);

   /* Print results */
   DriverPrintResultHeader("ARCreateEntry");
   PrintReturnCode(result);
   PrintChar("Entry id: ", entryId);
   PrintARStatusList(&status);

   /* Cleanup */
   FreeARFieldValueList(&fieldList, FALSE);
   FreeARStatusList(&status, FALSE);
}


/*****************************************************************************/
/*                                                                           */
/*                              APIARDeleteEntry                             */
/*                                                                           */
/*****************************************************************************/

void APIARDeleteEntry()

{
   AREntryIdList   entryId;
   int             numItems;
   int             option;
   int             result;
   ARNameType      schema;
   ARStatusList    status={0, NULL};
   

   /* Get Parameters */
   DriverPrintHeader("DELETE ENTRY");
   strcpy(schema, GetChar("Schema: ", ""));
   numItems = GetInt("Number of Items in EntryId ? (1): ", 1);
   entryId.numItems = numItems;
   entryId.entryIdList = (AREntryIdType *) 
                         malloc(numItems * sizeof(AREntryIdType));
   if (entryId.entryIdList == NULL)
   {
      DriverPrintError("**** malloc failure for ARDeleteEntry call ****\n");
      return;
   }

   for (numItems = 0; numItems < (int) entryId.numItems; numItems++)
      strcpy(entryId.entryIdList[numItems], GetChar("Entry Id: ", ""));
   option = GetInt("DeleteEntry option ? (0): ", 0);

   /* Call routine */
   BeginAPICall();
   result = ARDeleteEntry(GetControlStructPtr(), schema, &entryId, option,
                          &status);
   EndAPICall(&status);

   /* Print results */
   DriverPrintResultHeader("ARDeleteEntry");
   PrintReturnCode(result);
   PrintARStatusList(&status);

   /* Cleanup */
   FreeARStatusList(&status, FALSE);
   FreeAREntryIdList(&entryId, FALSE);
}


/*****************************************************************************/
/*                                                                           */
/*                              APIARMergeEntry                              */
/*                                                                           */
/*****************************************************************************/

void APIARMergeEntry()

{
   AREntryIdType      entryId;
   ARFieldValueList   fieldList;
   unsigned int       mergeType;
   int                result;
   ARNameType         schema;
   ARStatusList       status={0, NULL};

   /* Get Parameters */
   DriverPrintHeader("MERGE ENTRY");
   strcpy(schema, GetChar("Schema: ", ""));
   DriverPrintPrompt("Field/value pairs to merge:\n");
   GetARFieldValueList(&fieldList);
   mergeType = GetInt(
       "Operation on duplicate entry (error, new ID, overwrite, merge) (1-4): ",
       500);

   /* Call routine */
   BeginAPICall();
   result = ARMergeEntry(GetControlStructPtr(), schema, &fieldList, mergeType,
                         entryId, &status);
   EndAPICall(&status);

   /* Print results */
   DriverPrintResultHeader("ARMergeEntry");
   PrintReturnCode(result);
   PrintChar("Entry id: ", entryId);
   PrintARStatusList(&status);

   /* Cleanup */
   FreeARFieldValueList(&fieldList, FALSE);
   FreeARStatusList(&status, FALSE);
}


/*****************************************************************************/
/*                                                                           */
/*                              APIARGetListEntry                            */
/*                                                                           */
/*****************************************************************************/

void APIARGetListEntry()

{
   AREntryListList        entryList;
   AREntryListFieldList   getListFields;
   unsigned int           firstRetrieve;
   unsigned int           maxRetrieve;
   unsigned int           numMatches;
   unsigned int          *numMatchesPtr;
   ARQualifierStruct      qualifier;
   int                    result;
   ARNameType             schema;
   ARSortList             sortList;
   ARStatusList           status={0, NULL};

   /* Get Parameters */
   DriverPrintHeader("GETLIST ENTRY");
   strcpy(schema, GetChar("Schema: ", ""));
   GetARQualifierStruct(&qualifier);
   GetAREntryListFieldList(&getListFields);
   GetARSortList(&sortList);
   firstRetrieve = GetInt("First Retrieve (0): ", 0);
   maxRetrieve = GetInt("Maximum number of entries to retrieve (500): ", 500);
   if (GetARBoolean("Get number of matches? (F): ", FALSE))
      numMatchesPtr = &numMatches;
   else
      numMatchesPtr = NULL;

   /* Call routine */
   BeginAPICall();
   result = ARGetListEntry(GetControlStructPtr(), schema, &qualifier,
                           &getListFields, &sortList, firstRetrieve, maxRetrieve, &entryList,
                           numMatchesPtr, &status);
   EndAPICall(&status);

   /* Print results */
   DriverPrintResultHeader("ARGetListEntry");
   PrintReturnCode(result);
   if (result < AR_RETURN_ERROR)
   {
      PrintAREntryListList(&entryList);
      if (numMatchesPtr != NULL)
         PrintInt("Number of matches: ", numMatches);
   }
   PrintARStatusList(&status);

   /* Cleanup */
   FreeARQualifierStruct(&qualifier, FALSE);
   FreeAREntryListFieldList(&getListFields, FALSE);
   FreeARSortList(&sortList, FALSE);
   if (result < AR_RETURN_ERROR)
      FreeAREntryListList(&entryList, FALSE);
   FreeARStatusList(&status, FALSE);
}


/*****************************************************************************/
/*                                                                           */
/*                        APIARGetListEntryWithFields                        */
/*                                                                           */
/*****************************************************************************/

void APIARGetListEntryWithFields()

{
   AREntryListFieldValueList   entryList;
   AREntryListFieldList        getListFields;
   unsigned int                firstRetrieve;
   unsigned int                maxRetrieve;
   unsigned int                numMatches;
   unsigned int               *numMatchesPtr;
   ARQualifierStruct           qualifier;
   int                         result;
   ARSortList                  sortList;
   ARNameType                  schema;
   ARStatusList                status={0, NULL};

   /* Get Parameters */
   DriverPrintHeader("GETLIST ENTRY WITH FIELDS");
   strcpy(schema, GetChar("Schema: ", ""));
   GetARQualifierStruct(&qualifier);
   GetAREntryListFieldList(&getListFields);
   GetARSortList(&sortList);
   firstRetrieve = GetInt("First Retrieve (0): ", 0);
   maxRetrieve = GetInt("Maximum number of entries to retrieve (500): ", 500);
   if (GetARBoolean("Get number of matches? (F): ", FALSE))
      numMatchesPtr = &numMatches;
   else
      numMatchesPtr = NULL;

   /* Call routine */
   BeginAPICall();
   result = ARGetListEntryWithFields(GetControlStructPtr(), schema, &qualifier,
                                     &getListFields, &sortList, firstRetrieve, maxRetrieve,
                                     &entryList, numMatchesPtr, &status);
   EndAPICall(&status);

   /* Print results */
   DriverPrintResultHeader("ARGetListEntryWithFields");
   PrintReturnCode(result);
   if (result < AR_RETURN_ERROR)
   {
      PrintAREntryListFieldValueList(&entryList);
      if (numMatchesPtr != NULL)
         PrintInt("Number of matches: ", numMatches);
      SaveEntryListFieldValueEntryIds(&entryList);
   }
   PrintARStatusList(&status);

   /* Cleanup */
   FreeARQualifierStruct(&qualifier, FALSE);
   FreeAREntryListFieldList(&getListFields, FALSE);
   FreeARSortList(&sortList, FALSE);
   if (result < AR_RETURN_ERROR)
      FreeAREntryListFieldValueList(&entryList, FALSE);
   FreeARStatusList(&status, FALSE);
}


/*****************************************************************************/
/*                                                                           */
/*                            APIARGetEntryStatistics                        */
/*                                                                           */
/*****************************************************************************/

void APIARGetEntryStatistics()

{
   ARInternalIdList            groupByList;
   ARQualifierStruct           qualifier;
   int                         result;
   ARStatisticsResultList      results;
   ARNameType                  schema;
   unsigned int                statistic;
   ARStatusList                status={0, NULL};
   ARFieldValueOrArithStruct   target;

   /* Get Parameters */
   DriverPrintHeader("GET ENTRY STATISTICS");
   strcpy(schema, GetChar("Schema: ", ""));
   GetARQualifierStruct(&qualifier);
   DriverPrintPrompt("Target Expression:\n");
   GetARFieldValueOrArithStruct(&target);
   statistic = GetInt(
        "Statistic operation to perform (Count,Sum,Avg,Min,Max) (1-5)(1): ", 1);
   DriverPrintPrompt("Ids of fields to group statistics by:\n");
   GetARInternalIdList(&groupByList);

   /* Call routine */
   BeginAPICall();
   result = ARGetEntryStatistics(GetControlStructPtr(), schema, &qualifier,
                                 &target, statistic, &groupByList, &results,
                                 &status);
   EndAPICall(&status);

   /* Print results */
   DriverPrintResultHeader("ARGetEntryStatistics");
   PrintReturnCode(result);
   if (result < AR_RETURN_ERROR)
   {
      PrintARStatisticsResultList("", &results);
   }
   PrintARStatusList(&status);

   /* Cleanup */
   FreeARQualifierStruct(&qualifier, FALSE);
   FreeARFieldValueOrArithStruct(&target, FALSE);
   if (result < AR_RETURN_ERROR)
      FreeARStatisticsResultList(&results, FALSE);
   FreeARStatusList(&status, FALSE);
}


/*****************************************************************************/
/*                                                                           */
/*                            APIARGetEntryBLOB                              */
/*                                                                           */
/*****************************************************************************/

void APIARGetEntryBLOB()

{
   AREntryIdList   entryId;
   ARInternalId    fieldId;
   ARLocStruct     loc;
   int             numItems;
   int             result;
   ARNameType      schema;
   ARStatusList    status={0, NULL};
   
   /* Get Parameters */
   DriverPrintHeader("GET ENTRY BLOB");
   strcpy(schema, GetChar("Schema: ", ""));
   numItems = GetInt("Number of Items in EntryId ? (1): ", 1);
   entryId.numItems = numItems;
   entryId.entryIdList = (AREntryIdType *) 
                         malloc(numItems * sizeof(AREntryIdType));
   if (entryId.entryIdList == NULL)
   {
      DriverPrintError("**** malloc failure for ARGetEntryBLOB call ****\n");
      return;
   }

   for (numItems = 0; numItems < (int) entryId.numItems; numItems++)
      strcpy(entryId.entryIdList[numItems], GetChar("Entry Id: ", ""));
   fieldId = GetLong("Field Id: ", 1L);
   GetARLocStructForGet(&loc);

   /* Call routine */
   BeginAPICall();
   result = ARGetEntryBLOB(GetControlStructPtr(), schema, &entryId, 
                           fieldId, &loc, &status);
   EndAPICall(&status);

   /* Print results */
   DriverPrintResultHeader("ARGetEntryBLOB");
   PrintReturnCode(result);
   PrintARStatusList(&status);

   /* Cleanup */
   FreeARStatusList(&status, FALSE);
   FreeARLocStruct(&loc, FALSE);
   FreeAREntryIdList(&entryId, FALSE);
}


/*****************************************************************************/
/*                                                                           */
/*                           APIARLoadARQualifierStruct                      */
/*                                                                           */
/*****************************************************************************/

void APIARLoadARQualifierStruct()

{
   ARNameType          displayTag;
   ARQualifierStruct   qualifier;
   char                qualString[1024];
   int                 result;
   ARNameType          schema;
   ARStatusList        status={0, NULL};

   /* Get Parameters */
   DriverPrintHeader("LOAD ARQualifierStruct");
   strcpy(schema, GetChar("Schema: ", ""));
   strcpy(displayTag, GetChar("Display Tag (): ", ""));
   strcpy(qualString, GetChar("Qualification string: ", ""));

   /* Call routine */
   BeginAPICall();
   result = ARLoadARQualifierStruct(GetControlStructPtr(), schema, displayTag,
                                    qualString, &qualifier, &status);
   EndAPICall(&status);

   /* Print results */
   DriverPrintResultHeader("ARLoadARQualifierStruct");
   PrintReturnCode(result);
   if (result < AR_RETURN_ERROR)
      PrintARQualifierStruct("", &qualifier);
   PrintARStatusList(&status);

   /* Cleanup */
   FreeARQualifierStruct(&qualifier, FALSE);
   FreeARStatusList(&status, FALSE);
}


/*****************************************************************************/
/**********                                                         **********/
/**********                         STRUCTURE                       **********/
/**********                                                         **********/
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*                              APIARGetContainer                            */
/*                                                                           */
/*****************************************************************************/

void APIARGetContainer()
{
   ARInternalIdList      admingrpList;
   char                 *changeDiary;
   ARNameType            containerName;
   char                 *description;
   ARPermissionList      groupList;
   char                 *helpText;
   char                 *label;
   ARAccessNameType      lastChanged;
   ARPropList            objPropList;
   ARAccessNameType      owner;
   ARContainerOwnerObjList ownerObjList;
   ARReferenceList       references;
   ARReferenceTypeList   refTypes;
   int                   result;
   ARStatusList          status={0, NULL};
   ARTimestamp           timestamp;
   unsigned int          type;

   /* Get Parameters */
   DriverPrintHeader("GET CONTAINER");
   strcpy(containerName, GetChar("Container name: ", ""));
   GetARReferenceTypeList(&refTypes);

   /* Call routine */
   BeginAPICall();
   result = ARGetContainer(GetControlStructPtr(), containerName, &refTypes,
                           &groupList, &admingrpList, &ownerObjList, &label,
                           &description, &type, &references, &helpText, owner,
                           &timestamp, lastChanged, &changeDiary, &objPropList,
                           &status);
   EndAPICall(&status);

   /* Print results */
   DriverPrintResultHeader("ARGetContainer");
   PrintReturnCode(result);
   if (result < AR_RETURN_ERROR)
   {
      PrintARPermissionList(&groupList, FALSE);
      PrintARInternalIdList("Sub Admin Group list: ", "   ",
                            &admingrpList);
      PrintARContainerOwnerObjList(&ownerObjList);
      PrintChar("Label: ", label);
      PrintChar("Description: ", description);
      PrintInt("Type:", type);
      PrintARReferenceList(&references);
      PrintChar("Help text: ", helpText);
      PrintARTimestamp("Last update time: ", timestamp);
      PrintChar("Owner: ", owner);
      PrintChar("Last changed: ", lastChanged);
      APIPrintDiary("Change Diary: ", changeDiary);
      PrintARObjectPropList("", "", &objPropList);
   }
   PrintARStatusList(&status);

   /* Cleanup */
   if (result < AR_RETURN_ERROR)
   {
      FreeARPermissionList(&groupList, FALSE);
      FreeARInternalIdList(&admingrpList, FALSE);
      FreeARReferenceList(&references, FALSE);
      FreeARPropList(&objPropList, FALSE);
      FreeARContainerOwnerObjList(&ownerObjList, FALSE);
      if (helpText != NULL)
         free(helpText);
      if (changeDiary != NULL)
         free(changeDiary);
   }
   FreeARStatusList(&status, FALSE);
}


/*****************************************************************************/
/*                                                                           */
/*                              APIARSetContainer                            */
/*                                                                           */
/*****************************************************************************/

void APIARSetContainer()
{
   ARInternalIdList      admingrpList;
   ARInternalIdList     *admingrpListPtr;
   char                  changeDiary[1024];
   char                 *changeDiaryPtr;
   ARNameType            containerName;
   char                  description[ARMAX_CON_DESCRIPTION_LEN + 1];
   char                 *descriptionPtr;
   ARPermissionList      groupList;
   ARPermissionList     *groupListPtr;
   char                  helpText[1024];
   char                 *helpTextPtr;
   char                  label[256];
   char                 *labelPtr;
   ARNameType            newName;
   char                 *newNamePtr;
   ARPropList            objPropList;
   ARPropList           *objPropListPtr;
   ARAccessNameType      owner;
   ARContainerOwnerObjList  ownerObjList;
   ARContainerOwnerObjList *ownerObjListPtr;
   char                 *ownerPtr;
   ARReferenceList       references;
   ARReferenceList      *referencesPtr;
   ARBoolean             removeFlag;
   int                   result;
   ARStatusList          status={0, NULL};
   unsigned int          type;
   unsigned int         *typePtr;

   /* Get Parameters */
   DriverPrintHeader("SET CONTAINER");
   strcpy(containerName, GetChar("Container name: ", ""));
   if (GetARBoolean("Change container name? (F): ", FALSE))
   {
      strcpy(newName, GetChar("New name (): ", ""));
      newNamePtr = newName;
   }
   else
      newNamePtr = NULL;
   if (GetARBoolean("Change group list? (F): ", FALSE))
   {
      GetARPermissionList(FALSE, &groupList);
      groupListPtr = &groupList;
   }
   else
      groupListPtr = NULL;
   if (GetARBoolean("Change sub admin group list? (F): ", FALSE))
   {
      DriverPrintPrompt("Ids of sub admin groups allowed to access container:\n");
      GetARInternalIdList(&admingrpList);
      admingrpListPtr = &admingrpList;
   }
   else
      admingrpListPtr = NULL;
   if (GetARBoolean("Change container owner object list? (F): ", FALSE))
   {
      GetARContainerOwnerObjList(&ownerObjList);
      ownerObjListPtr = &ownerObjList;
   }
   else
      ownerObjListPtr = NULL;
   if (GetARBoolean("Change label? (F): ", FALSE))
   {
      strcpy(label, GetChar("   Label (): ", ""));
      labelPtr = label;
   }
   else
      labelPtr = NULL;
   if (GetARBoolean("Change description? (F): ", FALSE))
   {
      strcpy(description, GetChar("   Description (): ", ""));
      descriptionPtr = description;
   }
   else
      descriptionPtr = NULL;
   if (GetARBoolean("Change container type? (F): ", FALSE))
   {
      type = GetInt("Active Link Guide, Application, Filter Guide (1-2,4) (2): ", 2);
      typePtr = &type;
   }
   else
      typePtr = NULL;
   if (GetARBoolean("Change reference list? (F): ", FALSE))
   {
      GetARReferenceList(&references);
      referencesPtr = &references;
      removeFlag = GetARBoolean("Remove invalid references? (F): ", FALSE);
   }
   else
      referencesPtr = NULL;
   if (GetARBoolean("Change help text? (F): ", FALSE))
   {
      strcpy(helpText, GetChar("   Help Text (): ", ""));
      helpTextPtr = helpText;
   }
   else
      helpTextPtr = NULL;
   if (GetARBoolean("Change owner? (F): ", FALSE))
   {
      strcpy(owner, GetChar("   Owner (): ", ""));
      ownerPtr = owner;
   }
   else
      ownerPtr = NULL;
   if (GetARBoolean("Add to change diary? (F): ", FALSE))
   {
      strcpy(changeDiary, GetChar("   Change Diary (): ", ""));
      changeDiaryPtr = changeDiary;
   }
   else
      changeDiaryPtr = NULL;
   if (GetARBoolean("Change object properties? (F): ", FALSE))
   {
      GetARPropList(&objPropList);
      objPropListPtr = &objPropList;
   }
   else
      objPropListPtr = NULL;

   /* Call routine */
   BeginAPICall();
   result = ARSetContainer(GetControlStructPtr(), containerName, newNamePtr,
                           groupListPtr, admingrpListPtr, ownerObjListPtr, labelPtr,
                           descriptionPtr, typePtr, referencesPtr, removeFlag,
                           helpTextPtr, ownerPtr, changeDiaryPtr,
                           objPropListPtr, &status);
   EndAPICall(&status);

   /* Print results */
   DriverPrintResultHeader("ARSetContainer");
   PrintReturnCode(result);
   PrintARStatusList(&status);

   /* Cleanup */
   if (groupListPtr != NULL)
      FreeARPermissionList(&groupList, FALSE);
   if (admingrpListPtr != NULL)
      FreeARInternalIdList(&admingrpList, FALSE);
   if (ownerObjListPtr != NULL)
      FreeARContainerOwnerObjList(&ownerObjList, FALSE);
   if (referencesPtr != NULL)
      FreeARReferenceList(&references, FALSE);
   if (objPropListPtr != NULL)
      FreeARPropList(&objPropList, FALSE);
   FreeARStatusList(&status, FALSE);
}


/*****************************************************************************/
/*                                                                           */
/*                            APIARCreateContainer                           */
/*                                                                           */
/*****************************************************************************/

void APIARCreateContainer()
{
   ARInternalIdList        admingrpList;
   char                    changeDiary[1024];
   ARNameType              containerName;
   char                    description[ARMAX_CON_DESCRIPTION_LEN + 1];
   ARPermissionList        groupList;
   char                    helpText[1024];
   char                    label[256];
   ARPropList              objPropList;
   ARAccessNameType        owner;
   ARContainerOwnerObjList ownerObjList;
   ARReferenceList         references;
   ARBoolean               removeFlag;
   int                     result;
   ARStatusList            status={0, NULL};
   unsigned int            type;
   
   /* Get Parameters */
   DriverPrintHeader("CREATE CONTAINER");
   strcpy(containerName, GetChar("Container name: ", ""));
   GetARPermissionList(FALSE, &groupList);
   DriverPrintPrompt("Ids of sub admin groups allowed to access container:\n");
   GetARInternalIdList(&admingrpList);
   GetARContainerOwnerObjList(&ownerObjList);
   strcpy(label, GetChar("   Label (): ", ""));
   strcpy(description, GetChar("   Description (): ", ""));
   type = GetInt("Active Link Guide, Application, Filter Guide (1-2,4) (2): ", 2);
   GetARReferenceList(&references);
   removeFlag = GetARBoolean("Remove invalid references? (F): ", FALSE);
   strcpy(helpText, GetChar("Help Text (): ", ""));
   strcpy(owner, GetChar("Owner (): ", ""));
   strcpy(changeDiary, GetChar("Change Diary (): ", ""));
   GetARPropList(&objPropList);

   /* Call routine */
   BeginAPICall();
   result = ARCreateContainer(GetControlStructPtr(), containerName, &groupList,
                              &admingrpList, &ownerObjList, label, description,
                              type, &references, removeFlag, helpText, owner,
                              changeDiary, &objPropList, &status);
   EndAPICall(&status);

   /* Print results */
   DriverPrintResultHeader("ARCreateContainer");
   PrintReturnCode(result);
   PrintARStatusList(&status);

   /* Cleanup */
   FreeARPermissionList(&groupList, FALSE);
   FreeARInternalIdList(&admingrpList, FALSE);
   FreeARContainerOwnerObjList(&ownerObjList, FALSE);
   FreeARReferenceList(&references, FALSE);
   FreeARPropList(&objPropList, FALSE);
   FreeARStatusList(&status, FALSE);
}


/*****************************************************************************/
/*                                                                           */
/*                            APIARDeleteContainer                           */
/*                                                                           */
/*****************************************************************************/

void APIARDeleteContainer()
{
   ARNameType     containerName;
   int            result;
   ARStatusList   status={0, NULL};
   
   /* Get Parameters */
   DriverPrintHeader("DELETE CONTAINER");
   strcpy(containerName, GetChar("Container name: ", ""));

   /* Call routine */
   BeginAPICall();
   result = ARDeleteContainer(GetControlStructPtr(), containerName, &status);
   EndAPICall(&status);

   /* Print results */
   DriverPrintResultHeader("ARDeleteContainer");
   PrintReturnCode(result);
   PrintARStatusList(&status);

   /* Cleanup */
   FreeARStatusList(&status, FALSE);
}


/*****************************************************************************/
/*                                                                           */
/*                            APIARGetListContainer                          */
/*                                                                           */
/*****************************************************************************/

void APIARGetListContainer()
{
   unsigned int            attributes;
   ARTimestamp             changedSince;
   ARContainerInfoList     containerList;
   ARContainerTypeList     containerTypes;
   ARContainerOwnerObjList ownerObjList;
   int                     result;
   ARStatusList            status={0, NULL};

   /* Get Parameters */
   DriverPrintHeader("GETLIST CONTAINER");
   changedSince = GetARTimestamp("Get all changed since (0): ",
                                 (ARTimestamp) 0);
   GetARContainerTypeList(&containerTypes);
   attributes = GetInt("Container attributes (none, hidden) (0-1) (0): ", 0);
   GetARContainerOwnerObjList(&ownerObjList);

   /* Call routine */
   BeginAPICall();
   result = ARGetListContainer(GetControlStructPtr(), changedSince,
                               &containerTypes, attributes, &ownerObjList,
                               &containerList, &status);
   EndAPICall(&status);

   /* Print results */
   DriverPrintResultHeader("ARGetListContainer");
   PrintReturnCode(result);
   if (result < AR_RETURN_ERROR)
      PrintARContainerInfoList("Container list: ", "   ", &containerList);
   PrintARStatusList(&status);

   /* Cleanup */
   if (result < AR_RETURN_ERROR)
      FreeARContainerInfoList(&containerList, FALSE);
   FreeARContainerOwnerObjList(&ownerObjList, FALSE);
   FreeARContainerTypeList(&containerTypes, FALSE);
   FreeARStatusList(&status, FALSE);
}


/*****************************************************************************/
/*                                                                           */
/*                                APIARGetSchema                             */
/*                                                                           */
/*****************************************************************************/

void APIARGetSchema()

{
   ARInternalIdList       admingrpList;
   char                  *changeDiary;
   ARCompoundSchema       comSchema;
   AREntryListFieldList   getListFields;
   ARPermissionList       groupList;
   char                  *helpText;
   ARIndexList            indexList;
   ARNameType             defaultVui;
   ARAccessNameType       lastChanged;
   ARPropList             objPropList;
   ARAccessNameType       owner;
   int                    result;
   ARNameType             schemaName;
   ARSortList             sortList;
   ARStatusList           status={0, NULL};
   ARTimestamp            timestamp;
   
   /* Get Parameters */
   DriverPrintHeader("GET SCHEMA");
   strcpy(schemaName, GetChar("Schema name: ", ""));

   /* Call routine */
   BeginAPICall();
   result = ARGetSchema(GetControlStructPtr(), schemaName, &comSchema,
                        &groupList, &admingrpList, &getListFields, &sortList,
                        &indexList, defaultVui, &helpText, &timestamp, owner,
                        lastChanged, &changeDiary, &objPropList, &status);
   EndAPICall(&status);

   /* Print results */
   DriverPrintResultHeader("ARGetSchema");
   PrintReturnCode(result);
   if (result < AR_RETURN_ERROR)
   {
      PrintARCompoundSchema(&comSchema);
      PrintARPermissionList(&groupList, FALSE);
      PrintARInternalIdList("Sub Admin Group list: ", "   ", &admingrpList);
      PrintAREntryListFieldList(&getListFields);
      PrintARSortList(&sortList);
      PrintARIndexList(&indexList);
      PrintChar("Default VUI: ", defaultVui);
      PrintChar("Help text: ", helpText);
      PrintARTimestamp("Last update time: ", timestamp);
      PrintChar("Owner: ", owner);
      PrintChar("Last changed: ", lastChanged);
      APIPrintDiary("Change Diary: ", changeDiary);
      PrintARObjectPropList("", "", &objPropList);
   }
   PrintARStatusList(&status);

   /* Cleanup */
   if (result < AR_RETURN_ERROR)
   {
      FreeARCompoundSchema(&comSchema, FALSE);
      FreeARPermissionList(&groupList, FALSE);
      FreeARInternalIdList(&admingrpList, FALSE);
      FreeAREntryListFieldList(&getListFields, FALSE);
      FreeARSortList(&sortList, FALSE);
      FreeARIndexList(&indexList, FALSE);
      FreeARPropList(&objPropList, FALSE);
      if (helpText != NULL)
         free(helpText);
      if (changeDiary != NULL)
         free(changeDiary);
   }
   FreeARStatusList(&status, FALSE);
}


/*****************************************************************************/
/*                                                                           */
/*                                APIARSetSchema                             */
/*                                                                           */
/*****************************************************************************/

void APIARSetSchema()

{
   ARInternalIdList       admingrpList;
   ARInternalIdList      *admingrpListPtr;
   char                   changeDiary[1024];
   char                  *changeDiaryPtr;
   ARCompoundSchema       comSchema;
   ARCompoundSchema      *comSchemaPtr;
   AREntryListFieldList   getListFields;
   AREntryListFieldList  *getListFieldsPtr;
   ARPermissionList       groupList;
   ARPermissionList      *groupListPtr;
   char                   helpText[1024];
   char                  *helpTextPtr;
   ARIndexList            indexList;
   ARIndexList           *indexListPtr;
   ARNameType             defaultVui;
   char                  *defaultVuiPtr;
   ARNameType             newName;
   char                  *newNamePtr;
   ARPropList             objPropList;
   ARPropList            *objPropListPtr;
   ARAccessNameType       owner;
   char                  *ownerPtr;
   int                    result;
   ARNameType             schemaName;
   ARSortList             sortList;
   ARSortList            *sortListPtr;
   ARStatusList           status={0, NULL};
   
   /* Get Parameters */
   DriverPrintHeader("SET SCHEMA");
   strcpy(schemaName, GetChar("Schema name: ", ""));
   if (GetARBoolean("Change schema name? (F): ", FALSE))
   {
      strcpy(newName, GetChar("New name (): ", ""));
      newNamePtr = newName;
   }
   else
      newNamePtr = NULL;
   if (GetARBoolean("Change compound schema info? (F): ", FALSE))
   {
      GetARCompoundSchema(&comSchema);
      comSchemaPtr = &comSchema;
   }
   else
      comSchemaPtr = NULL;
   if (GetARBoolean("Change group list? (F): ", FALSE))
   {
      GetARPermissionList(FALSE, &groupList);
      groupListPtr = &groupList;
   }
   else
      groupListPtr = NULL;
   if (GetARBoolean("Change sub admin group list? (F): ", FALSE))
   {
      DriverPrintPrompt("Ids of sub admin groups allowed to access schema:\n");
      GetARInternalIdList(&admingrpList);
      admingrpListPtr = &admingrpList;
   }
   else
      admingrpListPtr = NULL;
   if (GetARBoolean("Change query list fields? (F): ", FALSE))
   {
      GetAREntryListFieldList(&getListFields);
      getListFieldsPtr = &getListFields;
   }
   else
      getListFieldsPtr = NULL;
   if (GetARBoolean("Change sort list? (F): ", FALSE))
   {
      GetARSortList(&sortList);
      sortListPtr = &sortList;
   }
   else
      sortListPtr = NULL;
   if (GetARBoolean("Change index list? (F): ", FALSE))
   {
      GetARIndexList(&indexList);
      indexListPtr = &indexList;
   }
   else
      indexListPtr = NULL;
   if (GetARBoolean("Change default VUI? (F): ", FALSE))
   {
      strcpy(defaultVui, GetChar("   Default VUI (): ", ""));
      defaultVuiPtr = defaultVui;
   }
   else
      defaultVuiPtr = NULL;
   if (GetARBoolean("Change help text? (F): ", FALSE))
   {
      strcpy(helpText, GetChar("   Help Text (): ", ""));
      helpTextPtr = helpText;
   }
   else
      helpTextPtr = NULL;
   if (GetARBoolean("Change owner? (F): ", FALSE))
   {
      strcpy(owner, GetChar("   Owner (): ", ""));
      ownerPtr = owner;
   }
   else
      ownerPtr = NULL;
   if (GetARBoolean("Add to change diary? (F): ", FALSE))
   {
      strcpy(changeDiary, GetChar("   Change Diary (): ", ""));
      changeDiaryPtr = changeDiary;
   }
   else
      changeDiaryPtr = NULL;
   if (GetARBoolean("Change object properties? (F): ", FALSE))
   {
      GetARPropList(&objPropList);
      objPropListPtr = &objPropList;
   }
   else
      objPropListPtr = NULL;

   /* Call routine */
   BeginAPICall();
   result = ARSetSchema(GetControlStructPtr(), schemaName, newNamePtr,
                        comSchemaPtr, groupListPtr, admingrpListPtr,
                        getListFieldsPtr, sortListPtr, indexListPtr,
                        defaultVuiPtr, helpTextPtr, ownerPtr,
                        changeDiaryPtr, objPropListPtr, &status);
   EndAPICall(&status);

   /* Print results */
   DriverPrintResultHeader("ARSetSchema");
   PrintReturnCode(result);
   PrintARStatusList(&status);

   /* Cleanup */
   if (comSchemaPtr != NULL)
      FreeARCompoundSchema(&comSchema, FALSE);
   if (groupListPtr != NULL)
      FreeARPermissionList(&groupList, FALSE);
   if (admingrpListPtr != NULL)
      FreeARInternalIdList(&admingrpList, FALSE);
   if (getListFieldsPtr != NULL)
      FreeAREntryListFieldList(&getListFields, FALSE);
   if (sortListPtr != NULL)
      FreeARSortList(&sortList, FALSE);
   if (indexListPtr != NULL)
      FreeARIndexList(&indexList, FALSE);
   if (objPropListPtr != NULL)
      FreeARPropList(&objPropList, FALSE);
   FreeARStatusList(&status, FALSE);
}


/*****************************************************************************/
/*                                                                           */
/*                              APIARCreateSchema                            */
/*                                                                           */
/*****************************************************************************/

void APIARCreateSchema()

{
   ARInternalIdList       admingrpList;
   char                   changeDiary[1024];
   ARCompoundSchema       comSchema;
   AREntryListFieldList   getListFields;
   ARPermissionList       groupList;
   char                   helpText[1024];
   ARIndexList            indexList;
   ARNameType             defaultVui;
   ARPropList             objPropList;
   ARAccessNameType       owner;
   int                    result;
   ARNameType             schemaName;
   ARSortList             sortList;
   ARStatusList           status={0, NULL};
   
   /* Get Parameters */
   DriverPrintHeader("CREATE SCHEMA");
   strcpy(schemaName, GetChar("Schema name: ", ""));
   GetARCompoundSchema(&comSchema);
   GetARPermissionList(FALSE, &groupList);
   DriverPrintPrompt("Ids of sub admin groups allowed to access schema:\n");
   GetARInternalIdList(&admingrpList);
   GetAREntryListFieldList(&getListFields);
   GetARSortList(&sortList);
   GetARIndexList(&indexList);
   strcpy(defaultVui, GetChar("Default Vui (): ", ""));
   strcpy(helpText, GetChar("Help Text (): ", ""));
   strcpy(owner, GetChar("Owner (): ", ""));
   strcpy(changeDiary, GetChar("Change Diary (): ", ""));
   GetARPropList(&objPropList);

   /* Call routine */
   BeginAPICall();
   result = ARCreateSchema(GetControlStructPtr(), schemaName, &comSchema,
                           &groupList, &admingrpList, &getListFields,
                           &sortList, &indexList, defaultVui, helpText,
                           owner, changeDiary, &objPropList, &status);
   EndAPICall(&status);

   /* Print results */
   DriverPrintResultHeader("ARCreateSchema");
   PrintReturnCode(result);
   PrintARStatusList(&status);

   /* Cleanup */
   FreeARCompoundSchema(&comSchema, FALSE);
   FreeARPermissionList(&groupList, FALSE);
   FreeARInternalIdList(&admingrpList, FALSE);
   FreeAREntryListFieldList(&getListFields, FALSE);
   FreeARSortList(&sortList, FALSE);
   FreeARIndexList(&indexList, FALSE);
   FreeARPropList(&objPropList, FALSE);
   FreeARStatusList(&status, FALSE);
}


/*****************************************************************************/
/*                                                                           */
/*                              APIARDeleteSchema                            */
/*                                                                           */
/*****************************************************************************/

void APIARDeleteSchema()

{
   unsigned int   deleteOption;
   int            result;
   ARNameType     schemaName;
   ARStatusList   status={0, NULL};

   /* Get Parameters */
   DriverPrintHeader("DELETE SCHEMA");
   strcpy(schemaName, GetChar("Schema name: ", ""));
   deleteOption = GetInt("Delete option ? (0): ", 0);

   /* Call routine */
   BeginAPICall();
   result = ARDeleteSchema(GetControlStructPtr(), schemaName, deleteOption,
                           &status);
   EndAPICall(&status);

   /* Print results */
   DriverPrintResultHeader("ARDeleteSchema");
   PrintReturnCode(result);
   PrintARStatusList(&status);

   /* Cleanup */
   FreeARStatusList(&status, FALSE);
}


/*****************************************************************************/
/*                                                                           */
/*                              APIARGetListSchema                           */
/*                                                                           */
/*****************************************************************************/

void APIARGetListSchema()

{
   ARTimestamp    changedSince;
   ARNameType     joinName;
   ARInternalIdList  fieldIdList;
   int            result;
   ARNameList     schemaList;
   int            schemaType;
   ARStatusList   status={0, NULL};

   /* Get Parameters */
   DriverPrintHeader("GETLIST SCHEMA");
   changedSince = GetARTimestamp("Get all changed since (0): ",
                                 (ARTimestamp) 0);
   schemaType = GetInt(
       "Type of schema (all,regular,join,view,uplink,downlink,dialog,\n                vendor) (0-6,8) (0): ", 0);
   if (GetARBoolean("Include hidden schemas (T): ", TRUE))
      schemaType += AR_HIDDEN_INCREMENT;
   if (((schemaType % AR_HIDDEN_INCREMENT) == AR_LIST_SCHEMA_UPLINK) ||
       ((schemaType % AR_HIDDEN_INCREMENT) == AR_LIST_SCHEMA_DOWNLINK))
      strcpy(joinName, GetChar("Schema name: ", ""));
   else
      joinName[0] = '\0';

   DriverPrintPrompt("Ids of fields, which must be on the schema:\n");
   GetARInternalIdList(&fieldIdList);

   /* Call routine */
   BeginAPICall();
   result = ARGetListSchema(GetControlStructPtr(), changedSince, schemaType,
                            joinName, &fieldIdList, &schemaList, &status);
   EndAPICall(&status);

   /* Print results */
   DriverPrintResultHeader("ARGetListSchema");
   PrintReturnCode(result);
   if (result < AR_RETURN_ERROR)
      PrintARNameList("Schema list: ", "   ", &schemaList);
   PrintARStatusList(&status);

   /* Cleanup */
   if (result < AR_RETURN_ERROR)
      FreeARNameList(&schemaList, FALSE);
   FreeARStatusList(&status, FALSE);
   FreeARInternalIdList(&fieldIdList, FALSE);
}


/*****************************************************************************/
/*                                                                           */
/*                           APIARGetMultipleSchemas                         */
/*                                                                           */
/*****************************************************************************/

void APIARGetMultipleSchemas()

{
   ARTimestamp          changedSince;
   ARUnsignedIntList    schemaTypeList;
   ARNameList           nameList;
   ARInternalIdList     fieldIdList;
   int                  result;
   unsigned int         i;

   ARBooleanList        existList;
   ARNameList           schemaNameList;
   ARCompoundSchemaList schemaList;
   ARPermissionListList groupListList;
   ARInternalIdListList admingrpListList;
   AREntryListFieldListList   getListFieldsList;
   ARSortListList       sortListList;
   ARIndexListList      indexListList;
   ARNameList           defaultVuiList;
   ARTextStringList     helpTextList;
   ARTimestampList      timestampList;
   ARAccessNameList     ownerList;
   ARAccessNameList     lastChangedList;
   ARTextStringList     changeDiaryList;
   ARPropListList       objPropListList;
   ARStatusList         status={0, NULL};

   /* Get Parameters */
   DriverPrintHeader("GETMULTIPLE SCHEMAS");
   changedSince = GetARTimestamp("Get all changed since (0): ",
                                 (ARTimestamp) 0);
   DriverPrintPrompt("Get schema type list:\n");
   GetARSchemaTypeList(&schemaTypeList);
   DriverPrintPrompt("Get schemas by name:\n");
   GetARNameList(&nameList, "   ");
   DriverPrintPrompt("Ids of fields, which must be on the schema:\n");
   GetARInternalIdList(&fieldIdList);

   /* Call routine */
   BeginAPICall();
   result = ARGetMultipleSchemas(GetControlStructPtr(), changedSince, &schemaTypeList,
                                 &nameList, &fieldIdList,
                                 &existList, &schemaNameList, &schemaList,
                                 &groupListList, &admingrpListList,
                                 &getListFieldsList, &sortListList, &indexListList,
                                 &defaultVuiList, &helpTextList, &timestampList,
                                 &ownerList, &lastChangedList, &changeDiaryList,
                                 &objPropListList, &status);
   EndAPICall(&status);

   /* Print results */
   DriverPrintResultHeader("ARGetMultipleSchemas");
   PrintReturnCode(result);
   if (result < AR_RETURN_ERROR)
   {
      if (nameList.numItems)
      {
         DriverPrintResultHeader("Exist list:");
         for (i = 0; i < nameList.numItems; i++)
         {
            DriverPrintResult("   Schema : %s\n", nameList.nameList[i]);
            PrintARBoolean("    exists: ", existList.booleanList[i]);
         }
      }
      for (i = 0; i < existList.numItems; i++)
      {
         if (existList.booleanList[i])
         {
            PrintChar("\nSchema name: ", schemaNameList.nameList[i]);
            PrintARCompoundSchema(&schemaList.compoundSchema[i]);
            PrintARPermissionList(&groupListList.permissionList[i], FALSE);
            PrintARInternalIdList("Admin group list: ", "   ",
                                  &admingrpListList.internalIdListList[i]);
            PrintAREntryListFieldList(&getListFieldsList.listFieldList[i]);
            PrintARSortList(&sortListList.sortListList[i]);
            PrintARIndexList(&indexListList.indexListList[i]);
            PrintChar("Default VUI: ", defaultVuiList.nameList[i]);
            PrintChar("Help Text: ", helpTextList.stringList[i]);
            PrintARTimestamp("Timestamp: ", timestampList.timestampList[i]);
            PrintChar("Owner: ", ownerList.nameList[i]);
            PrintChar("Last changed: ", lastChangedList.nameList[i]);
            APIPrintDiary("Change diary: ", changeDiaryList.stringList[i]);
            PrintARObjectPropList("Object property: ", "   ", &objPropListList.propsList[i]);
         }
      }
   }
   PrintARStatusList(&status);

   /* Cleanup */
   if (result < AR_RETURN_ERROR)
   {
      FreeARBooleanList(&existList, FALSE);
      FreeARNameList(&schemaNameList, FALSE);
      FreeARCompoundSchemaList(&schemaList, FALSE);
      FreeARPermissionListList(&groupListList, FALSE);
      FreeARInternalIdListList(&admingrpListList, FALSE);
      FreeAREntryListFieldListList(&getListFieldsList, FALSE);
      FreeARSortListList(&sortListList, FALSE);
      FreeARIndexListList(&indexListList, FALSE);
      FreeARNameList(&defaultVuiList, FALSE);
      FreeARTextStringList(&helpTextList, FALSE);
      FreeARTimestampList(&timestampList, FALSE);
      FreeARAccessNameList(&ownerList, FALSE);
      FreeARAccessNameList(&lastChangedList, FALSE);
      FreeARTextStringList(&changeDiaryList, FALSE);
      FreeARPropListList(&objPropListList, FALSE);
   }
   FreeARStatusList(&status, FALSE);
   FreeARUnsignedIntList(&schemaTypeList, FALSE);
   FreeARNameList(&nameList, FALSE);
   FreeARInternalIdList(&fieldIdList, FALSE);
}


/*****************************************************************************/
/*                                                                           */
/*                                APIARGetField                              */
/*                                                                           */
/*****************************************************************************/

void APIARGetField()

{
   char                   *changeDiary;
   unsigned int            createMode;
   unsigned int            dataType;
   ARValueStruct           defaultVal;
   ARDisplayInstanceList   dInstanceList;
   ARInternalId            fieldId;
   ARFieldMappingStruct    fieldMap;
   ARNameType              fieldName;
   char                   *helpText;
   ARAccessNameType        lastChanged;
   ARFieldLimitStruct      limit;  
   unsigned int            option;
   ARAccessNameType        owner;
   ARPermissionList        permissions;
   int                     result;
   ARNameType              schemaName;
   ARStatusList            status={0, NULL};
   ARTimestamp             timestamp;
   
   /* Get Parameters */
   DriverPrintHeader("GET FIELD");
   strcpy(schemaName, GetChar("Schema name: ", ""));
   fieldId = GetLong("Field id (1): ", 1);

   /* Call routine */
   BeginAPICall();
   result = ARGetField(GetControlStructPtr(), schemaName, fieldId, fieldName,
                       &fieldMap, &dataType, &option, &createMode, &defaultVal,
                       &permissions, &limit, &dInstanceList, &helpText,
                       &timestamp, owner,lastChanged, &changeDiary, &status);
   EndAPICall(&status);

   /* Print results */
   DriverPrintResultHeader("ARGetField");
   PrintReturnCode(result);
   if (result < AR_RETURN_ERROR)
   {
      PrintChar("Field name: ", fieldName);
      PrintARFieldMapping("Field mapping: ", &fieldMap);
      PrintInt("Datatype: ", dataType);
      PrintInt("Field option: ", option);
      PrintInt("Field create mode: ", createMode);
      PrintARValueStruct("", &defaultVal);
      PrintARPermissionList(&permissions, TRUE);
      PrintARFieldLimitStruct(&limit);
      PrintARDisplayInstanceList(&dInstanceList);
      PrintChar("Help text: ", helpText);
      PrintARTimestamp("Last update time: ", timestamp);
      PrintChar("Owner: ", owner);
      PrintChar("Last changed: ", lastChanged);
      APIPrintDiary("Change Diary: ", changeDiary);
   }
   PrintARStatusList(&status);

   /* Cleanup */
   if (result < AR_RETURN_ERROR)
   {
      FreeARValueStruct(&defaultVal, FALSE);
      FreeARPermissionList(&permissions, FALSE);
      FreeARFieldLimitStruct(&limit, FALSE);
      FreeARDisplayInstanceList(&dInstanceList, FALSE);
      if (helpText != NULL)
         free(helpText);
      if (changeDiary != NULL)
         free(changeDiary);
   }
   FreeARStatusList(&status, FALSE);
}


/*****************************************************************************/
/*                                                                           */
/*                                APIARSetField                              */
/*                                                                           */
/*****************************************************************************/

void APIARSetField()

{
   char                    changeDiary[1024];
   char                   *changeDiaryPtr;
   unsigned int            createMode;
   unsigned int           *createModePtr;
   unsigned int            dataType;
   ARValueStruct           defaultVal;
   ARValueStruct          *defaultValPtr;
   ARDisplayInstanceList   dInstanceList;
   ARDisplayInstanceList  *dInstanceListPtr;
   ARInternalId            fieldId;
   ARNameType              fieldName;
   char                   *fieldNamePtr;
   char                    helpText[1024];
   char                   *helpTextPtr;
   ARFieldLimitStruct      limit;
   ARFieldLimitStruct     *limitPtr;
   unsigned int            option;
   unsigned int           *optionPtr;
   ARAccessNameType        owner;
   char                   *ownerPtr;
   ARPermissionList        permissions;
   ARPermissionList       *permissionsPtr;
   int                     result;
   ARNameType              schemaName;
   ARStatusList            status={0, NULL};
   ARFieldMappingStruct    fieldMap;
   ARFieldMappingStruct   *fieldMapPtr;

   /* Get Parameters */
   DriverPrintHeader("SET FIELD");
   strcpy(schemaName, GetChar("Schema name: ", ""));
   fieldId = GetLong("Field id (1): ", 1);
   dataType = GetInt(
"Datatype Null/Key/Int/Real/Char/Diary/Enum/Time/Bitmask/Byte/Decimal/\n         attach/join/trim/control/Table/Column/ulong/coords/view/display\n         (0 - 11, 30-34, 40-43) (2): ",
                     AR_DATA_TYPE_INTEGER);
   if (GetARBoolean("Change fieldName? (F): ", FALSE))
   {
      strcpy(fieldName, GetChar("Field name (): ", ""));
      fieldNamePtr = fieldName;
   }
   else
      fieldNamePtr = NULL;
   if (GetARBoolean("Change field mapping? (F): ", FALSE))
   {
      GetARFieldMapping(&fieldMap);
      fieldMapPtr = &fieldMap;
   }
   else
      fieldMapPtr = NULL;
   if (GetARBoolean("Have new option? (F): ", FALSE))
   {
      option = GetInt("Required, Optional, System, or Display-only (1-4) (2): ",
                      2);
      optionPtr = &option;
   }
   else
      optionPtr = NULL;
   if (GetARBoolean("Have new create mode? (F): ", FALSE))
   {
      createMode = GetInt("Open or Protected at create (1 or 2) (1): ", 1);
      createModePtr = &createMode;
   }
   else
      createModePtr = NULL;
   if ((dataType != AR_DATA_TYPE_ATTACH) &&
       (dataType != AR_DATA_TYPE_TABLE) &&
       (dataType != AR_DATA_TYPE_COLUMN) &&
       (GetARBoolean("Have new default value? (F): ", FALSE)))
   {
      defaultVal.dataType = dataType;
      GetValue(&defaultVal);
      defaultValPtr = &defaultVal;
   }
   else
      defaultValPtr = NULL;
   if (GetARBoolean("Have new permissions? (F): ", FALSE))
   {
      GetARPermissionList(TRUE, &permissions);
      permissionsPtr = &permissions;
   }
   else
      permissionsPtr = NULL;
   if (GetARBoolean("Have new field limits? (F): ", FALSE))
   {
      GetARFieldLimitStruct(dataType, &limit);
      limitPtr = &limit;
   }
   else
      limitPtr = NULL;
   if (GetARBoolean("Have new display instance info? (F): ", FALSE))
   {
      GetARDisplayInstanceList(&dInstanceList);
      dInstanceListPtr = &dInstanceList;
   }
   else
      dInstanceListPtr = NULL;
   if (GetARBoolean("Change help text? (F): ", FALSE))
   {
      strcpy(helpText, GetChar("   Help Text (): ", ""));
      helpTextPtr = helpText;
   }
   else
      helpTextPtr = NULL;
   if (GetARBoolean("Change owner? (F): ", FALSE))
   {
      strcpy(owner, GetChar("Owner (): ", ""));
      ownerPtr = owner;
   }
   else
      ownerPtr = NULL;
   if (GetARBoolean("Add to change diary? (F): ", FALSE))
   {
      strcpy(changeDiary, GetChar("   Change Diary (): ", ""));
      changeDiaryPtr = changeDiary;
   }
   else
      changeDiaryPtr = NULL;

   /* Call routine */
   BeginAPICall();
   result = ARSetField(GetControlStructPtr(), schemaName, fieldId, fieldNamePtr,
                       fieldMapPtr, optionPtr, createModePtr, defaultValPtr,
                       permissionsPtr, limitPtr, dInstanceListPtr, helpTextPtr,
                       ownerPtr, changeDiaryPtr, &status);
   EndAPICall(&status);

   /* Print results */
   DriverPrintResultHeader("ARSetField");
   PrintReturnCode(result);
   PrintARStatusList(&status);

   /* Cleanup */
   if (defaultValPtr != NULL)
      FreeARValueStruct(&defaultVal, FALSE);
   if (permissionsPtr != NULL)
      FreeARPermissionList(&permissions, FALSE);
   if (limitPtr != NULL)
      FreeARFieldLimitStruct(&limit, FALSE);
   if (dInstanceListPtr != NULL)
      FreeARDisplayInstanceList(&dInstanceList, FALSE);
   FreeARStatusList(&status, FALSE);
}


/*****************************************************************************/
/*                                                                           */
/*                              APIARCreateField                             */
/*                                                                           */
/*****************************************************************************/

void APIARCreateField()

{
   char                    changeDiary[1024];
   unsigned int            createMode;
   unsigned int            dataType;
   ARValueStruct           defaultVal;
   ARDisplayInstanceList   dInstanceList;
   ARInternalId            fieldId;
   ARFieldMappingStruct    fieldMap;
   ARNameType              fieldName;
   char                    helpText[1024];
   ARFieldLimitStruct      limit;
   unsigned int            option;
   ARAccessNameType        owner;
   ARPermissionList        permissions;
   ARBoolean               requiredIdOK;
   int                     result;
   ARNameType              schemaName;
   ARStatusList            status={0, NULL};

   /* Get Parameters */
   DriverPrintHeader("CREATE FIELD");
   strcpy(schemaName, GetChar("Schema name: ", ""));
   fieldId = GetLong("Field id (0): ", 0);
   requiredIdOK = GetARBoolean("Create even if ID is reserved? (F): ", FALSE);
   strcpy(fieldName, GetChar("Field name: ", ""));
   dataType = GetInt(
"Datatype Null/Key/Int/Real/Char/Diary/Enum/Time/Bitmask/Byte/Decimal/\n         attach/join/trim/control/table/column/ulong/coords/view/display\n         (0 - 11, 30-34, 40-43) (2): ",
                     AR_DATA_TYPE_INTEGER);
   option = GetInt("Required, Optional, System, or Display-only (1-4) (2): ",2);
   createMode = GetInt("Open or Protected at create (1 or 2) (1): ", 1);
   if ((dataType != AR_DATA_TYPE_ATTACH) &&
       (dataType != AR_DATA_TYPE_TABLE) &&
       (dataType != AR_DATA_TYPE_COLUMN) &&
       (GetARBoolean("Have default value? (T): ", TRUE)))
   {
      defaultVal.dataType = dataType;
      GetValue(&defaultVal);
   }
   else
   {
      memset(&defaultVal, '\0', sizeof(defaultVal));
      defaultVal.dataType = AR_DEFAULT_VALUE_NONE;
   }
   GetARPermissionList(TRUE, &permissions);
   if (GetARBoolean("Have field limits? (T): ", TRUE))
      GetARFieldLimitStruct(dataType, &limit);
   else
      limit.dataType = AR_FIELD_LIMIT_NONE;
   GetARDisplayInstanceList(&dInstanceList);
   strcpy(helpText, GetChar("Help Text: ", ""));
   strcpy(owner, GetChar("Owner: ", GetControlStructPtr()->user));
   strcpy(changeDiary, GetChar("Change Diary: ", ""));
   GetARFieldMapping(&fieldMap);

   /* Call routine */
   BeginAPICall();
   result = ARCreateField(GetControlStructPtr(), schemaName, &fieldId,
                          requiredIdOK, fieldName, &fieldMap, dataType, option,
                          createMode, &defaultVal, &permissions, &limit,
                          &dInstanceList, helpText, owner, changeDiary,
                          &status);
   EndAPICall(&status);

   /* Print results */
   DriverPrintResultHeader("ARCreateField");
   PrintReturnCode(result);
   PrintARInternalId("Field id: ", fieldId);
   PrintARStatusList(&status);

   /* Cleanup */
   FreeARValueStruct(&defaultVal, FALSE);
   FreeARPermissionList(&permissions, FALSE);
   FreeARFieldLimitStruct(&limit, FALSE);
   FreeARDisplayInstanceList(&dInstanceList, FALSE);
   FreeARStatusList(&status, FALSE);
}


/*****************************************************************************/
/*                                                                           */
/*                              APIARDeleteField                             */
/*                                                                           */
/*****************************************************************************/

void APIARDeleteField()

{
   unsigned int   deleteOption;
   ARInternalId   fieldId;
   int            result;
   ARNameType     schemaName;
   ARStatusList   status={0, NULL};

   /* Get Parameters */
   DriverPrintHeader("DELETE FIELD");
   strcpy(schemaName, GetChar("Schema name: ", ""));
   fieldId = GetLong("Field id (0): ", 0);
   deleteOption = GetInt("Delete option - clean, data, cascade (0 - 2) (0): ",
                         0);

   /* Call routine */
   BeginAPICall();
   result = ARDeleteField(GetControlStructPtr(), schemaName, fieldId,
                          deleteOption, &status);
   EndAPICall(&status);

   /* Print results */
   DriverPrintResultHeader("ARDeleteField");
   PrintReturnCode(result);
   PrintARStatusList(&status);

   /* Cleanup */
   FreeARStatusList(&status, FALSE);
}


/*****************************************************************************/
/*                                                                           */
/*                          APIARDeleteMultipleFields                        */
/*                                                                           */
/*****************************************************************************/

void APIARDeleteMultipleFields()

{
   unsigned int       deleteOption;
   ARInternalIdList   fieldIdList;
   int                result;
   ARNameType         schemaName;
   ARStatusList       status={0, NULL};

   /* Get Parameters */
   DriverPrintHeader("DELETE MULTIPLE FIELDS");
   strcpy(schemaName, GetChar("Schema name: ", ""));
   DriverPrintPrompt("Ids of fields to delete:\n");
   GetARInternalIdList(&fieldIdList);
   deleteOption = GetInt("Delete option - clean, data, cascade (0 - 2) (0): ",
                         0);
 
   /* Call routine */
   BeginAPICall();
   result = ARDeleteMultipleFields(GetControlStructPtr(), schemaName,
                                   &fieldIdList, deleteOption, &status);
   EndAPICall(&status);

   /* Print results */
   DriverPrintResultHeader("ARDeleteMultipleFields");
   PrintReturnCode(result);
   PrintARStatusList(&status);
 
   /* Cleanup */
   FreeARInternalIdList(&fieldIdList, FALSE);
   FreeARStatusList(&status, FALSE);
}


/*****************************************************************************/
/*                                                                           */
/*                              APIARGetListField                            */
/*                                                                           */
/*****************************************************************************/

void APIARGetListField()

{
   ARTimestamp        changedSince;
   ARInternalIdList   fieldIdList;
   unsigned long      fieldType;
   int                result;
   ARNameType         schemaName;
   ARStatusList       status={0, NULL};

   /* Get Parameters */
   DriverPrintHeader("GETLIST FIELD");
   strcpy(schemaName, GetChar("Schema name: ", ""));
   changedSince = GetARTimestamp("Get all changed since (0): ",
                                 (ARTimestamp) 0);
   fieldType = GetLong("Field Type - data, trim, control, page, page holder, table, column, attach,\n             attach pool  (1, 2, 4, 8, 16, 32, 64, 128, 256) (1): ", 1);

   /* Call routine */
   BeginAPICall();
   result = ARGetListField(GetControlStructPtr(), schemaName, fieldType,
                           changedSince, &fieldIdList, &status);
   EndAPICall(&status);

   /* Print results */
   DriverPrintResultHeader("ARGetListField");
   PrintReturnCode(result);
   if (result < AR_RETURN_ERROR)
      PrintARInternalIdList("Field id list: ", "   ", &fieldIdList);
   PrintARStatusList(&status);

   /* Cleanup */
   if (result < AR_RETURN_ERROR)
      FreeARInternalIdList(&fieldIdList, FALSE);
   FreeARStatusList(&status, FALSE);
}


/*****************************************************************************/
/*                                                                           */
/*                            APIARGetMultipleFields                         */
/*                                                                           */
/*****************************************************************************/

void APIARGetMultipleFields()

{
   ARTextStringList          changeDiary;
   ARUnsignedIntList         createMode;
   ARUnsignedIntList         dataType;
   ARValueList               defaultVal;
   ARDisplayInstanceListList dInstanceList;
   ARBooleanList             existList;
   ARInternalIdList          fieldIdIn;
   ARInternalIdList          fieldIdOut;
   ARFieldMappingList        fieldMap;
   ARNameList                fieldName;
   ARTextStringList          helpText;
   ARAccessNameList          lastChanged;
   ARFieldLimitList          limit;  
   ARUnsignedIntList         option;
   ARAccessNameList          owner;
   ARPermissionListList      permissions;
   int                       result;
   ARNameType                schemaName;
   ARStatusList              status={0, NULL};
   ARTimestampList           timestamp;
   
   /* Get Parameters */
   DriverPrintHeader("GET MULTIPLE FIELDS");
   strcpy(schemaName, GetChar("Schema name: ", ""));
   DriverPrintPrompt("Ids of fields to retrieve:\n");
   GetARInternalIdList(&fieldIdIn);

   /* Call routine */
   BeginAPICall();
   result = ARGetMultipleFields(GetControlStructPtr(), schemaName, &fieldIdIn,
                       &existList, &fieldIdOut, &fieldName, &fieldMap,
                       &dataType, &option, &createMode, &defaultVal,
                       &permissions, &limit, &dInstanceList, &helpText,
                       &timestamp, &owner, &lastChanged, &changeDiary, &status);
   EndAPICall(&status);

   /* Print results */
   DriverPrintResultHeader("ARGetMultipleFields");
   PrintReturnCode(result);
   if (result < AR_RETURN_ERROR)
   {
      PrintARBooleanList("Field exists list", &existList);
      PrintARInternalIdList("Field ID list", "   ", &fieldIdOut);
      PrintARNameList("Field name list", "   ", &fieldName);
      PrintARFieldMappingList("Field mapping list", &fieldMap);
      PrintARUnsignedIntList("Datatype list", "   ", &dataType);
      PrintARUnsignedIntList("Field option list", "   ", &option);
      PrintARUnsignedIntList("Field create mode list", "   ", &createMode);
      PrintARValueList("", &defaultVal);
      PrintARPermissionListList("Permissions", &permissions);
      PrintARFieldLimitList("Field limits", &limit);
      PrintARDisplayInstanceListList("Display Instances", &dInstanceList);
      PrintARTextStringList("Help text list", "   Help text: ", &helpText);
      PrintARTimestampList("Last update time list: ", "   ", &timestamp);
      PrintARAccessNameList("Owner list", "   ", &owner);
      PrintARAccessNameList("Last changed list", "   ", &lastChanged);
      APIPrintDiaryList("Change Diary list", "   Change diary: ", &changeDiary);
   }
   PrintARStatusList(&status);

   /* Cleanup */
   if (result < AR_RETURN_ERROR)
   {
      FreeARInternalIdList(&fieldIdIn, FALSE);
      FreeARBooleanList(&existList, FALSE);
      FreeARInternalIdList(&fieldIdOut, FALSE);
      FreeARNameList(&fieldName, FALSE);
      FreeARFieldMappingList(&fieldMap, FALSE);
      FreeARUnsignedIntList(&dataType, FALSE);
      FreeARUnsignedIntList(&option, FALSE);
      FreeARUnsignedIntList(&createMode, FALSE);
      FreeARValueList(&defaultVal, FALSE);
      FreeARPermissionListList(&permissions, FALSE);
      FreeARFieldLimitList(&limit, FALSE);
      FreeARDisplayInstanceListList(&dInstanceList, FALSE);
      FreeARTextStringList(&helpText, FALSE);
      FreeARTimestampList(&timestamp, FALSE);
      FreeARAccessNameList(&owner, FALSE);
      FreeARAccessNameList(&lastChanged, FALSE);
      FreeARTextStringList(&changeDiary, FALSE);
   }
   FreeARStatusList(&status, FALSE);
}


/*****************************************************************************/
/*                                                                           */
/*                                APIARGetVUI                                */
/*                                                                           */
/*****************************************************************************/

void APIARGetVUI()

{
   char          *changeDiary;
   ARPropList     dPropList;
   char          *helpText;
   ARAccessNameType  lastChanged;
   ARLocaleType   locale;
   unsigned int   vuiType;
   ARAccessNameType  owner;
   int            result;
   ARNameType     schemaName;
   ARStatusList   status={0, NULL};
   ARTimestamp    timestamp;
   ARInternalId   vuiId;
   ARNameType     vuiName;

   /* Get Parameters */
   DriverPrintHeader("GET VUI");
   strcpy(schemaName, GetChar("Schema name: ", ""));
   vuiId = GetLong("VUI id (1): ", 1);

   /* Call routine */
   BeginAPICall();
   result = ARGetVUI(GetControlStructPtr(), schemaName, vuiId, vuiName,
                     locale, &vuiType, &dPropList, &helpText, &timestamp,
                     owner, lastChanged, &changeDiary, &status);
   EndAPICall(&status);

   /* Print results */
   DriverPrintResultHeader("ARGetVUI");
   PrintReturnCode(result);
   if (result < AR_RETURN_ERROR)
   {
      PrintChar("VUI name: ", vuiName);
      PrintChar("Locale: ", locale);
      PrintInt("VUI-Type: ", vuiType);
      PrintARDisplayPropList("", "", &dPropList);
      PrintChar("Help text: ", helpText);
      PrintARTimestamp("Last update time: ", timestamp);
      PrintChar("Owner: ", owner);
      PrintChar("Last changed: ", lastChanged);
      APIPrintDiary("Change Diary: ", changeDiary);
   }
   PrintARStatusList(&status);

   /* Cleanup */
   if (result < AR_RETURN_ERROR)
   {
      FreeARPropList(&dPropList, FALSE);
      if (helpText != NULL)
         free(helpText);
      if (changeDiary != NULL)
         free(changeDiary);
   }
   FreeARStatusList(&status, FALSE);
}


/*****************************************************************************/
/*                                                                           */
/*                                APIARSetVUI                                */
/*                                                                           */
/*****************************************************************************/

void APIARSetVUI()

{
   char           changeDiary[1024];
   char          *changeDiaryPtr;
   ARPropList     dPropList;
   ARPropList    *dPropListPtr;
   char           helpText[1024];
   char          *helpTextPtr;
   ARAccessNameType  owner;
   char          *ownerPtr;
   int            result;
   ARNameType     schemaName;
   ARStatusList   status={0, NULL};
   ARInternalId   vuiId;
   ARNameType     vuiName;
   char          *vuiNamePtr;
   ARLocaleType   locale;
   char          *localePtr;
   unsigned int   vuiType;
   unsigned int  *vuiTypePtr;

   /* Get Parameters */
   DriverPrintHeader("SET VUI");
   strcpy(schemaName, GetChar("Schema name: ", ""));
   vuiId = GetLong("VUI id (1): ", 1);
   if (GetARBoolean("Change VUI name? (F): ", FALSE))
   {
      strcpy(vuiName, GetChar("VUI name(): ", ""));
      vuiNamePtr = vuiName;
   }
   else
      vuiNamePtr = NULL;
   if (GetARBoolean("Change Locale? (F): ", FALSE))
   {
      strcpy(locale, GetChar("Locale(): ", ""));
      localePtr = locale;
   }
   else
      localePtr = NULL;
   if (GetARBoolean("Change VUI-Type? (F): ", FALSE))
   {
      vuiType = GetInt("VUI-Type (0): ", 0);
      vuiTypePtr = &vuiType;
   }
   else
      vuiTypePtr = NULL;
   if (GetARBoolean("Have new Prop List info? (F): ", FALSE))
   {
      GetARPropList(&dPropList);
      dPropListPtr = &dPropList;
   }
   else
      dPropListPtr = NULL;
   if (GetARBoolean("Change help text? (F): ", FALSE))
   {
      strcpy(helpText, GetChar("   Help Text (): ", ""));
      helpTextPtr = helpText;
   }
   else
      helpTextPtr = NULL;
   if (GetARBoolean("Change owner? (F): ", FALSE))
   {
      strcpy(owner, GetChar("Owner (): ", ""));
      ownerPtr = owner;
   }
   else
      ownerPtr = NULL;
   if (GetARBoolean("Add to change diary? (F): ", FALSE))
   {
      strcpy(changeDiary, GetChar("   Change Diary (): ", ""));
      changeDiaryPtr = changeDiary;
   }
   else
      changeDiaryPtr = NULL;

   /* Call routine */
   BeginAPICall();
   result = ARSetVUI(GetControlStructPtr(), schemaName, vuiId, vuiNamePtr,
                     localePtr, vuiTypePtr, dPropListPtr, helpTextPtr,
                     ownerPtr, changeDiaryPtr, &status);
   EndAPICall(&status);

   /* Print results */
   DriverPrintResultHeader("ARSetVUI");
   PrintReturnCode(result);
   PrintARStatusList(&status);

   /* Cleanup */
   if (dPropListPtr != NULL)
      FreeARPropList(&dPropList, FALSE);
   FreeARStatusList(&status, FALSE);
}


/*****************************************************************************/
/*                                                                           */
/*                              APIARCreateVUI                               */
/*                                                                           */
/*****************************************************************************/

void APIARCreateVUI()

{
   char           changeDiary[1024];
   ARPropList     dPropList;
   char           helpText[1024];
   ARAccessNameType  owner;
   int            result;
   ARNameType     schemaName;
   ARStatusList   status={0, NULL};
   ARInternalId   vuiId;
   ARNameType     vuiName;
   ARLocaleType   locale;
   unsigned int   vuiType;

   /* Get Parameters */
   DriverPrintHeader("CREATE VUI");
   strcpy(schemaName, GetChar("Schema name: ", ""));
   vuiId = GetLong("VUI id (0): ", 0);
   strcpy(vuiName, GetChar("VUI name: ", ""));
   strcpy(locale, GetChar("Locale: ", ""));
   vuiType = GetInt("VUI-Type (0): ", 0);
   GetARPropList(&dPropList);
   strcpy(helpText, GetChar("Help Text: ", ""));
   strcpy(owner, GetChar("Owner: ", GetControlStructPtr()->user));
   strcpy(changeDiary, GetChar("Change Diary: ", ""));

   /* Call routine */
   BeginAPICall();
   result = ARCreateVUI(GetControlStructPtr(), schemaName, &vuiId, vuiName,
                        locale, vuiType, &dPropList, helpText, owner,
                        changeDiary, &status);
   EndAPICall(&status);

   /* Print results */
   DriverPrintResultHeader("ARCreateVUI");
   PrintReturnCode(result);
   PrintARInternalId("VUI id: ", vuiId);
   PrintARStatusList(&status);

   /* Cleanup */
   FreeARPropList(&dPropList, FALSE);
   FreeARStatusList(&status, FALSE);
}


/*****************************************************************************/
/*                                                                           */
/*                              APIARDeleteVUI                               */
/*                                                                           */
/*****************************************************************************/

void APIARDeleteVUI()

{
   int            result;
   ARNameType     schemaName;
   ARStatusList   status={0, NULL};
   ARInternalId   vuiId;

   /* Get Parameters */
   DriverPrintHeader("DELETE VUI");
   strcpy(schemaName, GetChar("Schema name: ", ""));
   vuiId = GetLong("VUI id (0): ", 0);

   /* Call routine */
   BeginAPICall();
   result = ARDeleteVUI(GetControlStructPtr(), schemaName, vuiId, &status);
   EndAPICall(&status);

   /* Print results */
   DriverPrintResultHeader("ARDeleteVUI");
   PrintReturnCode(result);
   PrintARStatusList(&status);

   /* Cleanup */
   FreeARStatusList(&status, FALSE);
}


/*****************************************************************************/
/*                                                                           */
/*                              APIARGetListVUI                              */
/*                                                                           */
/*****************************************************************************/

void APIARGetListVUI()

{
   ARTimestamp        changedSince;
   int                result;
   ARNameType         schemaName;
   ARStatusList       status={0, NULL};
   ARInternalIdList   vuiIdList;

   /* Get Parameters */
   DriverPrintHeader("GETLIST VUI");
   strcpy(schemaName, GetChar("Schema name: ", ""));
   changedSince = GetARTimestamp("Get all changed since (0): ",
                                 (ARTimestamp) 0);

   /* Call routine */
   BeginAPICall();
   result = ARGetListVUI(GetControlStructPtr(), schemaName, changedSince,
                         &vuiIdList, &status);
   EndAPICall(&status);

   /* Print results */
   DriverPrintResultHeader("ARGetListVUI");
   PrintReturnCode(result);
   if (result < AR_RETURN_ERROR)
      PrintARInternalIdList("VUI id list: ", "   ", &vuiIdList);
   PrintARStatusList(&status);

   /* Cleanup */
   if (result < AR_RETURN_ERROR)
      FreeARInternalIdList(&vuiIdList, FALSE);
   FreeARStatusList(&status, FALSE);
}


/*****************************************************************************/
/*                                                                           */
/*                              APIARGetCharMenu                             */
/*                                                                           */
/*****************************************************************************/

void APIARGetCharMenu()

{
   char              *changeDiary;
   char              *helpText;
   ARAccessNameType   lastChanged;
   ARCharMenuStruct   menuDefn;
   ARNameType         name;
   ARPropList         objPropList;
   ARAccessNameType   owner;
   unsigned int       refreshCode;
   int                result;
   ARStatusList       status={0, NULL};
   ARTimestamp        timestamp;

   /* Get Parameters */
   DriverPrintHeader("GET CHAR MENU");
   strcpy(name, GetChar("Character Menu name: ", ""));

   /* Call routine */
   BeginAPICall();
   result = ARGetCharMenu(GetControlStructPtr(), name, &refreshCode, &menuDefn,
                          &helpText, &timestamp, owner, lastChanged,
                          &changeDiary, &objPropList, &status);
   EndAPICall(&status);

   /* Print results */
   DriverPrintResultHeader("ARGetCharMenu");
   PrintReturnCode(result);
   if (result < AR_RETURN_ERROR)
   {
      PrintInt("Refresh code: ", refreshCode);
      PrintARCharMenuStruct("", &menuDefn);
      PrintChar("Help text: ", helpText);
      PrintARTimestamp("Last update time: ", timestamp);
      PrintChar("Owner: ", owner);
      PrintChar("Last changed: ", lastChanged);
      APIPrintDiary("Change Diary: ", changeDiary);
      PrintARObjectPropList("", "", &objPropList);
   }
   PrintARStatusList(&status);

   /* Cleanup */
   if (result < AR_RETURN_ERROR)
   {
      FreeARCharMenuStruct(&menuDefn, FALSE);
      FreeARPropList(&objPropList, FALSE);
      if (helpText != NULL)
         free(helpText);
      if (changeDiary != NULL)
         free(changeDiary);
   }
   FreeARStatusList(&status, FALSE);
}


/*****************************************************************************/
/*                                                                           */
/*                              APIARSetCharMenu                             */
/*                                                                           */
/*****************************************************************************/

void APIARSetCharMenu()

{
   char               changeDiary[1024];
   char              *changeDiaryPtr;
   char               helpText[1024];
   char              *helpTextPtr;
   ARCharMenuStruct   menuDefn;
   ARCharMenuStruct  *menuDefnPtr;
   ARNameType         name;
   ARNameType         newName;
   char              *newNamePtr;
   ARPropList         objPropList;
   ARPropList        *objPropListPtr;
   ARAccessNameType   owner;
   char              *ownerPtr;
   unsigned int       refreshCode;
   unsigned int      *refreshCodePtr;
   int                result;
   ARStatusList       status={0, NULL};

   /* Get Parameters */
   DriverPrintHeader("SET CHAR MENU");
   strcpy(name, GetChar("Character Menu name: ", ""));
   if (GetARBoolean("Change character menu name? (F): ", FALSE))
   {
      strcpy(newName, GetChar("New name (): ", ""));
      newNamePtr = newName;
   }
   else
      newNamePtr = NULL;
   if (GetARBoolean("Change refresh code? (F): ", FALSE))
   {
      refreshCode = GetInt("Refresh on connect, open, interval (1 - 3) (1): ",
                           1);
      refreshCodePtr = &refreshCode;
   }
   else
      refreshCodePtr = NULL;
   if (GetARBoolean("Change menu definition(s)? (F): ", FALSE))
   {
      DriverPrintPrompt("Character menu struct:\n");
      GetARCharMenuStruct(&menuDefn);
      menuDefnPtr = &menuDefn;
   }
   else
      menuDefnPtr = NULL;
   if (GetARBoolean("Change help text? (F): ", FALSE))
   {
      strcpy(helpText, GetChar("   Help Text (): ", ""));
      helpTextPtr = helpText;
   }
   else
      helpTextPtr = NULL;
   if (GetARBoolean("Change owner? (F): ", FALSE))
   {
      strcpy(owner, GetChar("   Owner (): ", ""));
      ownerPtr = owner;
   }
   else
      ownerPtr = NULL;
   if (GetARBoolean("Add to change diary? (F): ", FALSE))
   {
      strcpy(changeDiary, GetChar("   Change Diary (): ", ""));
      changeDiaryPtr = changeDiary;
   }
   else
      changeDiaryPtr = NULL;
   if (GetARBoolean("Change object properties? (F): ", FALSE))
   {
      GetARPropList(&objPropList);
      objPropListPtr = &objPropList;
   }
   else
      objPropListPtr = NULL;

   /* Call routine */
   BeginAPICall();
   result = ARSetCharMenu(GetControlStructPtr(), name, newNamePtr,
                          refreshCodePtr, menuDefnPtr, helpTextPtr, ownerPtr,
                          changeDiaryPtr, objPropListPtr, &status);
   EndAPICall(&status);

   /* Print results */
   DriverPrintResultHeader("ARSetCharMenu");
   PrintReturnCode(result);
   PrintARStatusList(&status);

   /* Cleanup */
   if (menuDefnPtr != NULL)
      FreeARCharMenuStruct(&menuDefn, FALSE);
   if (objPropListPtr != NULL)
      FreeARPropList(&objPropList, FALSE);
   FreeARStatusList(&status, FALSE);
}


/*****************************************************************************/
/*                                                                           */
/*                            APIARCreateCharMenu                            */
/*                                                                           */
/*****************************************************************************/

void APIARCreateCharMenu()

{
   char               changeDiary[1024];
   char               helpText[1024];
   ARCharMenuStruct   menuDefn;
   ARNameType         name;
   ARPropList         objPropList;
   ARAccessNameType   owner;
   unsigned int       refreshCode;
   int                result;
   ARStatusList       status={0, NULL};

   /* Get Parameters */
   DriverPrintHeader("CREATE CHAR MENU");
   strcpy(name, GetChar("Character Menu name: ", ""));
   refreshCode = GetInt("Refresh on connect, open, interval (1 - 3) (1): ", 1);
   DriverPrintPrompt("Character menu struct:\n");
   GetARCharMenuStruct(&menuDefn);
   strcpy(helpText, GetChar("Help Text (): ", ""));
   strcpy(owner, GetChar("Owner (): ", ""));
   strcpy(changeDiary, GetChar("Change Diary (): ", ""));
   GetARPropList(&objPropList);

   /* Call routine */
   BeginAPICall();
   result = ARCreateCharMenu(GetControlStructPtr(), name, refreshCode,
                             &menuDefn, helpText, owner, changeDiary,
                             &objPropList, &status);
   EndAPICall(&status);

   /* Print results */
   DriverPrintResultHeader("ARCreateCharMenu");
   PrintReturnCode(result);
   PrintARStatusList(&status);

   /* Cleanup */
   FreeARCharMenuStruct(&menuDefn, FALSE);
   FreeARPropList(&objPropList, FALSE);
   FreeARStatusList(&status, FALSE);
}


/*****************************************************************************/
/*                                                                           */
/*                            APIARDeleteCharMenu                            */
/*                                                                           */
/*****************************************************************************/

void APIARDeleteCharMenu()

{
   ARNameType     name;
   int            result; 
   ARStatusList   status={0, NULL};

   /* Get Parameters */
   DriverPrintHeader("DELETE CHAR MENU");
   strcpy(name, GetChar("Character Menu name: ", ""));

   /* Call routine */
   BeginAPICall();
   result = ARDeleteCharMenu(GetControlStructPtr(), name, &status);
   EndAPICall(&status);

   /* Print results */
   DriverPrintResultHeader("ARDeleteCharMenu");
   PrintReturnCode(result);
   PrintARStatusList(&status);

   /* Cleanup */
   FreeARStatusList(&status, FALSE);
}


/*****************************************************************************/
/*                                                                           */
/*                            APIARGetListCharMenu                           */
/*                                                                           */
/*****************************************************************************/

void APIARGetListCharMenu()

{
   ARTimestamp    changedSince;
   ARNameList     nameList;
   int            result;
   ARStatusList   status={0, NULL};

   /* Get Parameters */
   DriverPrintHeader("GETLIST CHAR MENU");
   changedSince = GetARTimestamp("Get all changed since (0): ",
                                 (ARTimestamp) 0);

   /* Call routine */
   BeginAPICall();
   result = ARGetListCharMenu(GetControlStructPtr(), changedSince, &nameList,
                              &status);
   EndAPICall(&status);

   /* Print results */
   DriverPrintResultHeader("ARGetListCharMenu");
   PrintReturnCode(result);
   if (result < AR_RETURN_ERROR)
      PrintARNameList("Character Menu list: ", "   ", &nameList);
   PrintARStatusList(&status);

   /* Cleanup */
   if (result < AR_RETURN_ERROR)
      FreeARNameList(&nameList, FALSE);
   FreeARStatusList(&status, FALSE);
}


/*****************************************************************************/
/*                                                                           */
/*                            APIARExpandCharMenu                            */
/*                                                                           */
/*****************************************************************************/

void APIARExpandCharMenu()

{
   ARCharMenuStruct   menu;
   ARCharMenuStruct   menuDefn; 
   int                result;
   ARStatusList       status={0, NULL};

   /* Get Parameters */
   DriverPrintHeader("EXPAND CHAR MENU");
   DriverPrintPrompt("Character menu struct:\n");
   GetARCharMenuStruct(&menuDefn);

   /* Call routine */
   BeginAPICall();
   result = ARExpandCharMenu(GetControlStructPtr(), &menuDefn, &menu, &status);
   EndAPICall(&status);

   /* Print results */
   DriverPrintResultHeader("ARExpandCharMenu");
   PrintReturnCode(result);
   if (result < AR_RETURN_ERROR)
      PrintARCharMenuStruct("", &menu);
   PrintARStatusList(&status);

   /* Cleanup */
   FreeARCharMenuStruct(&menuDefn, FALSE);
   if (result < AR_RETURN_ERROR)
      FreeARCharMenuStruct(&menu, FALSE);
   FreeARStatusList(&status, FALSE);
}


/*****************************************************************************/
/**********                                                         **********/
/**********                          FILTER                         **********/
/**********                                                         **********/
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*                               APIARGetFilter                              */
/*                                                                           */
/*****************************************************************************/

void APIARGetFilter()

{
   ARFilterActionList        actionList;
   char                     *changeDiary;
   ARFilterActionList        elseList;
   unsigned int              enable;
   char                     *helpText;
   ARAccessNameType          lastChanged;
   ARNameType                name;
   ARPropList                objPropList;
   unsigned int              opSet;
   unsigned int              order;
   ARAccessNameType          owner;
   ARQualifierStruct         query;
   int                       result;
   ARStatusList              status={0, NULL};
   ARTimestamp               timestamp;
   ARWorkflowConnectStruct   workflowConnect;

   /* Get Parameters */
   DriverPrintHeader("GET FILTER");
   strcpy(name, GetChar("Filter name: ", ""));

   /* Call routine */
   BeginAPICall();
   result = ARGetFilter(GetControlStructPtr(), name, &order, &workflowConnect,
                        &opSet, &enable, &query, &actionList, &elseList,
                        &helpText, &timestamp, owner, lastChanged, &changeDiary,
                        &objPropList, &status);
   EndAPICall(&status);

   /* Print results */
   DriverPrintResultHeader("ARGetFilter");
   PrintReturnCode(result);
   if (result < AR_RETURN_ERROR)
   {
      PrintInt("Filter order: ", order);
      PrintInt("Workflow connect type: ", workflowConnect.type);
      if (workflowConnect.type == AR_WORKFLOW_CONN_SCHEMA_LIST)
      {
         PrintInt("Number of schema list items is: ",
                  workflowConnect.u.schemaList->numItems);
         PrintARNameList("Schema list: ", " ", workflowConnect.u.schemaList);    
      }
      PrintInt("Operations bitmask: ", opSet);
      PrintInt("Enable flag : ", enable);
      PrintARQualifierStruct("", &query);
      PrintARFilterActionList(&actionList, TRUE);
      PrintARFilterActionList(&elseList, FALSE);
      PrintChar("Help text: ", helpText);
      PrintARTimestamp("Last update time: ", timestamp);
      PrintChar("Owner: ", owner);
      PrintChar("Last changed: ", lastChanged);
      APIPrintDiary("Change Diary: ", changeDiary);
      PrintARObjectPropList("", "", &objPropList);
   }
   PrintARStatusList(&status);

   /* Cleanup */
   if (result < AR_RETURN_ERROR)
   {
      FreeARQualifierStruct(&query, FALSE);
      FreeARFilterActionList(&actionList, FALSE);
      FreeARFilterActionList(&elseList, FALSE);
      if (helpText != NULL)
         free(helpText);
      if (changeDiary != NULL)
         free(changeDiary);
      FreeARWorkflowConnectStruct(&workflowConnect, FALSE);
      FreeARPropList(&objPropList, FALSE);
   }
   FreeARStatusList(&status, FALSE);
}


/*****************************************************************************/
/*                                                                           */
/*                               APIARSetFilter                              */
/*                                                                           */
/*****************************************************************************/

void APIARSetFilter()

{
   ARFilterActionList        actionList;
   ARFilterActionList       *actionListPtr;
   char                      changeDiary[1024];
   char                     *changeDiaryPtr;
   ARFilterActionList        elseList;
   ARFilterActionList       *elseListPtr;
   unsigned int              enable;
   unsigned int             *enablePtr;
   char                      helpText[1024];
   char                     *helpTextPtr;
   ARNameType                name;
   ARNameType                newName;
   char                     *newNamePtr;
   unsigned int              numItems;
   ARPropList                objPropList;
   ARPropList               *objPropListPtr;
   unsigned int              opSet;
   unsigned int             *opSetPtr;
   unsigned int              order;
   unsigned int             *orderPtr;
   ARAccessNameType          owner;
   char                     *ownerPtr;
   ARQualifierStruct         query;
   ARQualifierStruct        *queryPtr;
   int                       result;  
   ARStatusList              status={0, NULL};
   char                      tmpString[24];
   ARWorkflowConnectStruct   workflowConnect;
   ARWorkflowConnectStruct  *workflowConnectPtr;

   /* Get Parameters */
   DriverPrintHeader("SET FILTER");
   strcpy(name, GetChar("Filter name: ", ""));
   if (GetARBoolean("Change filter name? (F): ", FALSE))
   {
      strcpy(newName, GetChar("New name (): ", ""));
      newNamePtr = newName;
   }
   else
      newNamePtr = NULL;
   if (GetARBoolean("Change filter order? (F): ", FALSE))
   {
      order = GetInt("Filter order (0 - 1000) (500): ", 500);
      orderPtr = &order;
   }
   else
      orderPtr = NULL;
   workflowConnect.type = GetInt("Workflow type (1): ", 1);
   if (GetARBoolean("Change schema or schemaList? (F): ", FALSE))
   {
      if (workflowConnect.type == AR_WORKFLOW_CONN_SCHEMA_LIST)
      {
         workflowConnect.u.schemaList = (ARNameList *) malloc(sizeof(ARNameList));
         if (workflowConnect.u.schemaList == NULL)
         {
            DriverPrintError("**** malloc failure for ARSetFilter call ****\n");
            return;
         }
         workflowConnect.u.schemaList->numItems = GetInt("Number of schemas? (1): ", 1);
         workflowConnect.u.schemaList->nameList = 
            (ARNameType *) malloc(workflowConnect.u.schemaList->numItems *
                                  sizeof(ARNameType));
         if (workflowConnect.u.schemaList->nameList == NULL)
         {
            DriverPrintError("**** malloc failure for ARSetFilter call ****\n");
            free(workflowConnect.u.schemaList);
            return;
         }
         for (numItems = 0; numItems < workflowConnect.u.schemaList->numItems;
              numItems++)
         {
            sprintf(tmpString, "Schema name %u:", numItems + 1);
            strcpy(workflowConnect.u.schemaList->nameList[numItems], GetChar(tmpString, ""));
         }
      }
      workflowConnectPtr = &workflowConnect; 
   }
   else
      workflowConnectPtr = NULL;


   if (GetARBoolean("Change operations mask? (F): ", FALSE))
   {
      opSet = GetInt("Operations bit mask (0): ", 0);
      opSetPtr = &opSet;
   }
   else
      opSetPtr = NULL;
   if (GetARBoolean("Change enable/disable? (F): ", FALSE))
   {
      enable = GetInt("Disable/enable (0-1) (1): ", 1);
      enablePtr = &enable;
   }
   else
      enablePtr = NULL;
   if (GetARBoolean("Change query? (F): ", FALSE))
   {
      GetARQualifierStruct(&query);
      queryPtr = &query;
   }
   else
      queryPtr = NULL;
   if (GetARBoolean("Change action(s)? (F): ", FALSE))
   {
      GetARFilterActionList(TRUE, &actionList);
      actionListPtr = &actionList;
   }
   else
      actionListPtr = NULL;
   if (GetARBoolean("Change else(s)? (F): ", FALSE))
   {
      GetARFilterActionList(FALSE, &elseList);
      elseListPtr = &elseList;
   }
   else
      elseListPtr = NULL;
   if (GetARBoolean("Change help text? (F): ", FALSE))
   {
      strcpy(helpText, GetChar("   Help Text (): ", ""));
      helpTextPtr = helpText;
   }
   else
      helpTextPtr = NULL;
   if (GetARBoolean("Change owner? (F): ", FALSE))
   {
      strcpy(owner, GetChar("   Owner (): ", ""));
      ownerPtr = owner;
   }
   else
      ownerPtr = NULL;
   if (GetARBoolean("Add to change diary? (F): ", FALSE))
   {
      strcpy(changeDiary, GetChar("   Change Diary (): ", ""));
      changeDiaryPtr = changeDiary;
   }
   else
      changeDiaryPtr = NULL;
   if (GetARBoolean("Change object properties? (F): ", FALSE))
   {
      GetARPropList(&objPropList);
      objPropListPtr = &objPropList;
   }
   else
      objPropListPtr = NULL;

   /* Call routine */
   BeginAPICall();
   result = ARSetFilter(GetControlStructPtr(), name, newNamePtr, orderPtr,
                        workflowConnectPtr, opSetPtr, enablePtr, queryPtr,
                        actionListPtr, elseListPtr, helpTextPtr, ownerPtr,
                        changeDiaryPtr, objPropListPtr, &status);
   EndAPICall(&status);


   /* Print results */
   DriverPrintResultHeader("ARSetFilter");
   PrintReturnCode(result);
   PrintARStatusList(&status);

   /* Cleanup */
   if (queryPtr != NULL)
      FreeARQualifierStruct(&query, FALSE);
   if (actionListPtr != NULL)
      FreeARFilterActionList(&actionList, FALSE);
   if (elseListPtr != NULL)
      FreeARFilterActionList(&elseList, FALSE);
   if (objPropListPtr != NULL)
      FreeARPropList(&objPropList, FALSE);
   if (workflowConnectPtr != NULL)
      FreeARWorkflowConnectStruct(&workflowConnect, FALSE);

   FreeARStatusList(&status, FALSE);
}


/*****************************************************************************/
/*                                                                           */
/*                               APIARCreateFilter                           */
/*                                                                           */
/*****************************************************************************/

void APIARCreateFilter()

{
   ARFilterActionList        actionList;
   char                      changeDiary[1024];
   ARFilterActionList        elseList;
   unsigned int              enable;
   char                      helpText[1024];
   ARNameType                name;
   unsigned int              numItems;
   ARPropList                objPropList;
   unsigned int              opSet;
   unsigned int              order;
   ARAccessNameType          owner;
   ARQualifierStruct         query;
   int                       result;
   ARStatusList              status={0, NULL};
   char                      tmpString[24];
   ARWorkflowConnectStruct   workflowConnect;

   /* Get Parameters */
   DriverPrintHeader("CREATE FILTER");
   strcpy(name, GetChar("Filter name: ", ""));
   order = GetInt("Filter order (0 - 1000) (500): ", 500);
   workflowConnect.type = GetInt("Workflow type (1): ", 1);
   if (workflowConnect.type == AR_WORKFLOW_CONN_SCHEMA_LIST)
   {
      workflowConnect.u.schemaList = (ARNameList *) malloc(sizeof(ARNameList));
      if (workflowConnect.u.schemaList == NULL)
      {
         DriverPrintError("**** malloc failure for ARCreateFilter call ****\n");
         return;
      }
      workflowConnect.u.schemaList->numItems = GetInt("Number of schemas (1): ", 1);
      workflowConnect.u.schemaList->nameList =
         (ARNameType *) malloc(workflowConnect.u.schemaList->numItems *
                               sizeof(ARNameType));
      if (workflowConnect.u.schemaList->nameList == NULL)
      {
         DriverPrintError("**** malloc failure for ARCreateFilter call ****\n");
         free(workflowConnect.u.schemaList);
         return;
      }
      for (numItems = 0; numItems < workflowConnect.u.schemaList->numItems;
           numItems++)
      {
         sprintf(tmpString, "Schema name %u:", numItems + 1);
         strcpy(workflowConnect.u.schemaList->nameList[numItems],
                GetChar(tmpString, ""));
      }
   }

   opSet = GetInt("Operations bit mask (0): ", 0);
   enable = GetInt("Disable/Enable (0-1) (1): ", 1);
   GetARQualifierStruct(&query);
   GetARFilterActionList(TRUE, &actionList);
   GetARFilterActionList(FALSE, &elseList);
   strcpy(helpText, GetChar("Help Text: ", ""));
   strcpy(owner, GetChar("Owner: ", GetControlStructPtr()->user));
   strcpy(changeDiary, GetChar("Change Diary: ", ""));
   GetARPropList(&objPropList);

   /* Call routine */
   BeginAPICall();
   result = ARCreateFilter(GetControlStructPtr(), name, order, &workflowConnect,
                           opSet, enable, &query, &actionList, &elseList,
                           helpText, owner, changeDiary, &objPropList, &status);
   EndAPICall(&status);

  /* Print results */
   DriverPrintResultHeader("ARCreateFilter");
   PrintReturnCode(result);
   PrintARStatusList(&status);

   /* Cleanup */
   FreeARQualifierStruct(&query, FALSE);
   FreeARFilterActionList(&actionList, FALSE);
   FreeARFilterActionList(&elseList, FALSE);
   FreeARPropList(&objPropList, FALSE);
   FreeARWorkflowConnectStruct(&workflowConnect, FALSE);
   FreeARStatusList(&status, FALSE);
}


/*****************************************************************************/
/*                                                                           */
/*                               APIARDeleteFilter                           */
/*                                                                           */
/*****************************************************************************/

void APIARDeleteFilter()

{
   ARNameType     name;
   int            result;
   ARStatusList   status={0, NULL};

   /* Get Parameters */
   DriverPrintHeader("DELETE FILTER");
   strcpy(name, GetChar("Filter name: ", ""));

   /* Call routine */
   BeginAPICall();
   result = ARDeleteFilter(GetControlStructPtr(), name, &status);
   EndAPICall(&status);

   /* Print results */
   DriverPrintResultHeader("ARDeleteFilter");
   PrintReturnCode(result);
   PrintARStatusList(&status);

   /* Cleanup */
   FreeARStatusList(&status, FALSE);
}


/*****************************************************************************/
/*                                                                           */
/*                               APIARGetListFilter                          */
/*                                                                           */
/*****************************************************************************/

void APIARGetListFilter()

{
   ARTimestamp    changedSince;
   ARNameList     nameList;
   int            result;
   ARNameType     schema;
   char          *schemaPtr;
   ARStatusList   status={0, NULL};

   /* Get Parameters */
   DriverPrintHeader("GETLIST FILTER");
   if (GetARBoolean("For a specific schema? (F): ", FALSE))
   {
      strcpy(schema, GetChar("Schema name (): ", ""));
      schemaPtr = schema;
   }
   else
      schemaPtr = NULL;
   changedSince = GetARTimestamp("Get all changed since (0): ",
                                 (ARTimestamp) 0);

   /* Call routine */
   BeginAPICall();
   result = ARGetListFilter(GetControlStructPtr(), schemaPtr, changedSince,
                            &nameList, &status);
   EndAPICall(&status);

   /* Print results */
   DriverPrintResultHeader("ARGetListFilter");
   PrintReturnCode(result);
   if (result < AR_RETURN_ERROR)
      PrintARNameList("Filter list: ", "   ", &nameList);
   PrintARStatusList(&status);

   /* Cleanup */
   if (result < AR_RETURN_ERROR)
      FreeARNameList(&nameList, FALSE);
   FreeARStatusList(&status, FALSE);
}


/*****************************************************************************/
/**********                                                         **********/
/**********                       ACTIVE LINK                       **********/
/**********                                                         **********/
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*                             APIARGetActiveLink                            */
/*                                                                           */
/*****************************************************************************/

void APIARGetActiveLink()

{
   ARActiveLinkActionList    actionList;
   char                     *changeDiary;
   ARInternalId              controlField;
   ARActiveLinkActionList    elseList;
   unsigned int              enable;
   unsigned int              executeMask;
   ARInternalId              focusField;
   ARInternalIdList          groupList;
   char                     *helpText;
   ARAccessNameType          lastChanged;
   ARNameType                name;
   ARPropList                objPropList;
   unsigned int              order;
   ARAccessNameType          owner;
   ARQualifierStruct         query;
   int                       result; 
   ARStatusList              status={0, NULL};
   ARTimestamp               timestamp;
   ARWorkflowConnectStruct   workflowConnect = {AR_WORKFLOW_CONN_SCHEMA_LIST, NULL};

   /* Get Parameters */
   DriverPrintHeader("GET ACTIVE LINK");
   strcpy(name, GetChar("Active link name: ", ""));

   /* Call routine */
   BeginAPICall();
   result = ARGetActiveLink(GetControlStructPtr(), name, &order,
                            &workflowConnect, &groupList, &executeMask,
                            &controlField, &focusField, &enable, &query,
                            &actionList, &elseList, &helpText, &timestamp,
                            owner, lastChanged, &changeDiary, &objPropList,
                            &status);
   EndAPICall(&status);

   /* Print results */
   DriverPrintResultHeader("ARGetActiveLink");
   PrintReturnCode(result);
   if (result < AR_RETURN_ERROR)
   {
      PrintInt("Execution order: ", order);
      PrintInt("Workflow connect type: ", workflowConnect.type);
      if (workflowConnect.type == AR_WORKFLOW_CONN_SCHEMA_LIST)
      {
         PrintInt("Number of schema list items is: ",
                  workflowConnect.u.schemaList->numItems);
         PrintARNameList("Schema list: ", " ", workflowConnect.u.schemaList); 
      }
      PrintARInternalIdList("Group list: ", "   ", &groupList);
      PrintInt("Execute bitmask: ", executeMask);
      PrintARInternalId("Control Field: ", controlField);
      PrintARInternalId("Focus Field: ", focusField);
      PrintInt("Enable flag : ", enable);
      PrintARQualifierStruct("", &query);
      PrintARActiveLinkActionList(&actionList, TRUE);
      PrintARActiveLinkActionList(&elseList, FALSE);
      PrintChar("Help text: ", helpText);
      PrintARTimestamp("Last update time: ", timestamp);
      PrintChar("Owner: ", owner);
      PrintChar("Last changed: ", lastChanged);
      APIPrintDiary("Change Diary: ", changeDiary);
      PrintARObjectPropList("", "", &objPropList);
   }
   PrintARStatusList(&status);

   /* Cleanup */
   if (result < AR_RETURN_ERROR)
   {
      FreeARInternalIdList(&groupList, FALSE);
      FreeARQualifierStruct(&query, FALSE);
      FreeARActiveLinkActionList(&actionList, FALSE);
      FreeARActiveLinkActionList(&elseList, FALSE);
      if (helpText != NULL)
         free(helpText);
      if (changeDiary != NULL)
         free(changeDiary);
      FreeARPropList(&objPropList, FALSE);
      FreeARWorkflowConnectStruct(&workflowConnect, FALSE);
   }
   FreeARStatusList(&status, FALSE);
}


/*****************************************************************************/
/*                                                                           */
/*                             APIARSetActiveLink                            */
/*                                                                           */
/*****************************************************************************/

void APIARSetActiveLink()

{
   ARActiveLinkActionList    actionList;
   ARActiveLinkActionList   *actionListPtr;
   char                      changeDiary[1024];
   char                     *changeDiaryPtr;
   ARInternalId              controlField;
   ARInternalId             *controlFieldPtr;
   ARActiveLinkActionList    elseList;
   ARActiveLinkActionList   *elseListPtr;
   unsigned int              enable;
   unsigned int             *enablePtr;
   unsigned int              executeMask;
   unsigned int             *executeMaskPtr;
   ARInternalId              focusField;
   ARInternalId             *focusFieldPtr;
   ARInternalIdList          groupList;
   ARInternalIdList         *groupListPtr;
   char                      helpText[1024];
   char                     *helpTextPtr;
   ARNameType                name;
   ARNameType                newName;
   char                     *newNamePtr;
   unsigned int              numItems;
   ARPropList                objPropList;
   ARPropList               *objPropListPtr;
   unsigned int              order;
   unsigned int             *orderPtr;
   ARAccessNameType          owner;
   char                     *ownerPtr;
   ARQualifierStruct         query;
   ARQualifierStruct        *queryPtr;
   int                       result;
   ARStatusList              status={0, NULL};
   char                      tmpString[24];
   ARWorkflowConnectStruct   workflowConnect;
   ARWorkflowConnectStruct  *workflowConnectPtr;

   /* Get Parameters */
   DriverPrintHeader("SET ACTIVE LINK");
   strcpy(name, GetChar("Active link name: ", ""));
   if (GetARBoolean("Change active link name? (F): ", FALSE))
   {
      strcpy(newName, GetChar("New name (): ", ""));
      newNamePtr = newName;
   }
   else
      newNamePtr = NULL;
   if (GetARBoolean("Change execution order? (F): ", FALSE))
   {
      order = GetInt("Execution order (1 - 1000) (1): ", 1);
      orderPtr = &order;
   }
   else
      orderPtr = NULL;
   workflowConnect.type = GetInt("Workflow type (1): ", 1);
   if (GetARBoolean("Change schemaList? (F): ", FALSE))
   {
      if (workflowConnect.type == AR_WORKFLOW_CONN_SCHEMA_LIST)
      {
         workflowConnect.u.schemaList = (ARNameList *) malloc(sizeof(ARNameList));
         if (workflowConnect.u.schemaList == NULL)
         {
            DriverPrintError("**** malloc failure for ARSetActiveLink call ****\n");
            return;
         }
         workflowConnect.u.schemaList->numItems = GetInt("Number of schemas? (1): ", 1);
         workflowConnect.u.schemaList->nameList =
            (ARNameType *) malloc(workflowConnect.u.schemaList->numItems *
                                  sizeof(ARNameType));
         if (workflowConnect.u.schemaList->nameList == NULL)
         {
            DriverPrintError("**** malloc failure for ARSetActiveLink call ****\n");
            free(workflowConnect.u.schemaList);
            return;
         }
         for (numItems = 0; numItems < workflowConnect.u.schemaList->numItems;
              numItems++)
         {
            sprintf(tmpString, "Schema name %u:", numItems + 1);
            strcpy(workflowConnect.u.schemaList->nameList[numItems],
                   GetChar(tmpString, ""));
         }
      }
      workflowConnectPtr = &workflowConnect;
   }
   else
      workflowConnectPtr = NULL; 
   if (GetARBoolean("Change group list? (F): ", FALSE))
   {
      DriverPrintPrompt("Ids of groups allowed to perform active link:\n");
      GetARInternalIdList(&groupList);
      groupListPtr = &groupList;
   }
   else
      groupListPtr = NULL;
   if (GetARBoolean("Change execute mask? (F): ", FALSE))
   {
      executeMask = GetInt("Execute bit mask (1): ", 1);
      executeMaskPtr = &executeMask;
   }
   else
      executeMaskPtr = NULL;
   if (GetARBoolean("Change control field? (F): ", FALSE))
   {
      controlField = GetLong("Control Field to tie active link to (0): ", 0);
      controlFieldPtr = &controlField;
   }
   else
      controlFieldPtr = NULL;
   if (GetARBoolean("Change focus field? (F): ", FALSE))
   {
      focusField = GetLong("Focus Field to tie active link to (0): ", 0);
      focusFieldPtr = &focusField;
   }
   else
      focusFieldPtr = NULL;
   if (GetARBoolean("Change enable/disable? (F): ", FALSE))
   {
      enable = GetInt("Disable/enable (0-1) (1): ", 1);
      enablePtr = &enable;
   }
   else
      enablePtr = NULL;
   if (GetARBoolean("Change query? (F): ", FALSE))
   {
      GetARQualifierStruct(&query);
      queryPtr = &query;
   }
   else
      queryPtr = NULL;
   if (GetARBoolean("Change action(s)? (F): ", FALSE))
   {
      GetARActiveLinkActionList(TRUE, &actionList);
      actionListPtr = &actionList;
   }
   else
      actionListPtr = NULL;
   if (GetARBoolean("Change else(s)? (F): ", FALSE))
   {
      GetARActiveLinkActionList(FALSE, &elseList);
      elseListPtr = &elseList;
   }
   else
      elseListPtr = NULL;
   if (GetARBoolean("Change help text? (F): ", FALSE))
   {
      strcpy(helpText, GetChar("   Help Text (): ", ""));
      helpTextPtr = helpText;
   }
   else
      helpTextPtr = NULL;
   if (GetARBoolean("Change owner? (F): ", FALSE))
   {
      strcpy(owner, GetChar("Owner (): ", ""));
      ownerPtr = owner;
   }
   else
      ownerPtr = NULL;
   if (GetARBoolean("Add to change diary? (F): ", FALSE))
   {
      strcpy(changeDiary, GetChar("Change Diary (): ", ""));
      changeDiaryPtr = changeDiary;
   }
   else
      changeDiaryPtr = NULL;
   if (GetARBoolean("Change object properties? (F): ", FALSE))
   {
      GetARPropList(&objPropList);
      objPropListPtr = &objPropList;
   }
   else
      objPropListPtr = NULL;

   /* Call routine */
   BeginAPICall();
   result = ARSetActiveLink(GetControlStructPtr(), name, newNamePtr, orderPtr,
                            workflowConnectPtr, groupListPtr, executeMaskPtr,
                            controlFieldPtr, focusFieldPtr, enablePtr, queryPtr,
                            actionListPtr, elseListPtr, helpTextPtr, ownerPtr,
                            changeDiaryPtr, objPropListPtr, &status);
   EndAPICall(&status);

   /* Print results */
   DriverPrintResultHeader("ARSetActiveLink");
   PrintReturnCode(result);
   PrintARStatusList(&status);

   /* Cleanup */
   if (groupListPtr != NULL)
      FreeARInternalIdList(&groupList, FALSE);
   if (queryPtr != NULL)
      FreeARQualifierStruct(&query, FALSE);
   if (actionListPtr != NULL)
      FreeARActiveLinkActionList(&actionList, FALSE);
   if (elseListPtr != NULL)
      FreeARActiveLinkActionList(&elseList, FALSE);
   if (objPropListPtr != NULL)
      FreeARPropList(&objPropList, FALSE);
   if (workflowConnectPtr != NULL)
      FreeARWorkflowConnectStruct(&workflowConnect, FALSE);
   FreeARStatusList(&status, FALSE);
}


/*****************************************************************************/
/*                                                                           */
/*                           APIARCreateActiveLink                           */
/*                                                                           */
/*****************************************************************************/

void APIARCreateActiveLink()

{
   ARActiveLinkActionList    actionList;
   char                      changeDiary[1024];
   ARInternalId              controlField;
   ARActiveLinkActionList    elseList;
   unsigned int              enable;
   unsigned int              executeMask;
   ARInternalId              focusField;
   ARInternalIdList          groupList;
   char                      helpText[1024];
   ARNameType                name;
   unsigned int              numItems;
   ARPropList                objPropList;
   unsigned int              order;
   ARAccessNameType          owner;
   ARQualifierStruct         query;
   int                       result;
   ARStatusList              status={0, NULL};
   char                      tmpString[24];
   ARWorkflowConnectStruct   workflowConnect;

   /* Get Parameters */
   DriverPrintHeader("CREATE ACTIVE LINK");
   strcpy(name, GetChar("Active link name: ", ""));
   order = GetInt("Execution order (1 - 1000) (1): ", 1);
   workflowConnect.type = GetInt("Workflow type (1): ", 1);
   if (workflowConnect.type == AR_WORKFLOW_CONN_SCHEMA_LIST)
   {
      workflowConnect.u.schemaList = (ARNameList *) malloc(sizeof(ARNameList));
      if (workflowConnect.u.schemaList == NULL)
      {
         DriverPrintError("**** malloc failure for ARCreateActiveLink call ****\n");
         return;
      }
      workflowConnect.u.schemaList->numItems = GetInt("Number of schemas (1): ", 1);
      workflowConnect.u.schemaList->nameList = 
         (ARNameType *) malloc(workflowConnect.u.schemaList->numItems *
                               sizeof(ARNameType));
      if (workflowConnect.u.schemaList->nameList == NULL)
      {
         DriverPrintError("**** malloc failure for ARCreateActiveLink call ****\n");
         free(workflowConnect.u.schemaList);
         return;
      }
      for (numItems = 0; numItems < workflowConnect.u.schemaList->numItems;
           numItems++)
      {
         sprintf(tmpString, "Schema name %u:", numItems + 1);
         strcpy(workflowConnect.u.schemaList->nameList[numItems],
                GetChar(tmpString, ""));
      }
   }
   DriverPrintPrompt("Ids of groups allowed to perform active link:\n");
   GetARInternalIdList(&groupList);
   executeMask = GetInt("Execute bit mask (1): ", 1);
   controlField = GetLong("Control Field to tie active link to (0): ", 0);
   focusField = GetLong("Focus Field to tie active link to (0): ", 0);
   enable = GetInt("Disable/Enable (0-1) (1): ", 1);
   GetARQualifierStruct(&query);
   GetARActiveLinkActionList(TRUE, &actionList);
   GetARActiveLinkActionList(FALSE, &elseList);
   strcpy(helpText, GetChar("Help Text: ", ""));
   strcpy(owner, GetChar("Owner: ", GetControlStructPtr()->user));
   strcpy(changeDiary, GetChar("Change Diary: ", ""));
   GetARPropList(&objPropList);

   /* Call routine */
   BeginAPICall();
   result = ARCreateActiveLink(GetControlStructPtr(), name, order,
                               &workflowConnect, &groupList, executeMask,
                               &controlField, &focusField, enable, &query,
                               &actionList, &elseList, helpText, owner,
                               changeDiary, &objPropList, &status);
   EndAPICall(&status);

   /* Print results */
   DriverPrintResultHeader("ARCreateActiveLink");
   PrintReturnCode(result);
   PrintARStatusList(&status);

   /* Cleanup */
   FreeARInternalIdList(&groupList, FALSE);
   FreeARQualifierStruct(&query, FALSE);
   FreeARActiveLinkActionList(&actionList, FALSE);
   FreeARActiveLinkActionList(&elseList, FALSE);
   FreeARPropList(&objPropList, FALSE);
   FreeARWorkflowConnectStruct(&workflowConnect, FALSE);
   FreeARStatusList(&status, FALSE);
}


/*****************************************************************************/
/*                                                                           */
/*                           APIARDeleteActiveLink                           */
/*                                                                           */
/*****************************************************************************/

void APIARDeleteActiveLink()

{
   ARNameType     name;
   int            result;
   ARStatusList   status={0, NULL};

   /* Get Parameters */
   DriverPrintHeader("DELETE ACTIVE LINK");
   strcpy(name, GetChar("Active link name: ", ""));

   /* Call routine */
   BeginAPICall();
   result = ARDeleteActiveLink(GetControlStructPtr(), name, &status);
   EndAPICall(&status);

   /* Print results */
   DriverPrintResultHeader("ARDeleteActiveLink");
   PrintReturnCode(result);
   PrintARStatusList(&status);

   /* Cleanup */
   FreeARStatusList(&status, FALSE);
}


/*****************************************************************************/
/*                                                                           */
/*                           APIARGetListActiveLink                          */
/*                                                                           */
/*****************************************************************************/

void APIARGetListActiveLink()

{
   ARTimestamp    changedSince;
   ARNameList     nameList;
   int            result;
   ARNameType     schema;
   char          *schemaPtr;
   ARStatusList   status={0, NULL};

   /* Get Parameters */
   DriverPrintHeader("GETLIST ACTIVE LINK");
   if (GetARBoolean("For a specific schema? (F): ", FALSE))
   {
      strcpy(schema, GetChar("Schema name (): ", ""));
      schemaPtr = schema;
   }
   else
      schemaPtr = NULL;
   changedSince = GetARTimestamp("Get all changed since (0): ",
                                 (ARTimestamp) 0);

   /* Call routine */
   BeginAPICall();
   result = ARGetListActiveLink(GetControlStructPtr(), schemaPtr, changedSince,
                                &nameList, &status);
   EndAPICall(&status);

   /* Print results */
   DriverPrintResultHeader("ARGetListActiveLink");
   PrintReturnCode(result);
   if (result < AR_RETURN_ERROR)
      PrintARNameList("Active link list: ", "   ", &nameList);
   PrintARStatusList(&status);

   /* Cleanup */
   if (result < AR_RETURN_ERROR)
      FreeARNameList(&nameList, FALSE);
   FreeARStatusList(&status, FALSE);
}


/*****************************************************************************/
/*                                                                           */
/*                       APIARGetMultipleActiveLinks                         */
/*                                                                           */
/*****************************************************************************/

void APIARGetMultipleActiveLinks()

{
   ARTimestamp                   changedSince;
   ARNameList                    nameListIn;
   ARNameList                    nameListOut;
   ARBooleanList                 existList;
   ARUnsignedIntList             orderList;
   ARWorkflowConnectList         workflowConnectList;
   ARInternalIdListList          groupListList;
   ARUnsignedIntList             executeMaskList;
   ARInternalIdList              controlFieldList;
   ARInternalIdList              focusFieldList;
   ARUnsignedIntList             enableList;
   ARQualifierList               queryList;
   ARActiveLinkActionListList    actionListList;
   ARActiveLinkActionListList    elseListList;
   ARTextStringList              helpTextList;
   ARTimestampList               timestampList;
   ARAccessNameList              ownersList;
   ARAccessNameList              lastChangedList;
   ARTextStringList              changeDiaryList;
   ARPropListList                objPropListList;
   ARStatusList                  status={0, NULL};
   unsigned int                  i;
   int                           result = AR_RETURN_OK;

                                       /* print header */
   DriverPrintHeader("GET MULTIPLE ACTIVE LINKS");
                                       /* get parameter */
   changedSince = GetARTimestamp("Get all changed since (0): ", (ARTimestamp) 0);
   DriverPrintPrompt("Get active link by name:\n");
   GetARNameList(&nameListIn, "   ");

   /* Call routine */
   BeginAPICall();
   result = ARGetMultipleActiveLinks(GetControlStructPtr(), changedSince, &nameListIn, &existList,
                                     &nameListOut, &orderList, &workflowConnectList, &groupListList,  
                                     &executeMaskList, &controlFieldList, &focusFieldList, &enableList, 
                                     &queryList, &actionListList, &elseListList, &helpTextList, 
                                     &timestampList, &ownersList, &lastChangedList, 
                                     &changeDiaryList, &objPropListList, &status);
   EndAPICall(&status);

   /* Print results */
   DriverPrintResultHeader("ARGetMultipleActiveLinks");
   PrintReturnCode(result);
   if (result < AR_RETURN_ERROR)
   {
      if (nameListIn.numItems)
      {
         DriverPrintResultHeader("Exist list:");
         for (i = 0; i < nameListIn.numItems; i++)
         {
            DriverPrintResult("   Active Link : %s\n", nameListIn.nameList[i]);
            PrintARBoolean("    exists: ", existList.booleanList[i]);
         }
      } 
      for (i = 0; i < existList.numItems; i++)
      { 
         if (existList.booleanList[i])
         {
            PrintChar("\n Active Link name: ", nameListOut.nameList[i]);
            PrintInt("Execution order: ", orderList.intList[i]);
            PrintInt("Workflow connect type: ", workflowConnectList.workflowConnectList[i].type);
            if (workflowConnectList.workflowConnectList[i].type == AR_WORKFLOW_CONN_SCHEMA_LIST &&
                workflowConnectList.workflowConnectList[i].u.schemaList)
            {
               PrintInt("Number of schema list items is: ",
                        workflowConnectList.workflowConnectList[i].u.schemaList->numItems);
               PrintARNameList("Schema list: ", " ", workflowConnectList.workflowConnectList[i].u.schemaList); 
            }
            if (groupListList.internalIdListList)
               PrintARInternalIdList("Group list: ", "   ", &groupListList.internalIdListList[i]);
            PrintInt("Execute bitmask: ", executeMaskList.intList[i]);
            PrintARInternalId("Control Field: ", controlFieldList.internalIdList[i]);
            PrintARInternalId("Focus Field: ", focusFieldList.internalIdList[i]);
            PrintInt("Enable flag : ", enableList.intList[i]);
            PrintARQualifierStruct("", &queryList.qualifierList[i]);
            PrintARActiveLinkActionList(&actionListList.actionListList[i], TRUE);
            PrintARActiveLinkActionList(&elseListList.actionListList[i], FALSE);
            PrintChar("Help text: ", helpTextList.stringList[i]);
            PrintARTimestamp("Last update time: ", timestampList.timestampList[i]);
            PrintChar("Owner: ", ownersList.nameList[i]);
            PrintChar("Last changed: ", lastChangedList.nameList[i]);
            APIPrintDiary("Change Diary: ", changeDiaryList.stringList[i]);
            PrintARObjectPropList("", "", &objPropListList.propsList[i]);
         }
      }
   }
   PrintARStatusList(&status);
 
   /* Cleanup */
   if (result < AR_RETURN_ERROR)
   {
      FreeARNameList(&nameListIn, FALSE);
      FreeARBooleanList(&existList, FALSE);
      FreeARNameList(&nameListOut, FALSE);
      FreeARUnsignedIntList(&orderList, FALSE);
      FreeARWorkflowConnectList(&workflowConnectList, FALSE);
      FreeARInternalIdListList(&groupListList, FALSE);
      FreeARUnsignedIntList(&executeMaskList, FALSE);
      FreeARInternalIdList(&controlFieldList, FALSE);
      FreeARInternalIdList(&focusFieldList, FALSE);
      FreeARUnsignedIntList(&enableList, FALSE);
      FreeARQualifierList(&queryList, FALSE);
      FreeARActiveLinkActionListList(&actionListList, FALSE);
      FreeARActiveLinkActionListList(&elseListList, FALSE);
      FreeARTextStringList(&helpTextList, FALSE);
      FreeARTimestampList(&timestampList, FALSE);
      FreeARAccessNameList(&ownersList, FALSE);
      FreeARAccessNameList(&lastChangedList, FALSE);
      FreeARTextStringList(&changeDiaryList, FALSE);
      FreeARPropListList(&objPropListList, FALSE);
   }
   FreeARStatusList(&status, FALSE);
}
 

/*****************************************************************************/
/**********                                                         **********/
/**********                       SUPPORT FILE                      **********/
/**********                                                         **********/
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*                            APIARGetSupportFile                            */
/*                                                                           */
/*****************************************************************************/

void APIARGetSupportFile()

{
   ARInternalId   fileId;
   char           filename[256];
   FILE          *filePtr;
   unsigned int   fileType;
   ARInternalId   id2;
   ARNameType     name;
   int            result;
   ARStatusList   status={0, NULL};
   ARTimestamp    timestamp;

   /* Get Parameters */
   DriverPrintHeader("GET SUPPORT FILE");
   fileType = GetInt("File type -- external report (1) (1): ", 1);
   strcpy(name, GetChar("Name of associated object: ", ""));
   id2 = GetLong("Supporting ID for object (0): ", 0);
   fileId = GetLong("ID for the file (1): ", 1);

   filePtr = NULL;
   while (filePtr == NULL)
   {
      strcpy(filename, GetChar("Filename to hold returned file: ", ""));
#ifdef _WIN32
      filePtr = fopen(filename, "wb");
#else /* _WIN32 */
      filePtr = fopen(filename, "w");
#endif /* _WIN32 */
      if (filePtr == NULL)
         DriverPrintError("   ***  File open failure\n");
   }

   /* Call routine */
   BeginAPICall();
   result = ARGetSupportFile(GetControlStructPtr(), fileType, name, id2, fileId,
                             filePtr, &timestamp, &status);
   EndAPICall(&status);

   /* Print results */
   DriverPrintResultHeader("ARGetSupportFile");
   PrintReturnCode(result);
   if (result < AR_RETURN_ERROR)
      PrintARTimestamp("Last update time: ", timestamp);
   PrintARStatusList(&status);

   /* Cleanup */
   fclose(filePtr);
   FreeARStatusList(&status, FALSE);
}


/*****************************************************************************/
/*                                                                           */
/*                            APIARSetSupportFile                            */
/*                                                                           */
/*****************************************************************************/

void APIARSetSupportFile()

{
   ARInternalId   fileId;
   char           filename[256];
   FILE          *filePtr;
   unsigned int   fileType;
   ARInternalId   id2;
   ARNameType     name;
   int            result;
   ARStatusList   status={0, NULL};

   /* Get Parameters */
   DriverPrintHeader("SET SUPPORT FILE");
   fileType = GetInt("File type -- external report (1) (1): ", 1);
   strcpy(name, GetChar("Name of associated object: ", ""));
   id2 = GetLong("Supporting ID for object (0): ", 0);
   fileId = GetLong("ID for the file (1): ", 1);

   filePtr = NULL;
   while (filePtr == NULL)
   {
      strcpy(filename, GetChar("Filename to load to server: ", ""));
#ifdef _WIN32
      filePtr = fopen(filename, "rb");
#else /* _WIN32 */
      filePtr = fopen(filename, "r");
#endif /* _WIN32 */
      if (filePtr == NULL)
         DriverPrintError("   ***  File open failure\n");
   }

   /* Call routine */
   BeginAPICall();
   result = ARSetSupportFile(GetControlStructPtr(), fileType, name, id2, fileId,
                             filePtr, &status);
   EndAPICall(&status);

   /* Print results */
   DriverPrintResultHeader("ARSetSupportFile");
   PrintReturnCode(result);
   PrintARStatusList(&status);

   /* Cleanup */
   fclose(filePtr);
   FreeARStatusList(&status, FALSE);
}


/*****************************************************************************/
/*                                                                           */
/*                           APIARCreateSupportFile                          */
/*                                                                           */
/*****************************************************************************/

void APIARCreateSupportFile()

{
   ARInternalId   fileId;
   char           filename[256];
   FILE          *filePtr;
   unsigned int   fileType;
   ARInternalId   id2;
   ARNameType     name;
   int            result;
   ARStatusList   status={0, NULL};

   /* Get Parameters */
   DriverPrintHeader("CREATE SUPPORT FILE");
   fileType = GetInt("File type -- external report (1) (1): ", 1);
   strcpy(name, GetChar("Name of associated object: ", ""));
   id2 = GetLong("Supporting ID for object (0): ", 0);
   fileId = GetLong("ID for the file (1): ", 1);

   filePtr = NULL;
   while (filePtr == NULL)
   {
      strcpy(filename, GetChar("Filename to hold returned file: ", ""));
#ifdef _WIN32
      filePtr = fopen(filename, "rb");
#else /* _WIN32 */
      filePtr = fopen(filename, "r");
#endif /* _WIN32 */
      if (filePtr == NULL)
         DriverPrintError("   ***  File open failure\n");
   }

   /* Call routine */
   BeginAPICall();
   result = ARCreateSupportFile(GetControlStructPtr(), fileType, name, id2,
                                fileId, filePtr, &status);
   EndAPICall(&status);

   /* Print results */
   DriverPrintResultHeader("ARCreateSupportFile");
   PrintReturnCode(result);
   PrintARStatusList(&status);

   /* Cleanup */
   fclose(filePtr);
   FreeARStatusList(&status, FALSE);
}


/*****************************************************************************/
/*                                                                           */
/*                           APIARDeleteSupportFile                          */
/*                                                                           */
/*****************************************************************************/

void APIARDeleteSupportFile()

{
   ARInternalId   fileId;
   unsigned int   fileType;
   ARInternalId   id2;
   ARNameType     name;
   int            result;
   ARStatusList   status={0, NULL};

   /* Get Parameters */
   DriverPrintHeader("DELETE SUPPORT FILE");
   fileType = GetInt("File type -- external report (1) (1): ", 1);
   strcpy(name, GetChar("Name of associated object: ", ""));
   id2 = GetLong("Supporting ID for object (0): ", 0);
   fileId = GetLong("ID for the file (1): ", 1);

   /* Call routine */
   BeginAPICall();
   result = ARDeleteSupportFile(GetControlStructPtr(), fileType, name, id2,
                                fileId, &status);
   EndAPICall(&status);

   /* Print results */
   DriverPrintResultHeader("ARDeleteSupportFile");
   PrintReturnCode(result);
   PrintARStatusList(&status);

   /* Cleanup */
   FreeARStatusList(&status, FALSE);
}


/*****************************************************************************/
/*                                                                           */
/*                          APIARGetListSupportFile                          */
/*                                                                           */
/*****************************************************************************/

void APIARGetListSupportFile()

{
   ARTimestamp        changedSince;
   ARInternalIdList   fileIdList;
   unsigned int       fileType;
   ARInternalId       id2;
   ARNameType         name;
   int                result;
   ARStatusList       status={0, NULL};

   /* Get Parameters */
   DriverPrintHeader("GETLIST SUPPORT FILE");
   fileType = GetInt("File type -- external report (1) (1): ", 1);
   strcpy(name, GetChar("Name of associated object: ", ""));
   id2 = GetLong("Supporting ID for object (0): ", 0);
   changedSince = GetARTimestamp("Get all changed since (0): ",
                                 (ARTimestamp) 0);

   /* Call routine */
   BeginAPICall();
   result = ARGetListSupportFile(GetControlStructPtr(), fileType, name, id2,
                                 changedSince, &fileIdList, &status);
   EndAPICall(&status);

   /* Print results */
   DriverPrintResultHeader("ARGetListSupportFile");
   PrintReturnCode(result);
   if (result < AR_RETURN_ERROR)
      PrintARInternalIdList("File ID list: ", "   ", &fileIdList);
   PrintARStatusList(&status);

   /* Cleanup */
   if (result < AR_RETURN_ERROR)
      FreeARInternalIdList(&fileIdList, FALSE);
   FreeARStatusList(&status, FALSE);
}


/*****************************************************************************/
/**********                                                         **********/
/**********                         GROUP/USER                      **********/
/**********                                                         **********/
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*                               APIARGetListGroup                           */
/*                                                                           */
/*****************************************************************************/

void APIARGetListGroup()

{
   ARGroupInfoList   groupList;
   int               result;
   ARStatusList      status={0, NULL};
   ARAccessNameType  user;
   char             *userPtr;
   ARAccessNameType  password;
   char             *passwordPtr;

   /* Get Parameters */
   DriverPrintHeader("GETLIST GROUP");
   if (GetARBoolean("For a specific user? (F): ", FALSE))
   {
      strcpy(user, GetChar("Name of the user: ",
                   GetControlStructPtr()->user));
      userPtr = user;

      strcpy(password, GetChar("Password of the user: ",
                       GetControlStructPtr()->password));
      passwordPtr = password;
   }
   else
   {
      userPtr = NULL;
      passwordPtr = NULL;
   }

   /* Call routine */
   BeginAPICall();
   result = ARGetListGroup(GetControlStructPtr(), userPtr, passwordPtr,
                           &groupList, &status);
   EndAPICall(&status);

   /* Print results */
   DriverPrintResultHeader("ARGetListGroup");
   PrintReturnCode(result);
   if (result < AR_RETURN_ERROR)
      PrintARGroupInfoList(&groupList);
   PrintARStatusList(&status);

   /* Cleanup */
   if (result < AR_RETURN_ERROR)
      FreeARGroupInfoList(&groupList, FALSE);
   FreeARStatusList(&status, FALSE);
}

/*****************************************************************************/
/*                                                                           */
/*                               APIARGetListUser                            */
/*                                                                           */
/*****************************************************************************/

void APIARGetListUser()

{
   int              result;
   ARStatusList     status={0, NULL};
   ARUserInfoList   userList;
   unsigned int     userListType;
   ARTimestamp      changedSince;

   /* Get Parameters */
   DriverPrintHeader("GETLIST USER");
   userListType = GetInt(
                 "User List Type (myself, registered or current (0-2)(1): ", 1);
   changedSince = GetARTimestamp("Get registered users changed since (0 means all): ",
                                 (ARTimestamp) 0);

   /* Call routine */
   BeginAPICall();
   result = ARGetListUser(GetControlStructPtr(), userListType, changedSince,
                          &userList, &status);
   EndAPICall(&status);

   /* Print results */
   DriverPrintResultHeader("ARGetListUser");
   PrintReturnCode(result);
   if (result < AR_RETURN_ERROR)
      PrintARUserInfoList(&userList);
   PrintARStatusList(&status);

   /* Cleanup */
   if (result < AR_RETURN_ERROR)
      FreeARUserInfoList(&userList, FALSE);
   FreeARStatusList(&status, FALSE);
}


/*****************************************************************************/
/**********                                                         **********/
/**********                      MISCELLANEOUS                      **********/
/**********                                                         **********/
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*                              APIARVerifyUser                              */
/*                                                                           */
/*****************************************************************************/

void APIARVerifyUser()

{
   ARBoolean      adminFlag;
   ARBoolean      customFlag;
   int            result;
   ARStatusList   status={0, NULL};
   ARBoolean      subadminFlag;

   /* Get Parameters */
   DriverPrintHeader("VERIFY USER");

   /* Call routine */
   BeginAPICall();
   result = ARVerifyUser(GetControlStructPtr(), &adminFlag, &subadminFlag,
                         &customFlag, &status);
   EndAPICall(&status);

   /* Print results */
   DriverPrintResultHeader("ARVerifyUser");
   PrintReturnCode(result);
   if (result < AR_RETURN_ERROR)
   {
      PrintARBoolean("Admin flag: ", adminFlag);
      PrintARBoolean("SubAdmin flag: ", subadminFlag);
      PrintARBoolean("Customize flag: ", customFlag);
   }
   PrintARStatusList(&status);

   /* Cleanup */
   FreeARStatusList(&status, FALSE);
}


/*****************************************************************************/
/*                                                                           */
/*                                APIARExport                                */
/*                                                                           */
/*****************************************************************************/

void APIARExport()

{
   char              *buff;
   ARNameType         displayTag;
   unsigned int       vuiType;
   char               filename[256];
   FILE              *filePtr;
   int                result;
   ARStatusList       status={0, NULL};
   ARStructItemList   structItems;
   unsigned int       i;

   status.numItems = 0;
   status.statusList = NULL;;

   /* Get Parameters */
   DriverPrintHeader("EXPORT");
   GetARStructItemList(&structItems);

   /* If exporting any VIEWs, prompt for the display tag and vui-type. */
   strcpy(displayTag, "");
   vuiType = 0;
   for (i=0; i < structItems.numItems; i++)
   {
      if (structItems.structItemList[i].type == AR_STRUCT_ITEM_SCHEMA_VIEW ||
          structItems.structItemList[i].type == AR_STRUCT_ITEM_SCHEMA_VIEW_2)
      {
         strcpy(displayTag, GetChar("Display Tag: ", ""));
         vuiType = GetInt("VUI-Type (0): ", 0);
         break;
      }
   }

   filePtr = NULL;
   while (filePtr == NULL)
   {
      strcpy(filename, GetChar("Filename for exported data: ", ""));
      if(filename[0] != '\0')
         filePtr = fopen(filename, "w");
      if (filePtr == NULL)
         DriverPrintError("   ***  File open failure\n");
   }

   /* Call routine */
   BeginAPICall();
   result = ARExport(GetControlStructPtr(), &structItems, displayTag,
                     vuiType, &buff, &status);
   EndAPICall(&status);

   /* Print results */
   DriverPrintResultHeader("ARExport");
   PrintReturnCode(result);
   if ((result < AR_RETURN_ERROR) && (buff != NULL))
      fwrite(buff, strlen(buff), 1, filePtr);
   fclose(filePtr);
   PrintARStatusList(&status);

   /* Cleanup */
   if (result < AR_RETURN_ERROR)
      if (buff != NULL)
         free(buff);
   else
      unlink(filename);
   FreeARStructItemList(&structItems, FALSE);
   FreeARStatusList(&status, FALSE);
}


/*****************************************************************************/
/*                                                                           */
/*                                APIARImport                                */
/*                                                                           */
/*****************************************************************************/

void APIARImport()

{
   int                amtRead;
   int                amtRemaining;
   long               fileLen;
   char               filename[256];
   FILE              *filePtr;
   char              *importBuf;
   unsigned int       importOption;
   int                result;
   ARStatusList       status={0, NULL};
   ARStructItemList   structItems;

   /* Get Parameters */
   DriverPrintHeader("IMPORT");
   GetARStructItemList(&structItems);
   filePtr = NULL;
   while (filePtr == NULL)
   {
      strcpy(filename, GetChar("Filename containing import data: ", ""));
      if(filename[0] != '\0')
         filePtr = fopen(filename, "r");
      if (filePtr == NULL)
         DriverPrintError("   ***  File open failure\n");
   }
   fseek(filePtr, 0L, 2);
   fileLen = ftell(filePtr);
   importBuf = (char *) malloc((unsigned) (fileLen + 1));
   if (importBuf == NULL)
   {
      fclose(filePtr);
      DriverPrintError("**** malloc failure for import operation ****\n");
      return;
   }
   rewind(filePtr);

   /* Note that the length of the file (fileLen) may actually be */
   /* more than what can be read.  Some standard io libraries    */
   /* will compress carriage returns (\r) and new lines (\n) on  */
   /* fread(), but this may not be reflected in ftell().         */
   for (amtRead = 0, amtRemaining = fileLen; amtRead < fileLen; )
   {
      result = fread(&importBuf[amtRead], 1, amtRemaining, filePtr);
      if (result <= 0)
         break;
      amtRemaining -= result;
      amtRead += result;
   }
   importBuf[amtRead] = '\0';
   fclose(filePtr);

   importOption = GetInt("The import option : (0) ", 0);

   /* Call routine */
   BeginAPICall();
   result = ARImport(GetControlStructPtr(), &structItems, importBuf,
                     importOption, &status);
   EndAPICall(&status);

   /* Print results */
   DriverPrintResultHeader("ARImport");
   PrintReturnCode(result);
   PrintARStatusList(&status);

   /* Cleanup */
   free(importBuf);
   FreeARStructItemList(&structItems, FALSE);
   FreeARStatusList(&status, FALSE);
}


/*****************************************************************************/
/*                                                                           */
/*                              APIARGetListServer                           */
/*                                                                           */
/*****************************************************************************/

void APIARGetListServer()

{
   ARServerNameList   nameList;
   int                result;
   ARStatusList       status={0, NULL};

   /* Get Parameters */
   DriverPrintHeader("GETLIST SERVER");

   /* Call routine */
   BeginAPICall();
   result = ARGetListServer(GetControlStructPtr(), &nameList, &status);
   EndAPICall(&status);

   /* Print results */
   DriverPrintResultHeader("ARGetListServer");
   PrintReturnCode(result);
   if (result < AR_RETURN_ERROR)
      PrintARServerNameList("Server list: ", "   ", &nameList);
   PrintARStatusList(&status);

   /* Cleanup */
   if (result < AR_RETURN_ERROR)
      FreeARServerNameList(&nameList, FALSE);
   FreeARStatusList(&status, FALSE);
}


/*****************************************************************************/
/*                                                                           */
/*                              APIARInitialization                          */
/*                                                                           */
/*****************************************************************************/

void APIARInitialization()

{
   int            result;
   ARStatusList   status={0, NULL};

   /* Get Parameters */
   DriverPrintHeader("INITIALIZATION");

   /* Call routine */
   BeginAPICall();
   result = ARInitialization(GetControlStructPtr(), &status);
   EndAPICall(&status);

   /* Print results */
   DriverPrintResultHeader("ARInitialization");
   PrintReturnCode(result);
   PrintARStatusList(&status);

   /* Cleanup */
   FreeARStatusList(&status, FALSE);
}


/*****************************************************************************/
/*                                                                           */
/*                                APIARTermination                           */
/*                                                                           */
/*****************************************************************************/

void APIARTermination()

{
   int            result;
   ARStatusList   status={0, NULL};

   /* Get Parameters */
   DriverPrintHeader("TERMINATION");

   /* Call routine */
   BeginAPICall();
   result = ARTermination(GetControlStructPtr(), &status);
   EndAPICall(&status);

   /* Print results */
   DriverPrintResultHeader("ARTermination");
   PrintReturnCode(result);
   PrintARStatusList(&status);

   /* Cleanup */
   FreeARStatusList(&status, FALSE);
}


/*****************************************************************************/
/*                                                                           */
/*                              APIARGetServerInfo                           */
/*                                                                           */
/*****************************************************************************/

void APIARGetServerInfo()

{
   ARServerInfoRequestList   requestList;
   int                       result;
   ARServerInfoList          serverInfoList;
   ARStatusList              status={0, NULL};

   /* Get Parameters */
   DriverPrintHeader("GET SERVER INFO");
   GetARServerInfoRequestList(AR_MAX_SERVER_INFO_USED, &requestList);

   /* Call routine */
   BeginAPICall();
   result = ARGetServerInfo(GetControlStructPtr(), &requestList,
                            &serverInfoList, &status);
   EndAPICall(&status);

   /* Print results */
   DriverPrintResultHeader("ARGetServerInfo");
   PrintReturnCode(result);
   if (result < AR_RETURN_ERROR)
      PrintARServerInfoList(&serverInfoList);
   PrintARStatusList(&status);

   /* Cleanup */
   if (result < AR_RETURN_ERROR)
      FreeARServerInfoList(&serverInfoList, FALSE);
   FreeARServerInfoRequestList(&requestList, FALSE);
   FreeARStatusList(&status, FALSE);
}


/*****************************************************************************/
/*                                                                           */
/*                             APIARSetServerInfo                            */
/*                                                                           */
/*****************************************************************************/

void APIARSetServerInfo()

{
   int                result;
   ARServerInfoList   serverInfoList;
   ARStatusList       status={0, NULL};

   /* Get Parameters */
   DriverPrintHeader("SET SERVER INFO");
   GetARServerInfoList(&serverInfoList);

   /* Call routine */
   BeginAPICall();
   result = ARSetServerInfo(GetControlStructPtr(), &serverInfoList, &status);
   EndAPICall(&status);

   /* Print results */
   DriverPrintResultHeader("ARSetServerInfo");
   PrintReturnCode(result);
   PrintARStatusList(&status);

   /* Cleanup */
   if (result < AR_RETURN_ERROR)
      FreeARServerInfoList(&serverInfoList, FALSE);
   FreeARStatusList(&status, FALSE);
}


/*****************************************************************************/
/*                                                                           */
/*                              APIARSetServerPort                           */
/*                                                                           */
/*****************************************************************************/

void APIARSetServerPort()

{
   unsigned int   portNumber;
   unsigned int   privateSocket;
   int            result;
   ARStatusList   status={0, NULL};

   /* Get Parameters */
   DriverPrintHeader("SET SERVER PORT");
   portNumber = GetInt("The port number of server (0): ", 0);
   privateSocket = GetInt("Using private socket? (0): ", 0);

   /* Call routine */
   BeginAPICall();
   result = ARSetServerPort(GetControlStructPtr(),
                            GetControlStructPtr()->server, portNumber,
                            privateSocket, &status);
   EndAPICall(&status);

   /* Print results */
   DriverPrintResultHeader("ARSetServerPort");
   PrintReturnCode(result);
   PrintARStatusList(&status);

   /* Cleanup */
   FreeARStatusList(&status, FALSE);
}


/*****************************************************************************/
/*                                                                           */
/*                            APIARGetServerStatistics                       */
/*                                                                           */
/*****************************************************************************/

void APIARGetServerStatistics()

{
   ARServerInfoRequestList   requestList;
   int                       result; 
   ARServerInfoList          serverInfoList;
   ARStatusList              status={0, NULL};

   /* Get Parameters */
   DriverPrintHeader("GET SERVER STATISTICS");
   GetARServerInfoRequestList(AR_MAX_SERVER_STAT_USED, &requestList);

   /* Call routine */
   BeginAPICall();
   result = ARGetServerStatistics(GetControlStructPtr(), &requestList,
                                  &serverInfoList, &status);
   EndAPICall(&status);

   /* Print results */
   DriverPrintResultHeader("ARGetServerStatistics");
   PrintReturnCode(result);
   if (result < AR_RETURN_ERROR)
      PrintARServerInfoList2(&serverInfoList);
   PrintARStatusList(&status);

   /* Cleanup */
   if (result < AR_RETURN_ERROR)
      FreeARServerInfoList(&serverInfoList, FALSE);
   FreeARServerInfoRequestList(&requestList, FALSE);
   FreeARStatusList(&status, FALSE);
}


/*****************************************************************************/
/*                                                                           */
/*                          APIARGetTextForErrorMessage                      */
/*                                                                           */
/*****************************************************************************/

void APIARGetTextForErrorMessage()

{
   char  *messageBuf;
   int    messageId;

   /* Get Parameters */
   DriverPrintHeader("GET TEXT FOR ERROR MESSAGE");
   messageId = GetInt("The message identifier for the message (0): ", 0);

   /* Call routine */
   BeginAPICall();
   messageBuf = ARGetTextForErrorMessage(messageId);
   EndAPICall(NULL);

   /* Print results */
   DriverPrintResultHeader("ARGetTextForErrorMessage");
   PrintChar("Message Text: ", messageBuf);
   
   /* Cleanup */
   if (messageBuf)
      free(messageBuf);
}


/*****************************************************************************/
/**********                                                         **********/
/**********                  FULL TEXT INFORMATION                  **********/
/**********                                                         **********/
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*                           APIARGetFullTextInfo                            */
/*                                                                           */
/*****************************************************************************/

void APIARGetFullTextInfo()

{
   ARFullTextInfoList          fullTextInfoList;
   ARFullTextInfoRequestList   requestList;
   int                         result; 
   ARStatusList                status={0, NULL};

   /* Get Parameters */
   DriverPrintHeader("GET FULL TEXT INFO");
   GetARFullTextInfoRequestList(&requestList);

   /* Call routine */
   BeginAPICall();
   result = ARGetFullTextInfo(GetControlStructPtr(), &requestList,
                              &fullTextInfoList, &status);
   EndAPICall(&status);

   /* Print results */
   DriverPrintResultHeader("ARGetFullTextInfo");
   PrintReturnCode(result);
   if (result < AR_RETURN_ERROR)
      PrintARFullTextInfoList(&fullTextInfoList);
   PrintARStatusList(&status);

   /* Cleanup */
   if (result < AR_RETURN_ERROR)
      FreeARFullTextInfoList(&fullTextInfoList, FALSE);
   FreeARFullTextInfoRequestList(&requestList, FALSE);
   FreeARStatusList(&status, FALSE);
}

/*****************************************************************************/
/*                                                                           */
/*                           APIARSetFullTextInfo                            */
/*                                                                           */
/*****************************************************************************/

void APIARSetFullTextInfo()

{
   ARFullTextInfoList   fullTextInfoList;
   int                  result;
   ARStatusList         status={0, NULL};

   /* Get Parameters */
   DriverPrintHeader("SET FULL TEXT INFO");
   GetARFullTextInfoList(&fullTextInfoList);

   /* Call routine */
   BeginAPICall();
   result = ARSetFullTextInfo(GetControlStructPtr(), &fullTextInfoList,
                              &status);
   EndAPICall(&status);

   /* Print results */
   DriverPrintResultHeader("ARSetFullTextInfo");
   PrintReturnCode(result);
   PrintARStatusList(&status);

   /* Cleanup */
   if (result < AR_RETURN_ERROR)
      FreeARFullTextInfoList(&fullTextInfoList, FALSE);
   FreeARStatusList(&status, FALSE);
}


/*****************************************************************************/
/*                                                                           */
/*                              APIARGetListSQL                              */
/*                                                                           */
/*****************************************************************************/

void APIARGetListSQL()

{
   unsigned int      maxRetrieve;
   unsigned int      numMatches;
   unsigned int     *numMatchesPtr;
   int               result;
   char             *sqlCommand;
   ARStatusList      status={0, NULL};
   ARValueListList   valueListList;

   /* Get Parameters */
   DriverPrintHeader("GETLIST SQL");
   GetMallocChar("SQL command: ", "", &sqlCommand);
   maxRetrieve = GetInt("Maximum number of entries to retrieve (500): ", 500);
   if (GetARBoolean("Get number of matches? (F): ", FALSE))
      numMatchesPtr = &numMatches;
   else
      numMatchesPtr = NULL;

   /* Call routine */
   BeginAPICall();
   result = ARGetListSQL(GetControlStructPtr(), sqlCommand, maxRetrieve,
                         &valueListList, numMatchesPtr, &status);
   EndAPICall(&status);

   /* Print results */
   DriverPrintResultHeader("ARGetListSQL");
   PrintReturnCode(result);
   if (result < AR_RETURN_ERROR)
   {
      PrintARValueListList(&valueListList);
      if (numMatchesPtr != NULL)
         PrintInt("Number of matches: ", numMatches);
   }
   PrintARStatusList(&status);

   /* Cleanup */
   if (sqlCommand != NULL)
      free(sqlCommand);
   if (result < AR_RETURN_ERROR)
      FreeARValueListList(&valueListList, FALSE);
   FreeARStatusList(&status, FALSE);
}


/*****************************************************************************/
/*                                                                           */
/*                              APIARExecuteProcess                          */
/*                                                                           */
/*****************************************************************************/

void APIARExecuteProcess()

{
   char           command[1024];
   int            result;
   int            returnStatus;
   int           *returnStatusPtr;
   char          *returnString;
   char         **returnStringPtr;
   ARStatusList   status={0, NULL};

   /* Get Parameters */
   DriverPrintHeader("EXECUTE PROCESS");
   strcpy(command, GetChar("Command: ", ""));
   if (GetARBoolean("Wait for process to complete? (F): ", FALSE))
   {
      returnStatusPtr = &returnStatus;
      returnStringPtr = &returnString;
   }
   else
   {
      returnStatusPtr = NULL;
      returnStringPtr = NULL;
   }

   /* Call routine */
   BeginAPICall();
   result = ARExecuteProcess(GetControlStructPtr(), command, returnStatusPtr,
                             returnStringPtr, &status);
   EndAPICall(&status);

   /* Print results */
   DriverPrintResultHeader("ARExecuteProcess");
   PrintReturnCode(result);
   if (result < AR_RETURN_ERROR)
   {
      if (returnStatusPtr != NULL)
         PrintInt("Return Status: ", *returnStatusPtr);
      if (returnStringPtr != NULL)
         PrintChar("Process output: ", *returnStringPtr);
   }
   PrintARStatusList(&status);

   /* Cleanup */
   FreeARStatusList(&status, FALSE);
}


/*****************************************************************************/
/**********                                                         **********/
/**********                       ESCALATION                        **********/
/**********                                                         **********/
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*                            APIARGetEscalation                             */
/*                                                                           */
/*****************************************************************************/

void APIARGetEscalation()

{
   ARFilterActionList        actionList;
   char                     *changeDiary;
   ARFilterActionList        elseList;
   unsigned int              enable;
   AREscalationTmStruct      escalationTm;
   char                     *helpText;
   ARAccessNameType          lastChanged;
   ARNameType                name;
   ARPropList                objPropList;
   ARAccessNameType          owner;
   ARQualifierStruct         query;
   int                       result;
   ARStatusList              status={0, NULL};
   ARTimestamp               timestamp;
   ARWorkflowConnectStruct   workflowConnect;

   /* Get Parameters */
   DriverPrintHeader("GET ESCALATION");
   strcpy(name, GetChar("Escalation name: ", ""));

   /* Call routine */
   BeginAPICall();
   result = ARGetEscalation(GetControlStructPtr(), name, &escalationTm,
                            &workflowConnect, &enable, &query, &actionList,
                            &elseList, &helpText, &timestamp, owner,
                            lastChanged, &changeDiary, &objPropList, &status);
   EndAPICall(&status);

   /* Print results */
   DriverPrintResultHeader("ARGetEscalation");
   PrintReturnCode(result);
   if (result < AR_RETURN_ERROR)
   {
      if (escalationTm.escalationTmType == AR_ESCALATION_TYPE_INTERVAL)
         PrintInt("Escalation interval: ", escalationTm.u.interval);
      else
      {
         PrintInt("Escalation monthday: ", escalationTm.u.date.monthday);
         PrintInt("Escalation weekday: ", escalationTm.u.date.weekday);
         PrintInt("Escalation hourmask: ", escalationTm.u.date.hourmask);
         PrintInt("Escalation minute: ", escalationTm.u.date.minute);
      }
      PrintInt("Workflow connect type: ",workflowConnect.type);
      if (workflowConnect.type == AR_WORKFLOW_CONN_SCHEMA_LIST)
      {
         PrintInt("Number of schema list items is: ",
                  workflowConnect.u.schemaList->numItems);
         PrintARNameList("Schema list: ", " ", workflowConnect.u.schemaList);
      }
      PrintInt("Enable flag : ", enable);
      PrintARQualifierStruct("", &query);
      PrintARFilterActionList(&actionList, TRUE);
      PrintARFilterActionList(&elseList, FALSE);
      PrintChar("Help text: ", helpText);
      PrintARTimestamp("Last update time: ", timestamp);
      PrintChar("Owner: ", owner);
      PrintChar("Last changed: ", lastChanged);
      APIPrintDiary("Change Diary: ", changeDiary);
      PrintARObjectPropList("", "", &objPropList);
   }
   PrintARStatusList(&status);

   /* Cleanup */
   if (result < AR_RETURN_ERROR)
   {
      FreeARQualifierStruct(&query, FALSE);
      FreeARFilterActionList(&actionList, FALSE);
      FreeARFilterActionList(&elseList, FALSE);
      if (helpText != NULL)
         free(helpText);
      if (changeDiary != NULL)
         free(changeDiary);
      FreeARPropList(&objPropList, FALSE);
      FreeARWorkflowConnectStruct(&workflowConnect, FALSE);

   }
   FreeARStatusList(&status, FALSE);
}


/*****************************************************************************/
/*                                                                           */
/*                            APIARSetEscalation                             */
/*                                                                           */
/*****************************************************************************/

void APIARSetEscalation()

{
   ARFilterActionList        actionList;
   ARFilterActionList       *actionListPtr;
   char                      changeDiary[1024];
   char                     *changeDiaryPtr;
   int                       i, day, date;
   ARFilterActionList        elseList;
   ARFilterActionList       *elseListPtr;
   unsigned int              enable;
   unsigned int             *enablePtr;
   AREscalationTmStruct      escalationTm;
   AREscalationTmStruct     *escalationTmPtr;
   char                      helpText[1024];
   char                     *helpTextPtr;
   ARNameType                name;
   ARNameType                newName;
   char                     *newNamePtr;
   unsigned int              numItems;
   ARPropList                objPropList;
   ARPropList               *objPropListPtr;
   ARAccessNameType          owner;
   char                     *ownerPtr;
   ARQualifierStruct         query;
   ARQualifierStruct        *queryPtr;
   int                       result;
   ARStatusList              status={0, NULL};
   char                      tmpString[24];
   ARWorkflowConnectStruct   workflowConnect;
   ARWorkflowConnectStruct  *workflowConnectPtr;

   /* Get Parameters */
   DriverPrintHeader("SET ESCALATION");
   strcpy(name, GetChar("Escalation name: ", ""));
   if (GetARBoolean("Change escalation name? (F): ", FALSE))
   {
      strcpy(newName, GetChar("New name (): ", ""));
      newNamePtr = newName;
   }
   else
      newNamePtr = NULL;
   if (GetARBoolean("Change escalation time ? (F): ", FALSE))
   {
      escalationTm.escalationTmType =
                                GetInt("Escalation time format (1/2) (1): ", 1);
      if (escalationTm.escalationTmType == AR_ESCALATION_TYPE_INTERVAL)
         escalationTm.u.interval = GetInt("Escalation interval (> 0) (300): ",
                                          300);
      else
      {
         escalationTm.u.date.monthday = 0;
         escalationTm.u.date.weekday = 0;
         escalationTm.u.date.hourmask = 0;
         escalationTm.u.date.minute = 0;
         day = GetInt("how many days to set in day/month format? (> 0) (1): ",1);
         i = 0;
         while (i < day)
         {
            date = GetInt("date of the day, 0 to 30 for month/day (0): ", 0);
            if (date < 0 || date > 30)
               continue;
            else
            {
               escalationTm.u.date.monthday |= AR_SETDAY(date);
               i++;
            }
         }
         day = GetInt("how many days to set in day/week format? (> 0) (1): ", 1);
         i = 0;
         while (i < day)
         {
            date = GetInt("date of the day, 0 to 6 for sunday to saturday (0): ", 0);
            if (date < 0 || date > 6)
               continue;
            else
            {
               escalationTm.u.date.weekday |= AR_SETDAY(date);
               i++;
            }
         }
         day = GetInt("how many hour marks to set in a day? (> 0) (1): ", 1);
         i = 0;
         while (i < day)
         {
            date = GetInt("hour of the day, 0 to 23 for 24 hours of a day (0): ", 0);
            if (date < 0 || date > 23)
               continue;
            else
            {
               escalationTm.u.date.hourmask |= AR_SETHOUR(date);
               i++;
            }
         }
         day = GetInt("set minute mark of the hour ? (0-59) (0): ", 0);
         while (day < 0 || day > 59)
            day = GetInt("set minute mark of the hour ? (0-59) (0): ", 0);
         escalationTm.u.date.minute = day;
      }
      escalationTmPtr = &escalationTm;
   }
   else
      escalationTmPtr = NULL;
   workflowConnect.type = GetInt("Workflow type (1): ", 1);
   if (GetARBoolean("Change schema or schemaList? (F): ", FALSE))
   {
      if (workflowConnect.type == AR_WORKFLOW_CONN_SCHEMA_LIST)
      {
         workflowConnect.u.schemaList = (ARNameList *) malloc(sizeof(ARNameList));
         if (workflowConnect.u.schemaList == NULL)
         {
            DriverPrintError("**** malloc failure for ARSetEscalation call ****\n");
            return;
         }
         workflowConnect.u.schemaList->numItems = GetInt("Number of schemas? (1): ", 1);
         workflowConnect.u.schemaList->nameList =
            (ARNameType *) malloc(workflowConnect.u.schemaList->numItems *
                                  sizeof(ARNameType));
         if (workflowConnect.u.schemaList->nameList == NULL)
         {
            DriverPrintError("**** malloc failure for ARSetEscalation call ****\n");
            free(workflowConnect.u.schemaList);
            return;
         }
         for (numItems = 0; numItems < workflowConnect.u.schemaList->numItems;
              numItems++)
         {
            sprintf(tmpString, "Schema name %u:", numItems + 1);
            strcpy(workflowConnect.u.schemaList->nameList[numItems],
                   GetChar(tmpString, ""));
         }
      }
      workflowConnectPtr = &workflowConnect;
   }
   else
      workflowConnectPtr = NULL;

   if (GetARBoolean("Change enable flag? (F): ", FALSE))
   {
      enable = GetInt("Disable/Enable (0-1) (1): ", 1);
      enablePtr = &enable;
   }
   else
      enablePtr = NULL;
   if (GetARBoolean("Change query? (F): ", FALSE))
   {
      GetARQualifierStruct(&query);
      queryPtr = &query;
   }
   else
      queryPtr = NULL;
   if (GetARBoolean("Change action(s)? (F): ", FALSE))
   {
      GetARFilterActionList(TRUE, &actionList);
      actionListPtr = &actionList;
   }
   else
      actionListPtr = NULL;
   if (GetARBoolean("Change else(s)? (F): ", FALSE))
   {
      GetARFilterActionList(FALSE, &elseList);
      elseListPtr = &elseList;
   }
   else
      elseListPtr = NULL;
   if (GetARBoolean("Change help text? (F): ", FALSE))
   {
      strcpy(helpText, GetChar("   Help Text (): ", ""));
      helpTextPtr = helpText;
   }
   else
      helpTextPtr = NULL;
   if (GetARBoolean("Change owner? (F): ", FALSE))
   {
      strcpy(owner, GetChar("   Owner (): ", ""));
      ownerPtr = owner;
   }
   else
      ownerPtr = NULL;
   if (GetARBoolean("Add to change diary? (F): ", FALSE))
   {
      strcpy(changeDiary, GetChar("   Change Diary (): ", ""));
      changeDiaryPtr = changeDiary;
   }
   else
      changeDiaryPtr = NULL;
   if (GetARBoolean("Change object properties? (F): ", FALSE))
   {
      GetARPropList(&objPropList);
      objPropListPtr = &objPropList;
   }
   else
      objPropListPtr = NULL;

   /* Call routine */
   BeginAPICall();
   result = ARSetEscalation(GetControlStructPtr(), name, newNamePtr,
                            escalationTmPtr, workflowConnectPtr, enablePtr,
                            queryPtr, actionListPtr, elseListPtr, helpTextPtr,
                            ownerPtr, changeDiaryPtr, objPropListPtr, &status);
   EndAPICall(&status);

   /* Print results */
   DriverPrintResultHeader("ARSetEscalation");
   PrintReturnCode(result);
   PrintARStatusList(&status);

   /* Cleanup */
   if (queryPtr != NULL)
      FreeARQualifierStruct(&query, FALSE);
   if (actionListPtr != NULL)
      FreeARFilterActionList(&actionList, FALSE);
   if (elseListPtr != NULL)
      FreeARFilterActionList(&elseList, FALSE);
   if (objPropListPtr != NULL)
      FreeARPropList(&objPropList, FALSE);
   if (workflowConnectPtr != NULL)
      FreeARWorkflowConnectStruct(&workflowConnect, FALSE);

   FreeARStatusList(&status, FALSE);
}


/*****************************************************************************/
/*                                                                           */
/*                           APIARCreateEscalation                           */
/*                                                                           */
/*****************************************************************************/

void APIARCreateEscalation()

{
   ARFilterActionList        actionList;
   char                      changeDiary[1024];
   int                       i, day, date;
   ARFilterActionList        elseList;
   unsigned int              enable;
   AREscalationTmStruct      escalationTm;
   char                      helpText[1024];
   ARNameType                name;
   unsigned int              numItems;
   ARPropList                objPropList;
   ARAccessNameType          owner;
   ARQualifierStruct         query;
   int                       result;
   ARStatusList              status={0, NULL};
   char                      tmpString[24];
   ARWorkflowConnectStruct   workflowConnect;

   /* Get Parameters */
   DriverPrintHeader("CREATE ESCALATION");
   strcpy(name, GetChar("Escalation name: ", ""));
   escalationTm.escalationTmType = GetInt("Escalation time format (1/2) (1): ",
                                          1);
   if (escalationTm.escalationTmType == AR_ESCALATION_TYPE_INTERVAL)
      escalationTm.u.interval = GetInt("Escalation interval (> 0) (300): ",
                                       300);
   else
   {
      escalationTm.u.date.monthday = 0;
      escalationTm.u.date.weekday = 0;
      escalationTm.u.date.hourmask = 0;
      escalationTm.u.date.minute = 0;
      day = GetInt("how many days to set in day/month format? (> 0) (1): ", 1);
      i = 0;
      while (i < day)
      {
         date = GetInt("date of the day, 0 to 30 for month/day (0): ", 0);
         if (date < 0 || date > 30)
            continue;
         else
         {
            escalationTm.u.date.monthday |= AR_SETDAY(date);
            i++;
         }
      }
      day = GetInt("how many days to set in day/week format? (> 0) (1): ", 1);
      i = 0;
      while (i < day)
      {
         date = GetInt("date of the day, 0 to 6 for sunday to saturday (0): ", 0);
         if (date < 0 || date > 6)
            continue;
         else
         {
            escalationTm.u.date.weekday |= AR_SETDAY(date);
            i++;
         }
      }
      day = GetInt("how many hour marks to set in a day? (> 0) (1): ", 1);
      i = 0;
      while (i < day)
      {
         date = GetInt("hour of the day, 0 to 23 for 24 hours of a day (0): ", 0);
         if (date < 0 || date > 23)
            continue;
         else
         {
            escalationTm.u.date.hourmask |= AR_SETHOUR(date);
            i++;
         }
      }
      day = GetInt("set minute mark of the hour ? (0-59) (0): ", 0);
      while (day < 0 || day > 59)
         day = GetInt("set minute mark of the hour ? (0-59) (0): ", 0);
      escalationTm.u.date.minute = day;
   }
   workflowConnect.type = GetInt("Workflow type (1): ", 1);
   if (workflowConnect.type == AR_WORKFLOW_CONN_SCHEMA_LIST)
   {
      workflowConnect.u.schemaList = (ARNameList *) malloc(sizeof(ARNameList));
      if (workflowConnect.u.schemaList == NULL)
      {
         DriverPrintError("**** malloc failure for ARCreateEscalation call ****\n");
         return;
      }
      workflowConnect.u.schemaList->numItems = GetInt("Number of schemas (1): ",
                                                      1);
      workflowConnect.u.schemaList->nameList =
         (ARNameType *) malloc(workflowConnect.u.schemaList->numItems *
                               sizeof(ARNameType));
      if (workflowConnect.u.schemaList->nameList == NULL)
      {
         DriverPrintError("**** malloc failure for ARCreateEscalation call ****\n");
         free(workflowConnect.u.schemaList);
         return;
      }
      for (numItems = 0; numItems < workflowConnect.u.schemaList->numItems;
           numItems++)
      {
         sprintf(tmpString, "Schema name %u:", numItems + 1);
         strcpy(workflowConnect.u.schemaList->nameList[numItems],
                GetChar(tmpString, ""));
      }
   }

   enable = GetInt("Disable/Enable (0-1) (1): ", 1);
   GetARQualifierStruct(&query);
   GetARFilterActionList(TRUE, &actionList);
   GetARFilterActionList(FALSE, &elseList);
   strcpy(helpText, GetChar("Help Text: ", ""));
   strcpy(owner, GetChar("Owner: ", GetControlStructPtr()->user));
   strcpy(changeDiary, GetChar("Change Diary: ", ""));
   GetARPropList(&objPropList);

   /* Call routine */
   BeginAPICall();
   result = ARCreateEscalation(GetControlStructPtr(), name, &escalationTm,
                               &workflowConnect, enable, &query, &actionList,
                               &elseList, helpText, owner, changeDiary,
                               &objPropList, &status);
   EndAPICall(&status);

   /* Print results */
   DriverPrintResultHeader("ARCreateEscalation");
   PrintReturnCode(result);
   PrintARStatusList(&status);

   /* Cleanup */
   FreeARQualifierStruct(&query, FALSE);
   FreeARFilterActionList(&actionList, FALSE);
   FreeARFilterActionList(&elseList, FALSE);
   FreeARPropList(&objPropList, FALSE);
   FreeARWorkflowConnectStruct(&workflowConnect, FALSE);
   FreeARStatusList(&status, FALSE);
}


/*****************************************************************************/
/*                                                                           */
/*                           APIARDeleteEscalation                           */
/*                                                                           */
/*****************************************************************************/

void APIARDeleteEscalation()

{
   ARNameType     name;
   int            result;
   ARStatusList   status={0, NULL};

   /* Get Parameters */
   DriverPrintHeader("DELETE ESCALATION");
   strcpy(name, GetChar("Escalation name: ", ""));

   /* Call routine */
   BeginAPICall();
   result = ARDeleteEscalation(GetControlStructPtr(), name, &status);
   EndAPICall(&status);

   /* Print results */
   DriverPrintResultHeader("ARDeleteEscalation");
   PrintReturnCode(result);
   PrintARStatusList(&status);

   /* Cleanup */
   FreeARStatusList(&status, FALSE);
}


/*****************************************************************************/
/*                                                                           */
/*                           APIARGetListEscalation                          */
/*                                                                           */
/*****************************************************************************/

void APIARGetListEscalation()

{
   ARTimestamp    changedSince;
   ARNameList     nameList;
   int            result;
   ARNameType     schema;
   char          *schemaPtr;
   ARStatusList   status={0, NULL};

   /* Get Parameters */
   DriverPrintHeader("GETLIST ESCALATION");
   if (GetARBoolean("For a specific schema? (F): ", FALSE))
   {
      strcpy(schema, GetChar("Schema name (): ", ""));
      schemaPtr = schema;
   }
   else
      schemaPtr = NULL;
   changedSince = GetARTimestamp("Get all changed since (0 means all): ",
                                 (ARTimestamp) 0);

   /* Call routine */
   BeginAPICall();
   result = ARGetListEscalation(GetControlStructPtr(), schemaPtr, changedSince,
                                &nameList, &status);
   EndAPICall(&status);

   /* Print results */
   DriverPrintResultHeader("ARGetListEscalation");
   PrintReturnCode(result);
   if (result < AR_RETURN_ERROR)
      PrintARNameList("Escalation list: ", "   ", &nameList);
   PrintARStatusList(&status);

   /* Cleanup */
   if (result < AR_RETURN_ERROR)
      FreeARNameList(&nameList, FALSE);
   FreeARStatusList(&status, FALSE);
}


/*****************************************************************************/
/*                                                                           */
/*                           APIARCloseNetworkConnections                    */
/*                                                                           */
/*****************************************************************************/

void APIARCloseNetworkConnections()

{
   int            result;
   ARStatusList   status={0, NULL};

   /* Get Parameters */
   DriverPrintHeader("CLOSE NETWORK CONNECTIONS");

   /* Call routine */
   BeginAPICall();
   result = ARCloseNetworkConnections(GetControlStructPtr(), &status);
   EndAPICall(&status);

   /* Print results */
   DriverPrintResultHeader("ARCloseNetworkConnections");
   PrintReturnCode(result);
   PrintARStatusList(&status);

   /* Cleanup */
   FreeARStatusList(&status, FALSE);
}


/*****************************************************************************/
/*                                                                           */
/*                              APIARSetLogging                              */
/*                                                                           */
/*****************************************************************************/
 
void APIARSetLogging()
 
{
   char                 filename[256];
   FILE                *filePtr = NULL;
   unsigned long        logTypeMask;
   int                  result;
   ARStatusList         status={0, NULL};
   ThreadControlBlock  *threadControlBlockPtr;
   unsigned long        whereToWriteMask;

   threadControlBlockPtr = GetThreadControlBlockPtr();

   /* Get Parameters */
   DriverPrintHeader("SET LOGGING");
   logTypeMask = GetLong("NONE(0), SQL(1), Filter(2), API(16) (0): ", 0);
   
   /* if logging option is turned off ignore other values */
   if (logTypeMask == AR_DEBUG_SERVER_NONE)
   {
      whereToWriteMask = 0;
   }
   else
   {
      whereToWriteMask = 
         GetLong("Return logging records to file(1), status list(2)? (1): ",
                 AR_WRITE_TO_FILE);

      /* the file should be left open until the caller closes it with a call */
      /* to this function with logTypeMask equal to AR_DEBUG_SERVER_NONE     */
      if (whereToWriteMask & AR_WRITE_TO_FILE)
      {
         while (filePtr == NULL)
         {
            strcpy(filename,
                          GetChar("Name of file to hold returned records: ", ""));
#ifdef _WIN32
            filePtr = fopen(filename, "wb");
#else /* _WIN32 */
            filePtr = fopen(filename, "w");
#endif /* _WIN32 */
            if (filePtr == NULL)
               DriverPrintError("   ***  File open failure\n");
         }
      }
   }

   /* Call routine */
   BeginAPICall();
   result = ARSetLogging(GetControlStructPtr(), logTypeMask, whereToWriteMask,
                         filePtr, &status);
   EndAPICall(&status);
 
   /* Print results */
   DriverPrintResultHeader("ARSetLogging");
   PrintReturnCode(result);
   if (result < AR_RETURN_ERROR && logTypeMask != AR_DEBUG_SERVER_NONE)
      threadControlBlockPtr->loggingFile = filePtr;
   PrintARStatusList(&status);
 
   /* Cleanup */
   if (result < AR_RETURN_ERROR)
   {
      if (logTypeMask == AR_DEBUG_SERVER_NONE &&
          threadControlBlockPtr->loggingFile != NULL)
      {
         fclose(threadControlBlockPtr->loggingFile);
         threadControlBlockPtr->loggingFile = NULL;
      } 
   }
   else
   { 
      if (filePtr != NULL)
         fclose(filePtr);
   }
   FreeARStatusList(&status, FALSE);
}


/*****************************************************************************/
/*                                                                           */
/*                                APIARSignal                                */
/*                                                                           */
/*****************************************************************************/

void APIARSignal()
{
   int            result;
   ARSignalList   signalList;
   ARStatusList   status={0, NULL};
   
   /* Get Parameters */
   DriverPrintHeader("SIGNAL");
   GetARSignalList(&signalList);

   /* Call routine */
   BeginAPICall();
   result = ARSignal(GetControlStructPtr(), &signalList, &status);
   EndAPICall(&status);

   /* Print results */
   DriverPrintResultHeader("ARSignal");
   PrintReturnCode(result);
   PrintARStatusList(&status);

   /* Cleanup */
   FreeARSignalList(&signalList, FALSE);
   FreeARStatusList(&status, FALSE);
}


/*****************************************************************************/
/*                                                                           */
/*                           APIARValidateFormCache                          */
/*                                                                           */
/*****************************************************************************/

void APIARValidateFormCache()

{
   ARTimestamp    groupsLastChanged;
   ARNameList     guideChanged;
   ARNameList     menusChanged;
   ARTimestamp    mostRecentActiveLink;
   ARTimestamp    mostRecentGuide;
   ARTimestamp    mostRecentMenu;
   int            numActiveLinksChanged;
   int            numActiveLinksTotal;
   int            result;
   ARNameType     schema;
   ARTimestamp    schemaLastModified;
   ARStatusList   status={0, NULL};
   ARTimestamp    userLastChanged;

   /* Get Parameters */
   DriverPrintHeader("VALIDATE FORM CACHE");
   strcpy(schema, GetChar("Schema name (): ", ""));
   mostRecentActiveLink = GetARTimestamp("Get active links changed since (0): ",
                                         (ARTimestamp) 0);
   mostRecentMenu = GetARTimestamp("Get menus changed since (0): ",
                                   (ARTimestamp) 0);

   mostRecentGuide = GetARTimestamp("Get guides changed since (0): ",
                                   (ARTimestamp) 0);

   /* Call routine */
   BeginAPICall();
   result = ARValidateFormCache(GetControlStructPtr(), schema,
                                mostRecentActiveLink, mostRecentMenu, mostRecentGuide,
                                &schemaLastModified, &numActiveLinksTotal,
                                &numActiveLinksChanged, &menusChanged,
                                &groupsLastChanged, &userLastChanged, &guideChanged, &status);
   EndAPICall(&status);

   /* Print results */
   DriverPrintResultHeader("ARValidateFormCache");
   PrintReturnCode(result);
   if (result < AR_RETURN_ERROR)
   {
      PrintARTimestamp("Schema last modified: ", schemaLastModified);
      PrintInt("Number of active links for schema: ", numActiveLinksTotal);
      PrintInt("Number of active links changed: ", numActiveLinksChanged);
      PrintARNameList("Character menus changed: ", "   ", &menusChanged);
      PrintARTimestamp("Groups last changed: ", groupsLastChanged);
      PrintARTimestamp("User last changed: ", userLastChanged);
      PrintARNameList("Guides changed: ", "   ", &guideChanged);
   }
   PrintARStatusList(&status);

   /* Cleanup */
   if (result < AR_RETURN_ERROR)
      FreeARNameList(&menusChanged, FALSE);
   FreeARStatusList(&status, FALSE);
}


/*****************************************************************************/
/*                                                                           */
/*                        APIARGetListSchemaWithAlias                        */
/*                                                                           */
/*****************************************************************************/

void APIARGetListSchemaWithAlias()

{
   ARNameList         aliasList;
   ARTimestamp        changedSince;
   ARInternalIdList   fieldIdList;
   ARNameType         joinName;
   int                result;
   ARNameList         schemaList;
   int                schemaType;
   ARStatusList       status={0, NULL};
   ARNameType         vuiLabel;

   /* Get Parameters */
   DriverPrintHeader("GETLIST SCHEMA WITH ALIAS");
   changedSince = GetARTimestamp("Get all changed since (0): ",
                                 (ARTimestamp) 0);
   schemaType = GetInt(
       "Type of schema (all,regular,join,view,uplink,downlink,dialog) (0-6) (0): ", 0);
   if (GetARBoolean("Include hidden schemas (T): ", TRUE))
      schemaType += AR_HIDDEN_INCREMENT;
   if (((schemaType % AR_HIDDEN_INCREMENT) == AR_LIST_SCHEMA_UPLINK) ||
       ((schemaType % AR_HIDDEN_INCREMENT) == AR_LIST_SCHEMA_DOWNLINK))
      strcpy(joinName, GetChar("Schema name: ", ""));
   else
      joinName[0] = '\0';

   strcpy(vuiLabel, GetChar("Vui desired (): ", ""));

   DriverPrintPrompt("Ids of fields, which must be on the schema:\n");
   GetARInternalIdList(&fieldIdList);

   /* Call routine */
   BeginAPICall();
   result = ARGetListSchemaWithAlias(GetControlStructPtr(), changedSince,
                                     schemaType, joinName, &fieldIdList,
                                     vuiLabel, &schemaList, &aliasList,
                                     &status);
   EndAPICall(&status);

   /* Print results */
   DriverPrintResultHeader("ARGetListSchemaWithAlias");
   PrintReturnCode(result);
   if (result < AR_RETURN_ERROR)
   {
      PrintARNameList("Schema list: ", "   ", &schemaList);
      PrintARNameList("Alias list: ", "   ", &aliasList);
   }
   PrintARStatusList(&status);

   /* Cleanup */
   if (result < AR_RETURN_ERROR)
   {
      FreeARNameList(&schemaList, FALSE);
      FreeARNameList(&aliasList, FALSE);
   }
   FreeARStatusList(&status, FALSE);
   FreeARInternalIdList(&fieldIdList, FALSE);
}


/*****************************************************************************/
/*                                                                           */
/*                          APIARGetLocalizedValue                           */
/*                                                                           */
/*****************************************************************************/

void APIARGetLocalizedValue()
{
   int            result;
   ARStatusList   status={0, NULL};
   ARTimestamp    timestamp;
   ARTimestamp    *ptrTimestamp;

   ARCMenuType *returnValue = NULL;
   ARLocalizedRequestStruct lms;
   ARValueStruct value;
   ARValueStruct *ptrValue = NULL;

   /* Get Parameters */
   DriverPrintHeader("GET LOCALIZED TEXT");

   GetARLocalizedRequestStruct(&lms);
   
   if (GetARBoolean("   Get value struct? (T): ", TRUE))
      ptrValue = &value;
   else
      ptrValue = NULL;

   if (GetARBoolean("   Get timestamp? (T): ", TRUE))
      ptrTimestamp = &timestamp;
   else
      ptrTimestamp = NULL;

   /* Call routine */
   BeginAPICall();
   result = ARGetLocalizedValue(GetControlStructPtr(), &lms, ptrValue, ptrTimestamp, &status);
   EndAPICall(&status);

   /* Print results */
   DriverPrintResultHeader("ARGetLocalizedValue");
   PrintReturnCode(result);
   if (result < AR_RETURN_ERROR)
   {
      if (ptrValue)
         PrintARValueStruct("Localized Text: ", &value);
      if (ptrTimestamp)
         PrintARTimestamp("Timestamp: ", timestamp);
   }
   PrintARStatusList(&status);

   /* Cleanup */
   FreeARStatusList(&status, FALSE);
}


/*****************************************************************************/
/*                                                                           */
/*                      APIARGetMultipleLocalizedValues                      */
/*                                                                           */
/*****************************************************************************/

void APIARGetMultipleLocalizedValues()
{
   int            result;
   ARStatusList   status={0, NULL};
   unsigned int   i;

   ARLocalizedRequestList lml;
   ARValueList valueList;
   ARValueList *ptrValueList;
   ARTimestampList timestampList;
   ARTimestampList *ptrTimestampList;

   /* Get Parameters */
   DriverPrintHeader("GET MULT LOCALIZED TEXT");

   GetARLocalizedRequestList(&lml);
   if (GetARBoolean("   Get value list? (T): ", TRUE))
      ptrValueList = &valueList;
   else
      ptrValueList = NULL;

   if (GetARBoolean("   Get timestamp list? (T): ", TRUE))
      ptrTimestampList = &timestampList;
   else
      ptrTimestampList = NULL;

   /* Call routine */
   BeginAPICall();
   result = ARGetMultipleLocalizedValues(GetControlStructPtr(), &lml, ptrValueList, ptrTimestampList, &status);
   EndAPICall(&status);

   /* Print results */
   DriverPrintResultHeader("ARGetLocalizedValue");
   PrintReturnCode(result);
   if (result < AR_RETURN_ERROR)
   {
      if (ptrValueList)
      {
         for (i=0;i<valueList.numItems;i++)
         {
            PrintARValueStruct("Localized Text: ", &valueList.valueList[i]);
         }
      }

      if (ptrTimestampList)
      {
         for (i=0;i<timestampList.numItems;i++)
         {
            PrintARTimestamp("Timestamp: ", timestampList.timestampList[i]);
         }
      }
   }
   PrintARStatusList(&status);
   FreeARValueList(&valueList, FALSE);
   FreeARTimestampList(&timestampList, FALSE);

   /* Cleanup */
   FreeARStatusList(&status, FALSE);
}


/*****************************************************************************/
/*                                                                           */
/*                           APIARCreateAlertEvent                           */
/*                                                                           */
/*****************************************************************************/

void APIARCreateAlertEvent()

{
   int                alertPriority;
   ARNameType         alertSource;
   char               alertText[1024];
   AREntryIdType      entryId;
   ARNameType         formName;
   char               objectId[256];
   int                result;
   ARServerNameType   serverName;
   ARStatusList       status={0, NULL};
   ARAccessNameType   user;

   /* Get Parameters */
   DriverPrintHeader("CREATE ALERT EVENT");
   strcpy(user, GetChar("User name (): ", ""));
   strcpy(alertText, GetChar("Alert text (): ", ""));
   alertPriority = GetInt("Alert priority (0): ", 0);
   strcpy(alertSource, GetChar("Alert source (): ", ""));
   strcpy(serverName, GetChar("Server name (): ", ""));
   strcpy(formName, GetChar("Form name (): ", ""));
   strcpy(objectId, GetChar("Object Id (): ", ""));

   /* Call routine */
   BeginAPICall();
   result = ARCreateAlertEvent(GetControlStructPtr(), user, alertText,
                               alertPriority, alertSource, serverName,
                               formName, objectId, entryId, &status);
   EndAPICall(&status);

   /* Print results */
   DriverPrintResultHeader("ARCreateAlertEvent");
   PrintReturnCode(result);
   PrintChar("Entry id: ", entryId);
   PrintARStatusList(&status);

   /* Cleanup */
   FreeARStatusList(&status, FALSE);
}


/*****************************************************************************/
/*                                                                           */
/*                          APIARRegisterForAlerts                           */
/*                                                                           */
/*****************************************************************************/

void APIARRegisterForAlerts()

{
   int            clientPort;
   unsigned int   registrationFlags;
   int            result;
   ARStatusList   status={0, NULL};

   /* Get Parameters */
   DriverPrintHeader("REGISTER FOR ALERTS");
   clientPort = GetInt("Client port (0): ", 0);
   registrationFlags = GetInt("Registration flags (0): ", 0);

   /* Call routine */
   BeginAPICall();
   result = ARRegisterForAlerts(GetControlStructPtr(), clientPort,
                                registrationFlags, &status);
   EndAPICall(&status);

   /* Print results */
   DriverPrintResultHeader("ARRegisterForAlerts");
   PrintReturnCode(result);
   PrintARStatusList(&status);

   /* Cleanup */
   FreeARStatusList(&status, FALSE);
}


/*****************************************************************************/
/*                                                                           */
/*                         APIARDeregisterForAlerts                          */
/*                                                                           */
/*****************************************************************************/

void APIARDeregisterForAlerts()

{
   int            clientPort;
   int            result;
   ARStatusList   status={0, NULL};

   /* Get Parameters */
   DriverPrintHeader("DEREGISTER FOR ALERTS");
   clientPort = GetInt("Client port (0): ", 0);

   /* Call routine */
   BeginAPICall();
   result = ARDeregisterForAlerts(GetControlStructPtr(), clientPort, &status);
   EndAPICall(&status);

   /* Print results */
   DriverPrintResultHeader("ARDeregisterForAlerts");
   PrintReturnCode(result);
   PrintARStatusList(&status);

   /* Cleanup */
   FreeARStatusList(&status, FALSE);
}


/*****************************************************************************/
/*                                                                           */
/*                           APIARGetListAlertUser                           */
/*                                                                           */
/*****************************************************************************/

void APIARGetListAlertUser()

{
   int                result;
   ARStatusList       status={0, NULL};
   ARAccessNameList   userList;

   /* Get Parameters */
   DriverPrintHeader("GET LIST ALERT USER");

   /* Call routine */
   BeginAPICall();
   result = ARGetListAlertUser(GetControlStructPtr(), &userList, &status);
   EndAPICall(&status);

   /* Print results */
   DriverPrintResultHeader("ARGetListAlertUser");
   PrintReturnCode(result);
   if (result < AR_RETURN_ERROR)
      PrintARAccessNameList("User list: ", "   ", &userList);
   PrintARStatusList(&status);

   /* Cleanup */
   if (result < AR_RETURN_ERROR)
      FreeARAccessNameList(&userList, FALSE);
   FreeARStatusList(&status, FALSE);
}


/*****************************************************************************/
/*                                                                           */
/*                            APIARGetAlertCount                             */
/*                                                                           */
/*****************************************************************************/

void APIARGetAlertCount()

{
   unsigned int        count;
   int                 result;
   ARQualifierStruct   qualifier;
   ARStatusList        status={0, NULL};

   /* Get Parameters */
   DriverPrintHeader("GET ALERT COUNT");
   GetARQualifierStruct(&qualifier);

   /* Call routine */
   BeginAPICall();
   result = ARGetAlertCount(GetControlStructPtr(), &qualifier, &count, &status);
   EndAPICall(&status);

   /* Print results */
   DriverPrintResultHeader("ARGetAlertCount");
   PrintReturnCode(result);
   if (result < AR_RETURN_ERROR)
      PrintInt("Alert Count: ", count);
   PrintARStatusList(&status);

   /* Cleanup */
   FreeARStatusList(&status, FALSE);
}


/*****************************************************************************/
/*                                                                           */
/*                          APIARDecodeAlertMessage                          */
/*                                                                           */
/*****************************************************************************/

void APIARDecodeAlertMessage()

{
   char               *alertText = NULL;
   char               *formName = NULL;
   char               *messagePtr;
   char               *objectId = NULL;
   unsigned int        priority;
   int                 result;
   char               *serverName = NULL;
   char               *sourceTag = NULL;
   unsigned int        sourceType;
   ARStatusList        status={0, NULL};
   ARTimestamp         timestamp;

   /* Get Parameters */
   DriverPrintHeader("DECODE ALERT MESSAGE");
   messagePtr = GetChar("Alert message: ", "");

   /* Call routine */
   BeginAPICall();
   result = ARDecodeAlertMessage(GetControlStructPtr(), messagePtr,
                                 strlen(messagePtr) + 1, &timestamp,
                                 &sourceType, &priority, &alertText, &sourceTag,
                                 &serverName, &formName, &objectId, &status);
   EndAPICall(&status);

   /* Print results */
   DriverPrintResultHeader("ARDecodeAlertMessage");
   PrintReturnCode(result);
   if (result < AR_RETURN_ERROR)
   {
      PrintInt("Timestamp: ", timestamp);
      PrintInt("Source type: ", sourceType);
      PrintInt("Priority: ", priority);
      PrintChar("Alert text: ", alertText);
      PrintChar("Source tag: ", sourceTag);
      PrintChar("Server name: ", serverName);
      PrintChar("Form name: ", formName);
      PrintChar("Object Id: ", objectId);
   }
   PrintARStatusList(&status);

   /* Cleanup */
   if (result < AR_RETURN_ERROR)
   {
      if (alertText)
         free(alertText);
      if (sourceTag)
         free(sourceTag);
      if (serverName)
         free(serverName);
      if (formName)
         free(formName);
      if (objectId)
         free(objectId);
   }
   FreeARStatusList(&status, FALSE);
}


/*****************************************************************************/
/*                                                                           */
/*                         APIARDecodeARQualifierStruct                      */
/*                                                                           */
/*****************************************************************************/

void APIARDecodeARQualifierStruct()
{
   int                 result;
   ARQualifierStruct   qualStruct;
   char               *qualTextPtr;
   ARStatusList        status={0, NULL};

   status.numItems = 0;
   /* Get Parameters */
   DriverPrintHeader("DECODE STRING TO ARQUALIFIER STRUCT");
   qualTextPtr = GetChar("Qualifier Text: ", "");

   /* Call routine */
   result = ARDecodeARQualifierStruct(GetControlStructPtr(), qualTextPtr, &qualStruct,
                                      &status);
   /* Print results */
   DriverPrintResultHeader("ARDecodeARQualifierStruct");
   PrintReturnCode(result);
   if (result < AR_RETURN_ERROR)
      PrintARQualifierStruct("", &qualStruct);
   PrintARStatusList(&status);

   /* Cleanup */
   if (result < AR_RETURN_ERROR)
      FreeARQualifierStruct(&qualStruct, FALSE);
   FreeARStatusList(&status, FALSE);
}


/*****************************************************************************/
/*                                                                           */
/*                         APIAREncodeARQualifierStruct                      */
/*                                                                           */
/*****************************************************************************/

void APIAREncodeARQualifierStruct()
{
   int                  result;
   char                *qualTextPtr = NULL;
   ARQualifierStruct    qualStruct;
   ARStatusList         status={0, NULL};

   status.numItems = 0;
   /* Get Parameters */
   DriverPrintHeader("ENCODE ARQUALIFIER STRUCT TO STRING");
   GetARQualifierStruct(&qualStruct);

   /* Call routine */
   result = AREncodeARQualifierStruct(GetControlStructPtr(), &qualStruct, &qualTextPtr,
                                      &status);
   /* Print results */
   DriverPrintResultHeader("AREncodeARQualifierStruct");
   PrintReturnCode(result);
   if (result < AR_RETURN_ERROR)
      PrintChar("Qualifier Text: ", qualTextPtr);
   PrintARStatusList(&status);

   /* Cleanup */
   if (result < AR_RETURN_ERROR && qualTextPtr)
      free(qualTextPtr);
   FreeARQualifierStruct(&qualStruct, FALSE);
   FreeARStatusList(&status, FALSE);
}


/*****************************************************************************/
/*                                                                           */
/*                         APIARDecodeARAssignStruct                         */
/*                                                                           */
/*****************************************************************************/

void APIARDecodeARAssignStruct()
{
   int                  result;
   ARAssignStruct       assignStruct;
   char                *assignTextPtr;
   ARStatusList         status={0, NULL};

   status.numItems = 0;
   /* Get Parameters */
   DriverPrintHeader("DECODE STRING TO ARASSIGN STRUCT");
   assignTextPtr = GetChar("Assign Text: ", "");

   /* Call routine */
   result = ARDecodeARAssignStruct(GetControlStructPtr(), assignTextPtr, &assignStruct,
                                   &status);
   /* Print results */
   DriverPrintResultHeader("ARDecodeARAssignStruct");
   PrintReturnCode(result);
   if (result < AR_RETURN_ERROR)
      PrintARAssignStruct("", &assignStruct);
   PrintARStatusList(&status);

   /* Cleanup */
   if (result < AR_RETURN_ERROR)
      FreeARAssignStruct(&assignStruct, FALSE);
   FreeARStatusList(&status, FALSE);
}


/*****************************************************************************/
/*                                                                           */
/*                         APIAREncodeARAssignStruct                         */
/*                                                                           */
/*****************************************************************************/

void APIAREncodeARAssignStruct()
{
   int                  result;
   char                *assignTextPtr = NULL;
   ARAssignStruct       assignStruct;
   ARStatusList         status={0, NULL};

   status.numItems = 0;
   /* Get Parameters */
   DriverPrintHeader("ENCODE ARASSIGN STRUCT TO STRING");
   GetARAssignStruct(&assignStruct);

   /* Call routine */
   result = AREncodeARAssignStruct(GetControlStructPtr(), &assignStruct, &assignTextPtr,
                                   &status);
   /* Print results */
   DriverPrintResultHeader("AREncodeARAssignStruct");
   PrintReturnCode(result);
   if (result < AR_RETURN_ERROR)
      PrintChar("Assign Text: ", assignTextPtr);
   PrintARStatusList(&status);

   /* Cleanup */
   if (result < AR_RETURN_ERROR && assignTextPtr)
      free(assignTextPtr);
   FreeARAssignStruct(&assignStruct, FALSE);
   FreeARStatusList(&status, FALSE);
}


/*****************************************************************************/
/*                                                                           */
/*                         APIAREncodeStatusHistory                          */
/*                                                                           */
/*****************************************************************************/

void APIAREncodeStatusHistory()
{
   int                  result;
   char                *statHistString = NULL;
   ARStatusHistoryList  statHistList;
   ARStatusList         status={0, NULL};

   status.numItems = 0;
   /* Get Parameters */
   DriverPrintHeader("ENCODE STATUS HISTORY LIST TO STRING");
   GetARStatusHistoryList(&statHistList);

   /* Call routine */
   result = AREncodeStatusHistory(GetControlStructPtr(), &statHistList, &statHistString,
                                  &status);
   /* Print results */
   DriverPrintResultHeader("AREncodeStatusHistory");
   PrintReturnCode(result);
   if (result < AR_RETURN_ERROR)
      PrintChar("Status History String: ", statHistString);
   PrintARStatusList(&status);

   /* Cleanup */
   if (statHistString)
      free(statHistString);
   FreeARStatusHistoryList(&statHistList, FALSE);
   FreeARStatusList(&status, FALSE);
}


/*****************************************************************************/
/*                                                                           */
/*                              APIAREncodeDiary                             */
/*                                                                           */
/*****************************************************************************/

void APIAREncodeDiary()
{
   int                  result;
   char                *diaryString = NULL;
   ARDiaryList          diaryList;
   ARStatusList         status={0, NULL};

   status.numItems = 0;
   /* Get Parameters */
   DriverPrintHeader("ENCODE DIARY LIST TO STRING");
   GetARDiaryList(&diaryList);

   /* Call routine */
   result = AREncodeDiary(GetControlStructPtr(), &diaryList, &diaryString, &status);
                                  
   /* Print results */
   DriverPrintResultHeader("AREncodeDiary");
   PrintReturnCode(result);
   if (result < AR_RETURN_ERROR)
      PrintChar("Diary String: ", diaryString);
   PrintARStatusList(&status);

   /* Cleanup */
   if (diaryString)
      free(diaryString);
   FreeARDiaryList(&diaryList, FALSE);
   FreeARStatusList(&status, FALSE);
}


/*****************************************************************************/
/*                                                                           */
/*                             APIARExpandSSMenu                             */
/*                                                                           */
/*****************************************************************************/

void APIARExpandSSMenu()

{
   ARCharMenuStruct     menu;
   ARCharMenuSSStruct   menuDefn; 
   int                  result;
   ARStatusList         status={0, NULL};

   /* Get Parameters */
   DriverPrintHeader("EXPAND SS MENU");
   DriverPrintPrompt("Server-side menu struct:\n");
   GetARCharMenuSSStruct(&menuDefn);

   /* Call routine */
   BeginAPICall();
   result = ARExpandSSMenu(GetControlStructPtr(), &menuDefn, &menu, &status);
   EndAPICall(&status);

   /* Print results */
   DriverPrintResultHeader("ARExpandSSMenu");
   PrintReturnCode(result);
   if (result < AR_RETURN_ERROR)
      PrintARCharMenuStruct("", &menu);
   PrintARStatusList(&status);

   /* Cleanup */
   FreeARCharMenuSSStruct(&menuDefn, FALSE);
   if (result < AR_RETURN_ERROR)
      FreeARCharMenuStruct(&menu, FALSE);
   FreeARStatusList(&status, FALSE);
}


/*****************************************************************************/
/*                                                                           */
/*                         APIARGetListExtSchemaCandidates                   */
/*                                                                           */
/*****************************************************************************/

void APIARGetListExtSchemaCandidates()
{
   int                  result;
   unsigned int         schemaType;
   ARStatusList         status={0, NULL};
   ARCompoundSchemaList schema;

   status.numItems = 0;
   schema.numItems = 0;

   /* Get Parameters */
   DriverPrintHeader("GET LIST EXTERNAL SCHEMA CANDIDATES");
   schemaType = GetInt("Schema Type (0): ", 0);

   /* Call routine */
   result = ARGetListExtSchemaCandidates(GetControlStructPtr(), schemaType, &schema, &status);
                                  
   /* Print results */
   DriverPrintResultHeader("ARGetListExtSchemaCandidates");
   PrintReturnCode(result);
   if (result < AR_RETURN_ERROR)
      PrintARCompoundSchemaList(&schema);
   PrintARStatusList(&status);

   /* Cleanup */
   FreeARCompoundSchemaList(&schema, FALSE);
   FreeARStatusList(&status, FALSE);
}


/*****************************************************************************/
/*                                                                           */
/*                     APIARGetMultipleExtFieldCandidates                    */
/*                                                                           */
/*****************************************************************************/

void APIARGetMultipleExtFieldCandidates()
{
   int                  result;
   ARStatusList         status={0, NULL};
   ARCompoundSchema     schema;
   ARFieldMappingList   fieldMapping;
   ARFieldLimitList     limit;
   ARUnsignedIntList    dataType;

   status.numItems = 0;
   memset(&schema, 0, sizeof(schema));
   /* Get Parameters */
   DriverPrintHeader("GET MULTIPLE EXTERNAL FIELD CANDIDATES");
   GetARCompoundSchema(&schema);

   /* Call routine */
   result = ARGetMultipleExtFieldCandidates(GetControlStructPtr(), &schema, &fieldMapping, &limit, &dataType, &status);
                                  
   /* Print results */
   DriverPrintResultHeader("ARGetMultipleExtFieldCandidates");
   PrintReturnCode(result);
   if (result < AR_RETURN_ERROR)
   {
      unsigned int i;

      for (i = 0; i < fieldMapping.numItems; i++)
      {
         PrintARFieldMapping("", &fieldMapping.mappingList[i]);
         PrintARFieldLimitStruct(&limit.fieldLimitList[i]);
         PrintInt("Datatype: ", dataType.intList[i]);
      }
   }
   
   PrintARStatusList(&status);

   /* Cleanup */
   FreeARCompoundSchema(&schema, FALSE);
   FreeARStatusList(&status, FALSE);
}


/*****************************************************************************/
/*                                                                           */
/*                           APIARValidateLicense                            */
/*                                                                           */
/*****************************************************************************/

void APIARValidateLicense()

{
   ARLicenseValidStruct          licenseValid;
   ARLicenseNameType             licType;
   ARStatusList                  status={0, NULL};
   int                           result = 0;

                                       /* print header */
   DriverPrintHeader("VALIDATE LICENSE");
                                       /* get parameter */
   strcpy(licType, GetChar("License Type: ", ""));

   memset(&status, 0, sizeof(ARStatusList));
   memset(&licenseValid, 0, sizeof(ARLicenseValidStruct));
                                       /* Call routine */
   BeginAPICall();
   result = ARValidateLicense(GetControlStructPtr(),
                              licType, 
                              &licenseValid,
                              &status);
   EndAPICall(&status);

   /* Print results */
   DriverPrintResultHeader("ARValidateLicense");
   PrintReturnCode(result);
   if (result < AR_RETURN_ERROR)
   {
      PrintInt ("Number Of Licenses: ", licenseValid.numLicenses);
      PrintInt ("            IsDemo: ", licenseValid.isDemo);
      PrintInt ("Expire Date: month: ", licenseValid.expireDate.month);
      PrintInt ("               day: ", licenseValid.expireDate.day);
      PrintInt ("              year: ", licenseValid.expireDate.year);
      PrintChar("        Token List: ", licenseValid.tokenList);
   }
   PrintARStatusList(&status);
 
   /* Cleanup */

   FreeARStatusList(&status, FALSE);
   FreeARLicenseValidStruct(&licenseValid, FALSE);
}
 


/*****************************************************************************/
/*                                                                           */
/*                       APIARValidateMultipleLicenses                       */
/*                                                                           */
/*****************************************************************************/

void APIARValidateMultipleLicenses()

{
   ARLicenseNameList             licenseTypeList;
   ARLicenseValidList            licenseValidList;
   ARLicenseValidStruct         *temp;
   ARLicenseNameType            *tmpNameType;
   ARStatusList                  status={0, NULL};
   int                           num, i;
   int                           result = 0;
   unsigned int                  j;

   memset(&status, 0, sizeof(ARStatusList));
   memset(&licenseValidList, 0, sizeof(ARLicenseValidList));
   memset(&licenseTypeList, 0, sizeof(ARLicenseNameList));

                                       /* print header */
   DriverPrintHeader("VALIDATE MULTIPLE LICENSES");
                                       /* get parameter */

   num = GetInt("Number of licenses to validate (1): ", 1);

   if (num == 0)
   {
      /* don't want to validate anything */
      return;
   }
   else
   {
      licenseTypeList.nameList = (ARLicenseNameType *) calloc (
                                       num, sizeof(ARLicenseNameType));
      if (licenseTypeList.nameList == NULL)
      {
         DriverPrintError("**** malloc failure for ARValidateMultipleLicenses call ****\n");
         return;
      }

      for (i = 0; i < num; i++)
      {
         strcpy(licenseTypeList.nameList[i], GetChar("License Type: ", ""));
         licenseTypeList.numItems++;
      }
   }

   /* Call routine */
   BeginAPICall();
   result = ARValidateMultipleLicenses(GetControlStructPtr(),
                                       &licenseTypeList, 
                                       &licenseValidList,
                                       &status);
   EndAPICall(&status);

   /* Print results */
   DriverPrintResultHeader("ARValidateMultipleLicenses");
   PrintReturnCode(result);

   if (result < AR_RETURN_ERROR)
   {
      temp = licenseValidList.licenseValidateInfoList;
      tmpNameType = licenseTypeList.nameList;
      for (j = 0; j < licenseValidList.numItems; j++)
      {
         PrintChar("License Type: ", tmpNameType[j]);
         PrintInt ("  Number Of Licenses: ", temp[j].numLicenses);
         PrintInt ("              IsDemo: ", temp[j].isDemo);
         PrintInt ("  Expire Date: month: ", temp[j].expireDate.month);
         PrintInt ("                 day: ", temp[j].expireDate.day);
         PrintInt ("                year: ", temp[j].expireDate.year);
         PrintChar("          Token List: ", temp[j].tokenList);  
         PrintChar("", NULL);
      }
   }

   PrintARStatusList(&status);
 
   /* Cleanup */
   FreeARStatusList(&status, FALSE);
   FreeARLicenseNameList(&licenseTypeList, FALSE);
   FreeARLicenseValidList(&licenseValidList, FALSE);
}
 

/*****************************************************************************/
/*                                                                           */
/*                            APIARGetListLicense                            */
/*                                                                           */
/*****************************************************************************/

void APIARGetListLicense()

{
   ARLicenseInfoList             licenseInfoList;
   ARLicenseNameType             licType;
   ARLicenseInfoStruct          *tempInfo;
   ARStatusList                  status={0, NULL};
   unsigned int                  i;
   int                           result = 0;

                                       /* initialize */
   memset(&status, 0L, sizeof(ARStatusList));
   memset(&licenseInfoList, 0L, sizeof(ARLicenseInfoList));

                                       /* print header */
   DriverPrintHeader("GET LIST LICENSE");
                                       /* get parameter */
   strcpy(licType, GetChar("License Type: ", ""));

   /* Call routine */
   BeginAPICall();
   result = ARGetListLicense(GetControlStructPtr(),
                             licType, 
                             &licenseInfoList,
                             &status);
   EndAPICall(&status);

   /* Print results */
   DriverPrintResultHeader("ARGetListLicense");
   PrintReturnCode(result);

   if (result < AR_RETURN_ERROR)
   {
      tempInfo = licenseInfoList.licenseInfoList;
      PrintInt("Total Number of Licenses: ", licenseInfoList.numItems);
      for (i = 0; i < licenseInfoList.numItems; i++)
      {
         PrintChar("              License Type: ", tempInfo[i].licType);
         PrintChar("               License Key: ", tempInfo[i].licKey);
         PrintChar("          License Sub Type: ", tempInfo[i].licSubtype);
         PrintInt ("License Issued Date: month: ",tempInfo[i].issuedDate.month);
         PrintInt ("                       day: ", tempInfo[i].issuedDate.day);
         PrintInt ("                      year: ", tempInfo[i].issuedDate.year);
         PrintInt ("License Expire Date: month: ",tempInfo[i].expireDate.month);
         PrintInt ("                       day: ", tempInfo[i].expireDate.day);
         PrintInt ("                      year: ", tempInfo[i].expireDate.year);
         PrintChar("                 Site Name: ", tempInfo[i].siteName);
         PrintChar("                   Host ID: ", tempInfo[i].hostId);
         PrintInt ("        Number of Licenses: ", tempInfo[i].numLicenses);
         PrintChar("                Token List: ", tempInfo[i].tokenList);
         PrintChar("                   Comment: ", tempInfo[i].comment);
         PrintChar("", NULL);
      }
   }

   PrintARStatusList(&status);
 
   /* Cleanup */
   FreeARStatusList(&status, FALSE);
   FreeARLicenseInfoList(&licenseInfoList, FALSE);  
}
 

/*****************************************************************************/
/*                                                                           */
/*                            APIARCreateLicense                             */
/*                                                                           */
/*****************************************************************************/

void APIARCreateLicense()

{
   ARLicenseInfoStruct           licenseInfoStruct;
   ARStatusList                  status={0, NULL};
   int                           result = 0;
   char                          buffer[80];

   memset(&licenseInfoStruct, 0L, sizeof(ARLicenseInfoStruct));
   memset(&buffer, 0L, 80);
                                       /* print header */
   DriverPrintHeader("CREATE LICENSE");
                                       /* get parameter */
   strcpy(licenseInfoStruct.licKey, GetChar("License Key: ", ""));
   strcpy(licenseInfoStruct.licType, GetChar("License Type: ", ""));
   strcpy(licenseInfoStruct.licSubtype, GetChar("License Sub Type: ", ""));
   licenseInfoStruct.issuedDate.month = GetInt("Issued Month (January-December, 0-11): (0)", 0);
   licenseInfoStruct.issuedDate.day = GetInt("Issued Day (1-31): (1)", 1);
   licenseInfoStruct.issuedDate.year = GetInt("Issued Year (yyyy): (2001)", 2001);
   licenseInfoStruct.issuedDate.year = licenseInfoStruct.issuedDate.year - 1900;

   /* "never expire" day would be Jan-01-1970 that's the default time. */
   licenseInfoStruct.expireDate.month = GetInt("Expire Month (January-December, 0-11): (0)", 0);
   licenseInfoStruct.expireDate.day = GetInt("Expire Day (1-31): (1)", 1);
   licenseInfoStruct.expireDate.year = GetInt("Expire Year (yyyy): (1970)", 1970);
   licenseInfoStruct.expireDate.year = licenseInfoStruct.expireDate.year - 1900;
   
   strcpy(buffer, GetChar("Site Name: ", ""));
   licenseInfoStruct.siteName = strdup(buffer);
   memset(buffer, 0L, 80);

   strcpy(buffer, GetChar("Host ID:", ""));
   licenseInfoStruct.hostId = strdup(buffer);
   memset(buffer, 0L, 80);

   licenseInfoStruct.numLicenses = GetInt("Number of Licenses: (1)", 1);

   strcpy(buffer, GetChar("Token List: ", ""));
   licenseInfoStruct.tokenList = strdup(buffer);
   memset(buffer, 0L, 80);

   strcpy(buffer, GetChar("Comment: ", ""));
   licenseInfoStruct.comment = strdup(buffer);
   memset(buffer, 0L, 80);
   

   /* Call routine */
   BeginAPICall();
   result = ARCreateLicense(GetControlStructPtr(),
                            &licenseInfoStruct, 
                            &status);
   EndAPICall(&status);

   /* Print results */
   DriverPrintResultHeader("ARCreateLicense");
   PrintReturnCode(result);
   PrintARStatusList(&status);
 
   /* Cleanup */

   FreeARStatusList(&status, FALSE);
   FreeARLicenseInfoStruct(&licenseInfoStruct, FALSE);

}
 

/*****************************************************************************/
/*                                                                           */
/*                            APIARDeleteLicense                             */
/*                                                                           */
/*****************************************************************************/

void APIARDeleteLicense()

{
   ARLicenseKeyType              licKey;
   ARLicenseNameType             licType;
   ARStatusList                  status={0, NULL};
   int                           result = 0;

                                       /* print header */
   DriverPrintHeader("DELETE LICENSE");
                                       /* get parameter */
   strcpy(licType, GetChar("License Type: ", ""));
   strcpy(licKey, GetChar("License key: ", ""));

   /* Call routine */
   BeginAPICall();
   result = ARDeleteLicense(GetControlStructPtr(),
                            licType, 
                            licKey,
                            &status);
   EndAPICall(&status);

   /* Print results */
   DriverPrintResultHeader("ARDeleteLicense");
   PrintReturnCode(result);
   PrintARStatusList(&status);
 
   /* Cleanup */

   FreeARStatusList(&status, FALSE);
}
 

/*****************************************************************************/
/*                                                                           */
/*                       APIARGetListSQLForActiveLink                        */
/*                                                                           */
/*****************************************************************************/

void APIARGetListSQLForActiveLink()

{
   ARNameType        actlinkName;
   unsigned int      actionIndex;
   unsigned int      actionType;
   ARTimestamp       timestamp;
   ARFieldValueList  keywordList;
   unsigned int      maxRetrieve;
   unsigned int      numMatches;
   unsigned int     *numMatchesPtr;
   ARFieldValueList  parameterList;
   int               result;
   ARStatusList      status={0, NULL};
   ARValueListList   valueListList;
   char              *returnCommand = NULL;
   char              **returnCommandPtr;

   /* Get Parameters */
   DriverPrintHeader("GETLIST SQL FOR ACTIVE LINK");
   strcpy(actlinkName, GetChar("Active Link Name: ", ""));
   actionIndex = GetInt("Action Index (0): ", 0);
   actionType = GetInt("Action Type (0): ", 0);
   timestamp = GetInt("Timestamp (0): ", 0);
   DriverPrintPrompt("Keyword List:\n");
   GetARFieldValueList(&keywordList);
   DriverPrintPrompt("Parameter List:\n");
   GetARFieldValueList(&parameterList);
   maxRetrieve = GetInt("Maximum number of entries to retrieve (500): ", 500);
   if (GetARBoolean("Get number of matches? (F): ", FALSE))
      numMatchesPtr = &numMatches;
   else
      numMatchesPtr = NULL;

   if (GetARBoolean("Get return command? (F): ", FALSE))
      returnCommandPtr = &returnCommand;
   else
      returnCommandPtr = NULL;

   /* Call routine */
   BeginAPICall();
   result = ARGetListSQLForActiveLink(GetControlStructPtr(), actlinkName,
                                      actionIndex, actionType, timestamp,
                                      &keywordList, &parameterList, maxRetrieve, 
                                      &valueListList, numMatchesPtr,
                                      returnCommandPtr, &status);
   EndAPICall(&status);

   /* Print results */
   DriverPrintResultHeader("ARGetListSQLForActiveLink");
   PrintReturnCode(result);
   if (result < AR_RETURN_ERROR)
   {
      PrintARValueListList(&valueListList);
      if (numMatchesPtr != NULL)
          PrintInt("Number of matches: ", numMatches);
      if (returnCommandPtr != NULL)
          PrintChar("Returned Command: ", returnCommand);
   }
   PrintARStatusList(&status);

   /* Cleanup */
   if (result < AR_RETURN_ERROR)
      FreeARValueListList(&valueListList, FALSE);
   FreeARStatusList(&status, FALSE);
}


/*****************************************************************************/
/*                                                                           */
/*                     APIARExecuteProcessForActiveLink                      */
/*                                                                           */
/*****************************************************************************/

void APIARExecuteProcessForActiveLink()

{
   ARNameType        actlinkName;
   unsigned int      actionIndex;
   unsigned int      actionType;
   unsigned int      fieldid;
   ARTimestamp       timestamp;
   ARFieldValueList  keywordList;
   ARFieldValueList  parameterList;
   int               result;
   int               returnStatus;
   int              *returnStatusPtr;
   char             *returnString;
   char            **returnStringPtr;
   char             *returnCommand;
   char            **returnCommandPtr;
   ARStatusList      status={0, NULL};

   /* Get Parameters */
   DriverPrintHeader("EXECUTE PROCESS FOR ACTIVE LINK");
   strcpy(actlinkName, GetChar("Active Link Name: ", ""));
   actionIndex = GetInt("Action Index (0): ", 0);
   actionType = GetInt("actionType (0): ", 0);
   fieldid = GetInt("fieldid (0): ", 0);
   timestamp = GetInt("timestamp (0): ", 0);
   DriverPrintPrompt("Keyword List:\n");
   GetARFieldValueList(&keywordList);
   DriverPrintPrompt("Parameter List:\n");
   GetARFieldValueList(&parameterList);
   if (GetARBoolean("Wait for process to complete? (F): ", FALSE))
   {
      returnStatusPtr = &returnStatus;
      returnStringPtr = &returnString;
   }
   else
   {
      returnStatusPtr = NULL;
      returnStringPtr = NULL;
   }
   
   if (GetARBoolean("Get return command? (F): ", FALSE))
      returnCommandPtr = &returnCommand;
   else
      returnCommandPtr = NULL;

   /* Call routine */
   BeginAPICall();
   result = ARExecuteProcessForActiveLink(GetControlStructPtr(), actlinkName,
                                          actionIndex, actionType, fieldid,
                                          timestamp, &keywordList,
                                          &parameterList, returnStatusPtr,
                                          returnStringPtr, returnCommandPtr,
                                          &status);
   EndAPICall(&status);

   /* Print results */
   DriverPrintResultHeader("ARExecuteProcessForActiveLink");
   PrintReturnCode(result);
   if (result < AR_RETURN_ERROR)
   {
      if (returnStatusPtr != NULL)
         PrintInt("Return Status: ", *returnStatusPtr);
      if (returnStringPtr != NULL)
         PrintChar("Process output: ", *returnStringPtr);
      if (returnCommandPtr != NULL)
         PrintChar("Return command: ", *returnCommandPtr);
   }
   PrintARStatusList(&status);

   /* Cleanup */
   FreeARStatusList(&status, FALSE);
}


/*****************************************************************************/
/*                                                                           */
/*                        APIARGetSessionConfiguration                       */
/*                                                                           */
/*****************************************************************************/

void APIARGetSessionConfiguration()
{
   int               result;
   unsigned int      variableId;
   ARValueStruct     variableValue;
   ARStatusList      status={0, NULL};

   /* Get Parameters */
   DriverPrintHeader("GET SESSION CONFIGURATION");

   variableId = GetInt("Session variable Id (0): ", 0);

   /* Call routine */

   BeginAPICall();
   result = ARGetSessionConfiguration(GetControlStructPtr(), variableId,
                                      &variableValue, &status);

   EndAPICall(&status);

   /* Print results */
   DriverPrintResultHeader("ARGetSessionConfiguration");
   PrintReturnCode(result);

   if (result < AR_RETURN_ERROR)
      PrintARValueStruct("", &variableValue);
   PrintARStatusList(&status);

   /* Cleanup */
   if (result < AR_RETURN_ERROR)
      FreeARValueStruct(&variableValue, FALSE);
   FreeARStatusList(&status, FALSE);
}


/*****************************************************************************/
/*                                                                           */
/*                         APIARSetSessionConfiguration                      */
/*                                                                           */
/*****************************************************************************/

void APIARSetSessionConfiguration()
{
   int               result;
   int               dataType;
   unsigned int      variableId;
   ARValueStruct     variableValue;
   ARStatusList      status={0, NULL};

   /* Get Parameters */
   DriverPrintHeader("SET SESSION CONFIGURATION");

   variableId = GetInt("Session variable Id (0): ", 0);

   dataType = GetInt("Datatype Null/Key/Int/Real/Char/Diary/Enum/Time/Bitmask/Byte/Decimal/\n         attach/join/trim/control/table/column/ulong/coords/view/display\n         (0 - 11, 30-34, 40-43) (2): ",
                      AR_DATA_TYPE_INTEGER);
   variableValue.dataType = dataType;
   GetValue(&variableValue);

   /* Call routine */

   BeginAPICall();
   result = ARSetSessionConfiguration(GetControlStructPtr(), variableId,
                                      &variableValue, &status);

   EndAPICall(&status);

   /* Print results */
   DriverPrintResultHeader("ARSetSessionConfiguration");
   PrintReturnCode(result);
   PrintARStatusList(&status);

   /* Cleanup */
   FreeARValueStruct(&variableValue, FALSE);
   FreeARStatusList(&status, FALSE);
}
