/* File:  get.c */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>

#include "ar.h"
#include "get.h"
#include "globals.h"
#include "util.h"
#include "print.h"
#ifdef _WIN32
#include <winnt.h>
#include <winbase.h>
#endif /* _WIN32 */

/*****************************************************************************/
/*                                                                           */
/*                                GetARBoolean                               */
/*                                                                           */
/*****************************************************************************/

ARBoolean GetARBoolean(prompt, defaultVal)
char      *prompt;            /* IN; prompt to issue for the value */
ARBoolean  defaultVal;        /* IN; default if no value */

{
   ThreadControlBlock *threadControlBlockPtr;

   threadControlBlockPtr = (ThreadControlBlock *) GetThreadControlBlockPtr();

   DriverPrintPrompt("%s", prompt);
   GetInputLine();
   if (strlen(threadControlBlockPtr->buffer) == 0)
      return defaultVal;        /* no value so use default */

   if ((strcmp(threadControlBlockPtr->buffer, "T") == 0) ||
       (strcmp(threadControlBlockPtr->buffer, "t") == 0))
      return TRUE;
   else /* if not true, assume FALSE */
      return FALSE;
}


/*****************************************************************************/
/*                                                                           */
/*                                   GetInt                                  */
/*                                                                           */
/*****************************************************************************/

int GetInt(prompt, defaultVal)
char      *prompt;            /* IN; prompt to issue for the value */
int        defaultVal;        /* IN; default if no value */

{
   ThreadControlBlock *threadControlBlockPtr;

   threadControlBlockPtr = (ThreadControlBlock *) GetThreadControlBlockPtr();

   DriverPrintPrompt("%s", prompt);
   GetInputLine();
   if (strlen(threadControlBlockPtr->buffer) == 0)
      return defaultVal;        /* no value so use default */

   return atoi(threadControlBlockPtr->buffer);
}          


/*****************************************************************************/
/*                                                                           */
/*                                   GetLong                                 */
/*                                                                           */
/*****************************************************************************/

long GetLong(prompt, defaultVal)
char      *prompt;            /* IN; prompt to issue for the value */
long       defaultVal;        /* IN; default if no value */

{
   ThreadControlBlock *threadControlBlockPtr;

   threadControlBlockPtr = (ThreadControlBlock *) GetThreadControlBlockPtr();

   DriverPrintPrompt("%s", prompt);
   GetInputLine();
   if (strlen(threadControlBlockPtr->buffer) == 0)
      return defaultVal;        /* no value so use default */

   return atol(threadControlBlockPtr->buffer);
}          


/*****************************************************************************/
/*                                                                           */
/*                                   GetReal                                 */
/*                                                                           */
/*****************************************************************************/

double GetReal(prompt, defaultVal)
char      *prompt;            /* IN; prompt to issue for the value */
double     defaultVal;        /* IN; default if no value */

{
   ThreadControlBlock *threadControlBlockPtr;

   threadControlBlockPtr = (ThreadControlBlock *) GetThreadControlBlockPtr();

   DriverPrintPrompt("%s", prompt);
   GetInputLine();
   if (strlen(threadControlBlockPtr->buffer) == 0)
      return defaultVal;        /* no value so use default */

   return atof(threadControlBlockPtr->buffer);
}       


/*****************************************************************************/
/*                                                                           */
/*                                   GetChar                                 */
/*                                                                           */
/*****************************************************************************/

char *GetChar(prompt, defaultVal)
char      *prompt;            /* IN; prompt to issue for the value */
char      *defaultVal;        /* IN; default if no value */

{
   ThreadControlBlock *threadControlBlockPtr;

   threadControlBlockPtr = (ThreadControlBlock *) GetThreadControlBlockPtr();

   DriverPrintPrompt("%s", prompt);
   GetInputLine();
   if (strlen(threadControlBlockPtr->buffer) == 0)
      return defaultVal;        /* no value so use default */

                                /* translate keywords */
   if (strcmp(threadControlBlockPtr->buffer, "$USER$") == 0)
      strcpy(threadControlBlockPtr->buffer, GetControlStructPtr()->user);
   else if (strcmp(threadControlBlockPtr->buffer, "$SERVER$") == 0)
      strcpy(threadControlBlockPtr->buffer, GetControlStructPtr()->server);
   else if (strcmp(threadControlBlockPtr->buffer, "$FIRST_LIST_ID$") == 0)
      strcpy(threadControlBlockPtr->buffer, threadControlBlockPtr->firstListId);
   else if (strcmp(threadControlBlockPtr->buffer, "$SECOND_LIST_ID$") == 0)
      strcpy(threadControlBlockPtr->buffer, threadControlBlockPtr->secondListId);
   else if (strcmp(threadControlBlockPtr->buffer, "$LAST_LIST_ID$") == 0)
      strcpy(threadControlBlockPtr->buffer, threadControlBlockPtr->lastListId);

   return threadControlBlockPtr->buffer;
}          


/*****************************************************************************/
/*                                                                           */
/*                                GetARTimestamp                             */
/*                                                                           */
/*****************************************************************************/

ARTimestamp GetARTimestamp(prompt, defaultVal)
char       *prompt;           /* IN; prompt to issue for the value */
ARTimestamp defaultVal;       /* IN; default if no value */

{
   ThreadControlBlock *threadControlBlockPtr;

   threadControlBlockPtr = (ThreadControlBlock *) GetThreadControlBlockPtr();

   DriverPrintPrompt("%s", prompt);
   GetInputLine();
   if (strlen(threadControlBlockPtr->buffer) == 0)
      return defaultVal;        /* no value so use default */

   return atol(threadControlBlockPtr->buffer);
}          


/*****************************************************************************/
/*                                                                           */
/*                                GetMallocChar                              */
/*                                                                           */
/*****************************************************************************/

void GetMallocChar(prompt, defaultVal, value)
char      *prompt;            /* IN; prompt to issue for the value */
char      *defaultVal;        /* IN; default if no value */
char     **value;             /* IN; pointer to the loaded character string */

{
   char  *tempPtr;            /* working pointer */

   tempPtr = GetChar(prompt, defaultVal);
   if (tempPtr == NULL)
      *value = NULL;
   else
   {
      *value = (char *) malloc(strlen(tempPtr) + 1);
      if (*value != NULL)
         (void) strcpy(*value, tempPtr);
   }
}


/*****************************************************************************/
/*                                                                           */
/*                               GetAREntryIdList                            */
/*                                                                           */
/*****************************************************************************/

void GetAREntryIdList(value)
AREntryIdList *value;         /* OUT; loaded with value for buffer */

{
   int            i;          /* working index */
   AREntryIdType *tempPtr;    /* working pointer */

   value->numItems = GetInt("   Number of entry ids (0): ", 0);
   if (value->numItems == 0)
      value->entryIdList = NULL;
   else
   {
      value->entryIdList = (AREntryIdType *)
                                malloc(sizeof(AREntryIdType) * value->numItems);
      if (value->entryIdList != NULL)
      {
         tempPtr = value->entryIdList;
         for (i = 0; i < (int) value->numItems; i++)
         {
            (void) strcpy(*tempPtr, GetChar("   Entry Id (): ", ""));
            tempPtr++;
         }
      }
   }
}


/*****************************************************************************/
/*                                                                           */
/*                              GetARInternalIdList                          */
/*                                                                           */
/*****************************************************************************/

void GetARInternalIdList(value)
ARInternalIdList *value;      /* OUT; loaded with value for buffer */

{
   int           i;           /* working index */
   ARInternalId *tempPtr;     /* working pointer */

   value->numItems = GetInt("   Number of ids (0): ", 0);
   if (value->numItems == 0)
      value->internalIdList = NULL;
   else
   {
      value->internalIdList = (ARInternalId *)
                                 malloc(sizeof(ARInternalId) * value->numItems);
      if (value->internalIdList != NULL)
      {
         tempPtr = value->internalIdList;
         for (i = 0; i < (int) value->numItems; i++)
         {
            *tempPtr = GetLong("   Id (1): ", 1L);
            tempPtr++;
         }
      }
   }
}


/*****************************************************************************/
/*                                                                           */
/*                                 GetARNameList                             */
/*                                                                           */
/*****************************************************************************/

void GetARNameList(value)
ARNameList *value;            /* OUT; loaded with value for buffer */

{
   int           i;           /* working index */
   ARNameType   *tempPtr;     /* working pointer */

   value->numItems = GetInt("   Number of names (0): ", 0);
   if (value->numItems == 0)
      value->nameList = NULL;
   else
   {
      value->nameList = (ARNameType *)
                                   malloc(sizeof(ARNameType) * value->numItems);
      if (value->nameList != NULL)
      {
         tempPtr = value->nameList;
         for (i = 0; i < (int) value->numItems; i++)
         {
            (void) strcpy(*tempPtr, GetChar("   Name (): ", ""));
            tempPtr++;
         }
      }
   }
}

/*****************************************************************************/
/*                                                                           */
/*                               GetARControlStruct                          */
/*                                                                           */
/*****************************************************************************/

void GetARControlStruct()

{
   ThreadControlBlock *threadControlBlockPtr;

   threadControlBlockPtr = (ThreadControlBlock *) GetThreadControlBlockPtr();

   DriverPrintPrompt("Control record:\n");

   (void) sprintf(threadControlBlockPtr->buffer, "   User name  (%s) : ",
                  threadControlBlockPtr->control.user);
   (void) strcpy(threadControlBlockPtr->control.user,
                 GetChar(threadControlBlockPtr->buffer,
                         threadControlBlockPtr->control.user));

   /* always override password */
   threadControlBlockPtr->control.password[0] = '\0'; 
   (void) sprintf(threadControlBlockPtr->buffer, "   Password (%s) : ",
                  threadControlBlockPtr->control.password);
   (void) strcpy(threadControlBlockPtr->control.password,
                 GetChar(threadControlBlockPtr->buffer,
                 threadControlBlockPtr->control.password));

   (void) sprintf(threadControlBlockPtr->buffer, "   Language (%s) : ",
                  threadControlBlockPtr->control.language);
   (void) strcpy(threadControlBlockPtr->control.language,
                 GetChar(threadControlBlockPtr->buffer,
                         threadControlBlockPtr->control.language));

   (void) sprintf(threadControlBlockPtr->buffer, "   Server (%s) : ", 
                  threadControlBlockPtr->control.server);
   (void) strcpy(threadControlBlockPtr->control.server, 
                 GetChar(threadControlBlockPtr->buffer, 
                         threadControlBlockPtr->control.server));
}


/*****************************************************************************/
/*                                                                           */
/*                            GetARFilterStatusStruct                        */
/*                                                                           */
/*****************************************************************************/

void GetARFilterStatusStruct(value)
ARFilterStatusStruct *value;        /* OUT; loaded with value for buffer */

{
   DriverPrintPrompt("Status Struct:\n");
   value->messageType = GetInt("   Note, Warning, or Error (0, 1, 2) (2) : ",2);
   value->messageNum = GetLong("   Message number (0): ", 0L);
   GetMallocChar("   Message text (): ", "", &value->messageText);
}


/*****************************************************************************/
/*                                                                           */
/*                              GetARMessageStruct                           */
/*                                                                           */
/*****************************************************************************/

void GetARMessageStruct(value)
ARMessageStruct *value;        /* OUT; loaded with value for buffer */

{
   DriverPrintPrompt("Status Struct:\n");
   value->messageType = GetInt("   Note, Warning, or Error (0, 1, 2) (2) : ",2);
   value->messageNum = GetLong("   Message number (0): ", 0L);
   GetMallocChar("   Message text (): ", "", &value->messageText);
   value->usePromptingPane = GetARBoolean("   Use Prompting Pane? (T): ", TRUE);
}

/*****************************************************************************/
/*                                                                           */
/*                                 GetARCoordList                            */
/*                                                                           */
/*****************************************************************************/

void GetARCoordList(value)
ARCoordList **value;

{
   int            i;
   ARCoordStruct *tempPtr;

   /* Allocate memory for ARCoordList */
   *value = (ARCoordList *) malloc(sizeof(ARCoordList));
   if (*value != NULL)
   {
      (*value)->numItems = GetInt("Number of Items in the List (0): ", 0L);
      if ((*value)->numItems == 0)
         (*value)->coords = NULL;
      else
      {
         (*value)->coords = (ARCoordStruct *) malloc(sizeof(ARCoordStruct) *
                                                     (*value)->numItems);
         if ((*value)->coords != NULL)
         {
            tempPtr = (*value)->coords;
            for (i = 0; i < (int) (*value)->numItems; i++)
            {
               tempPtr->x = GetLong("x (0): ",  0L);
               tempPtr->y = GetLong("y (0): ",  0L);
               tempPtr++;
            }
         }
      }
   }
}


/*****************************************************************************/
/*                                                                           */
/*                                 GetARByteList                             */
/*                                                                           */
/*****************************************************************************/

void GetARByteList(value)
ARByteList **value;

{
   int             i;
   unsigned char  *tempPtr;

   /* Allocate memory for ARByteList struct */
   *value = (ARByteList *) malloc(sizeof(ARByteList));
   if (*value != NULL)
   {
      (*value)->type = GetLong("Byte List Type (0): ", 0L);
      (*value)->numItems = GetInt("Number of Items in the List (0): ", 0L);
      if ((*value)->numItems == 0)
         (*value)->bytes = NULL;
      else
      {
         (*value)->bytes = (unsigned char *) malloc((unsigned)
                                  (sizeof(unsigned char) * (*value)->numItems));
         if ((*value)->bytes != NULL)
         {
            tempPtr = (*value)->bytes;
            for (i = 0; i < (int) (*value)->numItems; i++)
            {
               tempPtr = (unsigned char *) GetChar("Byte (0): ",  "");
               tempPtr++;
            }
         }
      }
   }
}


/*****************************************************************************/
/*                                                                           */
/*                                 GetARLocStructForGet                      */
/*                                                                           */
/*****************************************************************************/

void GetARLocStructForGet(value)
ARLocStruct *value;
{
   GetMallocChar("Enter filename: ", "", &value->u.filename);
   /* indicate we're use filename for transfer */
   value->locType = AR_LOC_FILENAME;
}

/*****************************************************************************/
/*                                                                           */
/*                                 GetARAttach                               */
/*                                                                           */
/*****************************************************************************/

void GetARAttach(value)
ARAttachStruct **value;
{
   FILE        *fd;
   struct stat statBuf;
   int         choice;
   char        prompt[255];

   /* Allocate memory for ARAttach struct */
   *value = (ARAttachStruct *) calloc(1, sizeof(ARAttachStruct));
   if (*value != NULL)
   {
      /* get name */
      GetMallocChar("Attachment Name: ", "", &(*value)->name);

      /* determine how user will supply content: via fname or prompted */
      sprintf(prompt, "%s%s%s%s",
         "Choose:\n",
         "   1 - prompt for filename,\n",
         "   2 - prompt for string\n",
         "   default (1): ");

      choice = GetInt(prompt, 1);

      if ((choice < 1) || (choice > 4))
      {
         DriverPrintWarning("   ERROR: choice outside range 1-4; defaulting to prompt for filename(1)\n");
         choice = 1;
      }

      switch(choice)
      {
         case 1:   /* load file */
            for(;;)
            {
               GetMallocChar("Enter filename: ", "", &(*value)->loc.u.filename);
               (*value)->loc.locType = AR_LOC_FILENAME;

               if ((fd = fopen((*value)->loc.u.filename, "rb")) != NULL)
               {
                  fclose(fd);
                  stat((*value)->loc.u.filename, &statBuf);
                  (*value)->origSize = (unsigned long) statBuf.st_size;
                     
                     /* API layer will override the compression value with */
                     /* the real value, if any.                            */
                  (*value)->compSize = (*value)->origSize;
                  break;
               }
               else
               {
                  if ((*value)->loc.u.filename)
                  {
                     (void) free((*value)->loc.u.filename);
                  }
                  DriverPrintError("  ERROR: failed to open file\n");
               }
            }
            break;
         case 2: /* prompt for string */
            /* indicate we're using buffer to transfer string */
            (*value)->loc.locType = AR_LOC_BUFFER;

            GetMallocChar("Enter string: ", "", &(*value)->loc.u.buf.buffer);

            (*value)->origSize =
               (long)strlen((char *)(*value)->loc.u.buf.buffer) + 1;
               /* API layer will override the compression value with */
               /* the real value, if any.                            */
            (*value)->compSize = (*value)->origSize;

            (*value)->loc.u.buf.bufSize = (*value)->origSize;
            break;
         default:
            (*value)->origSize = 0L;
            (*value)->compSize = 0L;
            (*value)->loc.locType = AR_LOC_NULL;
            (*value)->loc.u.buf.bufSize = 0L;
            (*value)->loc.u.buf.buffer = NULL;
      }
   }
}



/*****************************************************************************/
/*                                                                           */
/*                                 GetValue                                  */
/*                                                                           */
/*****************************************************************************/

void GetValue(value)
ARValueStruct *value;         /* IN; contains data type for value */

{
   switch (value->dataType)
   {
      case AR_DATA_TYPE_NULL    :
         /* no value needed as is NULL */
         break;
      case AR_DATA_TYPE_KEYWORD :
         value->u.keyNum = GetInt("Keyword Code (0): ", 0);
         break;
      case AR_DATA_TYPE_INTEGER :
         value->u.intVal = GetLong("Integer Value (0): ", 0L);
         break;
      case AR_DATA_TYPE_REAL    :
         value->u.realVal = GetReal("Real Value (0): ", (double) 0);
         break;
      case AR_DATA_TYPE_CHAR    :
         GetMallocChar("Char Value (): ", "", &value->u.charVal);
         break;
      case AR_DATA_TYPE_DIARY   :
         GetMallocChar("Diary Value (): ", "", &value->u.diaryVal);
         break;
      case AR_DATA_TYPE_ENUM    :
         value->u.enumVal = GetLong("Enumerated Value (0): ", 0L);
         break;
      case AR_DATA_TYPE_TIME    :
         value->u.timeVal = GetARTimestamp("Time Value (0): ", (ARTimestamp) 0);
         break;
      case AR_DATA_TYPE_BITMASK :
         value->u.maskVal = GetLong("Bitmask Value (0): ", 0L);
         break;
      case AR_DATA_TYPE_BYTES   :
         GetARByteList(&value->u.byteListVal);
         break;
      case AR_DATA_TYPE_JOIN    :
      case AR_DATA_TYPE_TRIM    :
      case AR_DATA_TYPE_CONTROL :
      case AR_DATA_TYPE_PAGE    :
      case AR_DATA_TYPE_PAGE_HOLDER :
         break;
      case AR_DATA_TYPE_ULONG   :
         value->u.ulongVal = GetLong("Ulong Value (0): ", 0L);
         break;
      case AR_DATA_TYPE_COORDS  :
         GetARCoordList(&value->u.coordListVal);
         break;
      case AR_DATA_TYPE_ATTACH  :
         GetARAttach(&value->u.attachVal);
         break;
      case AR_DATA_TYPE_DECIMAL :
         GetMallocChar("Decimal Value (): ", "", &value->u.decimalVal);
         break;
      default                   :
         break;
   }
}


/*****************************************************************************/
/*                                                                           */
/*                               GetARValueStruct                            */
/*                                                                           */
/*****************************************************************************/

void GetARValueStruct(value)
ARValueStruct *value;         /* OUT; loaded with value for buffer */

{
   value->dataType = GetInt(
"Datatype Null/Key/Int/Real/Char/Diary/Enum/Time/Bitmask/Byte/\n         Decimal/attach/join/trim/control/ulong/coords\n         (0 - 11, 30-32, 40-41) (0): ",                            0);
   GetValue(value);
}


/*****************************************************************************/
/*                                                                           */
/*                                GetARValueList                             */
/*                                                                           */
/*****************************************************************************/

void GetARValueList(value)
ARValueList   *value;         /* OUT; loaded with value for buffer */

{
   int            i;          /* working index */
   ARValueStruct *tempPtr;    /* working pointer */

   value->numItems = GetInt("   Number of values (0): ", 0);
   if (value->numItems == 0)
      value->valueList = NULL;
   else
   {
      value->valueList = (ARValueStruct *)
                                malloc(sizeof(ARValueStruct) * value->numItems);
      if (value->valueList != NULL)
      {
         tempPtr = value->valueList;
         for (i = 0; i < (int) value->numItems; i++)
         {
            GetARValueStruct(tempPtr);
            tempPtr++;
         }
      }
   }
}


/*****************************************************************************/
/*                                                                           */
/*                           GetAREntryListFieldStruct                       */
/*                                                                           */
/*****************************************************************************/

void GetAREntryListFieldStruct(value)
AREntryListFieldStruct *value;   /* OUT; loaded with value for buffer */

{
   DriverPrintPrompt("   Entry list field entry\n");
   value->fieldId = GetLong("      Field id (0): ", 0L);
   value->columnWidth = GetInt("      Column width (20): ", 20);
   (void) strcpy(value->separator, GetChar("      Separator (  ): ", "  "));
}


/*****************************************************************************/
/*                                                                           */
/*                           GetAREntryListFieldList                         */
/*                                                                           */
/*****************************************************************************/

void GetAREntryListFieldList(value)
AREntryListFieldList *value;  /* OUT; loaded with value for buffer */

{
   int                     i;      /* working index */
   AREntryListFieldStruct *tempPtr;/* working pointer */

   value->numItems = GetInt("Number of query list fields (0): ", 0);
   if (value->numItems == 0)
      value->fieldsList = NULL;
   else
   {
      value->fieldsList = (AREntryListFieldStruct *)
                       malloc(sizeof(AREntryListFieldStruct) * value->numItems);
      if (value->fieldsList != NULL)
      {
         tempPtr = value->fieldsList;
         for (i = 0; i < (int) value->numItems; i++)
         {
            GetAREntryListFieldStruct(tempPtr);
            tempPtr++;
         }
      }
   }
}


/*****************************************************************************/
/*                                                                           */
/*                             GetARFieldValueStruct                         */
/*                                                                           */
/*****************************************************************************/

void GetARFieldValueStruct(value)
ARFieldValueStruct *value;    /* OUT; loaded with value for buffer */

{
   value->fieldId = GetLong("Field id: ", 0L);
   GetARValueStruct(&value->value);
}


/*****************************************************************************/
/*                                                                           */
/*                              GetARFieldValueList                          */
/*                                                                           */
/*****************************************************************************/

void GetARFieldValueList(value)
ARFieldValueList *value;      /* OUT; loaded with value for buffer */

{
   int                 i;      /* working index */
   ARFieldValueStruct *tempPtr;/* working pointer */

   value->numItems = GetInt("   Number of field/value pairs (0): ", 0);
   if (value->numItems == 0)
      value->fieldValueList = NULL;
   else
   {
      value->fieldValueList = (ARFieldValueStruct *)
                           malloc(sizeof(ARFieldValueStruct) * value->numItems);
      if (value->fieldValueList != NULL)
      {
         tempPtr = value->fieldValueList;
         for (i = 0; i < (int) value->numItems; i++)
         {
            GetARFieldValueStruct(tempPtr);
            tempPtr++;
         }
      }
   }
}


/*****************************************************************************/
/*                                                                           */
/*                         GetARFieldValueOrArithStruct                      */
/*                                                                           */
/*****************************************************************************/

void GetARFieldValueOrArithStruct(value)
ARFieldValueOrArithStruct *value;/* OUT; loaded with value for buffer */

{
   value->tag = GetInt(
"Fld,Value,Arith,Stat-Hist,Val-Set,Fld-tran,Fld-DB,L-Var,Query,Current\n  (1-5,50-53,99): ",
                       0);
   switch (value->tag)
   {
      case AR_FIELD        :
      case AR_FIELD_TRAN   :
      case AR_FIELD_DB     :
      case AR_FIELD_CURRENT:
         value->u.fieldId = GetLong("Field id (0): ", 0L);
         break;
      case AR_VALUE        :
         GetARValueStruct(&value->u.value);
         break;
      case AR_ARITHMETIC   :
         value->u.arithOp = (ARArithOpStruct *) malloc(sizeof(ARArithOpStruct));
         if (value->u.arithOp != NULL)
            GetARArithOpStruct((ARArithOpStruct *) value->u.arithOp);
         break;
      case AR_STAT_HISTORY :
         value->u.statHistory.enumVal = (unsigned long)
                                    GetLong("Enumerated value index (0): ", 0L);
         value->u.statHistory.userOrTime =
                                 GetInt("User or Time field (1 or 2) (2): ", 2);
         break;
      case AR_VALUE_SET    :
         GetARValueList(&value->u.valueSet);
         break;
      case AR_LOCAL_VARIABLE :
         value->u.variable = GetInt("Local Variable number (0 to 10) (0): ", 0);
         break;
      case AR_QUERY        :
         value->u.queryValue = (ARQueryValueStruct *)
                                             malloc(sizeof(ARQueryValueStruct));
         if (value->u.queryValue != NULL)
         {
            (void) strcpy(value->u.queryValue->schema, GetChar("Schema: ", ""));
            (void) strcpy(value->u.queryValue->server, GetChar("Server: ", ""));
            GetARQualifierStruct(value->u.queryValue->qualifier);
            value->u.queryValue->valueField = GetLong("Field id (0): ", 0L);
            value->u.queryValue->multiMatchCode =
              GetInt("Multi-Match code (error, first, set) (1, 2, 3) (1): ", 1);
         }
         break;
      default              :
         break;
   }
}


/*****************************************************************************/
/*                                                                           */
/*                              GetARArithOpStruct                           */
/*                                                                           */
/*****************************************************************************/

void GetARArithOpStruct(value)
ARArithOpStruct *value;       /* OUT; loaded with value for buffer */

{
   value->operation =
            GetInt("Arithmetic op code -- +, -, *, /, %, unary - (1 - 6): ", 0);
   if (value->operation != AR_ARITH_OP_NEGATE)
   {
      DriverPrintPrompt("Left operand - ");
      GetARFieldValueOrArithStruct(&value->operandLeft);
   }
   DriverPrintPrompt("Right operand - ");
   GetARFieldValueOrArithStruct(&value->operandRight);
}


/*****************************************************************************/
/*                                                                           */
/*                               GetARRelOpStruct                            */
/*                                                                           */
/*****************************************************************************/

void GetARRelOpStruct(value)
ARRelOpStruct *value;         /* OUT; loaded with value for buffer */

{
   value->operation =
   GetInt("Relational op code - EQ, GT, GE, LT, LE, NE, LIKE, IN (1 - 8): ", 0);
   DriverPrintPrompt("Left operand - ");
   GetARFieldValueOrArithStruct(&value->operandLeft);
   DriverPrintPrompt("Right operand - ");
   GetARFieldValueOrArithStruct(&value->operandRight);
}


/*****************************************************************************/
/*                                                                           */
/*                              GetARQualifierStruct                         */
/*                                                                           */
/*****************************************************************************/

void GetARQualifierStruct(value)
ARQualifierStruct *value;     /* OUT; loaded with value for buffer */

{
   DriverPrintPrompt("Qualifier Struct:\n");

   /* initialize the structure since it does not get completely */
   /* filled in all cases below                                 */
   memset((char *) value, '\0', sizeof(ARQualifierStruct));

   value->operation =
                GetInt("None, And, Or, Not, or Relop (0, 1, 2, 3, 4) (0): ", 0);
   switch (value->operation)
   {
      case AR_COND_OP_NONE   :
         /* nothing to load if no qualifier */
         break;
      case AR_COND_OP_AND    :
      case AR_COND_OP_OR     :
         value->u.andor.operandLeft =
                 (struct ARQualifierStruct *) malloc(sizeof(ARQualifierStruct));
         if (value->u.andor.operandLeft)
            GetARQualifierStruct(value->u.andor.operandLeft);
         if (value->operation == AR_COND_OP_AND)
            DriverPrintPrompt("AND\n");
         else
            DriverPrintPrompt("OR\n");
         value->u.andor.operandRight =
                 (struct ARQualifierStruct *) malloc(sizeof(ARQualifierStruct));
         if (value->u.andor.operandRight)
            GetARQualifierStruct(value->u.andor.operandRight);
         break;
      case AR_COND_OP_NOT    :
         value->u.not =
                 (struct ARQualifierStruct *) malloc(sizeof(ARQualifierStruct));
         if (value->u.not != NULL)
            GetARQualifierStruct(value->u.not);
         break;
      case AR_COND_OP_REL_OP :
         value->u.relOp = (ARRelOpStruct *) malloc(sizeof(ARRelOpStruct));
         if (value->u.relOp != NULL)
            GetARRelOpStruct(value->u.relOp);
         break;
      default                :
         break;
   }
}


/*****************************************************************************/
/*                                                                           */
/*                                 GetARSortStruct                           */
/*                                                                           */
/*****************************************************************************/

void GetARSortStruct(value)
ARSortStruct *value;          /* OUT; loaded with value for buffer */

{
   DriverPrintPrompt("   Sort list entry\n");
   value->fieldId = GetLong("      Internal id of field to sort on (0): ", 0L);
   value->sortOrder = GetInt("      Ascending or Descending (1 or 2) (1): ", 1);
}


/*****************************************************************************/
/*                                                                           */
/*                                 GetARSortList                             */
/*                                                                           */
/*****************************************************************************/

void GetARSortList(value)
ARSortList *value;            /* OUT; loaded with value for buffer */

{
   int           i;            /* working index */
   ARSortStruct *tempPtr;      /* working pointer */

   value->numItems = GetInt("Number of sort fields (0): ", 0);
   if (value->numItems == 0)
      value->sortList = NULL;
   else
   {
      value->sortList = (ARSortStruct *)
                                 malloc(sizeof(ARSortStruct) * value->numItems);
      if (value->sortList != NULL)
      {
         tempPtr = value->sortList;
         for (i = 0; i < (int) value->numItems; i++)
         {
            GetARSortStruct(tempPtr);
            tempPtr++;
         }
      }
   }
}


/*****************************************************************************/
/*                                                                           */
/*                                GetARIndexStruct                           */
/*                                                                           */
/*****************************************************************************/

void GetARIndexStruct(value)
ARIndexStruct *value;         /* OUT; loaded with value for buffer */

{
   int   i;                   /* working index */

   DriverPrintPrompt("   Index list entry\n");
   value->numFields = GetInt("      Number of fields in index (1): ", 1);
   for (i = 0; i < (int) value->numFields; i++)
      value->fieldIds[i] =
                    GetLong("      Internal id of field to index on (0): ", 0L);
   value->unique = GetARBoolean("      Unique index (F): ", FALSE);
}


/*****************************************************************************/
/*                                                                           */
/*                                GetARIndexList                             */
/*                                                                           */
/*****************************************************************************/

void GetARIndexList(value)
ARIndexList *value;           /* OUT; loaded with value for buffer */

{
   int            i;            /* working index */
   ARIndexStruct *tempPtr;      /* working pointer */

   value->numItems = GetInt("Number of indexes (0): ", 0);
   if (value->numItems == 0)
      value->indexList = NULL;
   else
   {
      value->indexList = (ARIndexStruct *)
                                malloc(sizeof(ARIndexStruct) * value->numItems);
      if (value->indexList != NULL)
      {
         tempPtr = value->indexList;
         for (i = 0; i < (int) value->numItems; i++)
         {
            GetARIndexStruct(tempPtr);
            tempPtr++;
         }
      }
   }
}

/*****************************************************************************/
/*                                                                           */
/*                                GetARCompoundSchema                        */
/*                                                                           */
/*****************************************************************************/

void GetARCompoundSchema(value)
ARCompoundSchema *value;         /* OUT; loaded with value for buffer */

{
   ThreadControlBlock *threadControlBlockPtr;

   threadControlBlockPtr = (ThreadControlBlock *) GetThreadControlBlockPtr();

   value->schemaType = GetInt("Schema Type (1): ", 1);
   if (value->schemaType > AR_SCHEMA_DIALOG || 
       value->schemaType < AR_SCHEMA_REGULAR)
      value->schemaType = AR_SCHEMA_REGULAR;
   switch (value->schemaType) {
      case AR_SCHEMA_REGULAR :
      case AR_SCHEMA_DIALOG :
         break;
      case AR_SCHEMA_JOIN :
         (void) strcpy(value->u.join.memberA, 
                       GetChar("   Join member A name : ", ""));
         (void) strcpy(value->u.join.memberB, 
                       GetChar("   Join member B name : ", ""));
         GetARQualifierStruct(&value->u.join.joinQual);
         value->u.join.option = GetInt("   Join option(0): ", 0);
         break;
      case AR_SCHEMA_VIEW :
         (void) strcpy(value->u.view.tableName,
                       GetChar("   View table name : ", ""));
         GetChar("   View qualification : ", "");
         value->u.view.viewQual = 
          malloc(strlen(threadControlBlockPtr->buffer) + 1);
         (void) strcpy(value->u.view.viewQual, threadControlBlockPtr->buffer);
         break;
   }
   
}

/*****************************************************************************/
/*                                                                           */
/*                                GetARFieldMapping                          */
/*                                                                           */
/*****************************************************************************/

void GetARFieldMapping(value)
ARFieldMappingStruct *value;         /* OUT; loaded with value for buffer */

{
   value->fieldType = GetInt("Field Type (1): ", 1);
   if (value->fieldType > AR_FIELD_VIEW || 
       value->fieldType < AR_FIELD_NONE)
      value->fieldType = AR_FIELD_REGULAR;
   switch (value->fieldType) {
      case AR_FIELD_REGULAR :
      case AR_FIELD_NONE:  
         break;
      case AR_FIELD_JOIN :
         value->u.join.schemaIndex = GetInt("   schema index (0): ", 0);
         if (value->u.join.schemaIndex > AR_FIELD_MAPPING_SECONDARY ||
             value->u.join.schemaIndex < AR_FIELD_MAPPING_PRIMARY)
            value->u.join.schemaIndex = AR_FIELD_MAPPING_PRIMARY;
         value->u.join.realId = GetLong("   real Id : ", "");
         break;
      case AR_FIELD_VIEW :
         (void) strcpy(value->u.view.fieldName, 
                       GetChar("   field name : ", ""));
         break;
   }
}


/*****************************************************************************/
/*                                                                           */
/*                               GetARPropList                               */
/*                                                                           */
/*****************************************************************************/

void GetARPropList(value)
ARPropList *value;         /* OUT; loaded with value for buffer */

{
   int              i;
   ARPropStruct    *tempPtr;

   value->numItems = GetInt("Number of Properties (0): ", 0);
   if (value->numItems == 0)
      value->props = NULL;
   else
   {
      value->props = (ARPropStruct *) malloc((unsigned)
                                      (sizeof(ARPropStruct) * value->numItems));
      if (value->props != NULL)
      {
         tempPtr = value->props;
         for (i = 0; i < (int) value->numItems; i++)
         {
            tempPtr->prop = GetLong("Prop Tag - (0-500): ", 0L);
            GetARValueStruct(&tempPtr->value);
            tempPtr++;
         }
      }
   }
}


/*****************************************************************************/
/*                                                                           */
/*                               GetARDisplayInstanceList                    */
/*                                                                           */
/*****************************************************************************/

void GetARDisplayInstanceList(value)
ARDisplayInstanceList *value;         /* OUT; loaded with value for buffer */

{
   int                     i;           /* working index */
   ARDisplayInstanceStruct *tempPtr;    /* working pointer */

   /* Get the common properties */
   DriverPrintPrompt("Getting the Common Properties for this field\n");
   GetARPropList(&value->commonProps);
   value->numItems = GetInt("Number of display Instances (0): ", 0);
   if (value->numItems == 0)
      value->dInstanceList = NULL;
   else
   {
      value->dInstanceList = (ARDisplayInstanceStruct *)
                              malloc(sizeof(ARDisplayInstanceStruct) * value->numItems);
      if (value->dInstanceList != NULL)
      {
         tempPtr = value->dInstanceList;
         for (i = 0; i < (int) value->numItems; i++)
         {
            tempPtr->vui = GetLong("vui ID (0): ",  0L);
            GetARPropList(&tempPtr->props);
            tempPtr++;
         }
      }
   }
}


/*****************************************************************************/
/*                                                                           */
/*                             GetARAssignFieldStruct                        */
/*                                                                           */
/*****************************************************************************/

void GetARAssignFieldStruct(value)
ARAssignFieldStruct *value;   /* OUT; loaded with value for buffer */

{
   (void) strcpy(value->server, GetChar("Server name (): ", ""));
   (void) strcpy(value->schema, GetChar("Schema name (): ", ""));
   value->tag = GetInt("Field or status history (1 or 4) (1): ", 1);
   if (value->tag == AR_FIELD)
      value->u.fieldId = GetLong("Field id (0): ", 0L);
   else if (value->tag == AR_STAT_HISTORY)
   {
         value->u.statHistory.enumVal = (unsigned long)
                                    GetLong("Enumerated value index (0): ", 0L);
         value->u.statHistory.userOrTime =
                                 GetInt("User or Time field (1 or 2) (2): ", 2);
   }
   /*else  no action as not recognized */
   GetARQualifierStruct(&value->qualifier);
   value->noMatchOption = GetInt("No match action -- error, null, no action, submit (1-4) (2): ",
                                 2);
   value->multiMatchOption = GetInt(
        "Multi match action -- error, null, first, picklist, mod all, no action, submit (1-7) (4): ",
        4);
}


/*****************************************************************************/
/*                                                                           */
/*                               GetARDDEStruct                              */
/*                                                                           */
/*****************************************************************************/

void GetARDDEStruct(value)
ARDDEStruct   *value;      /* OUT; loaded with value for buffer */

{
   DriverPrintPrompt("   Active Link DDE Struct:\n");
   GetMallocChar("      Service name (): ", "", &value->serviceName);
   GetMallocChar("      Topic (): ", "", &value->topic);
   value->action = GetInt("      Action (execute, poke, request) (1-3) (1): ",
                          1);
   GetMallocChar("      Path To Program (): ", "", &value->pathToProgram);
   if (value->action != AR_DDE_REQUEST)
      GetMallocChar("      Command (): ", "", &value->command);
   else
      value->command = NULL;
   if (GetARBoolean("      Set an item string? (F): ", FALSE))
      GetMallocChar("      Item text (): ", "", &value->item);
   else
      value->item = NULL;
}


/*****************************************************************************/
/*                                                                           */
/*                               GetARAutomationStruct                       */
/*                                                                           */
/*****************************************************************************/
 
void GetARAutomationStruct(value)
ARAutomationStruct   *value;      /* OUT; loaded with value for buffer */
 
{
   DriverPrintPrompt("   Active Link Automation Struct:\n");
   GetMallocChar("      Auto Server name (): ", "", &value->autoServerName);
   value->isVisible = GetARBoolean("      Showing Automation Server? (T): ", TRUE);
   GetMallocChar("      Class  Id (): ", "", &value->clsId);
   GetMallocChar("      Automation Action (): ", "", &value->action);
 
   GetARCOMMethodList(&value->methodList);
}
 
 
/*****************************************************************************/
/*                                                                           */
/*                               GetARCOMValueStruct                         */
/*                                                                           */
/*****************************************************************************/
 
void GetARCOMValueStruct(value)
ARCOMValueStruct   *value;      /* OUT; loaded with value for buffer */
 
{
   value->transId = GetLong("      Transient Id (1): ", 1L);
   GetMallocChar("      Value IId (used for method only)(): ", "", 
                 &value->valueIId);
   value->valueType = GetInt("      COM [Method(0,1)/Parm(0-2)] Type (0): ", 0);
   switch (value->valueType) {
      case AR_COM_PARM_FIELDID: /* AR_COM_METHOD_FIELDID has same type */
         value->u.fieldId = GetLong("      FieldId (1): ", 1L);
         break;
      case AR_COM_PARM_VALUE:
         GetARValueStruct(&value->u.value);
         break;
      default:
         break;
   }
}
 
 
/*****************************************************************************/
/*                                                                           */
/*                               GetARCOMMethodList                          */
/*                                                                           */
/*****************************************************************************/
 
void GetARCOMMethodList(value)
ARCOMMethodList *value;         /* OUT; loaded with value for buffer */
 
{
   int                i;
   ARCOMMethodStruct *tempPtr;
 
   value->numItems = GetInt("      Number of Methods (1): ", 1);
   if (value->numItems == 0)
      value->methodList = NULL;
   else
   {
      value->methodList = (ARCOMMethodStruct *) 
                calloc((unsigned)value->numItems, sizeof(ARCOMMethodStruct));
      if (value->methodList != NULL)
      {
         tempPtr = value->methodList;
         for (i = 0; i < (int) value->numItems; i++)
         {
            GetARCOMMethodStruct(tempPtr, i+1);
            tempPtr++;
         }
      }
   }
}
 
 
/*****************************************************************************/
/*                                                                           */
/*                               GetARCOMMethodStruct                        */
/*                                                                           */
/*****************************************************************************/
 
void GetARCOMMethodStruct(value, order)
ARCOMMethodStruct   *value;      /* OUT; loaded with value for buffer */
int                  order;   /* IN; order of the parameter in the list */
 
{
   DriverPrintPrompt("      Method Number %d:\n", order);
   GetMallocChar("      Method Name (): ", "", &value->methodName);
   GetMallocChar("      Method Interface Id (): ", "", &value->methodIId);
   value->methodType = GetInt("      COM Method Type (VT_EMPTY/VT_NULL/VT_I2/VT_I4/VT_R4/VT_R8/VT_CY/VT_DATE/VT_BSTR/VT_DISPATCH/VT_ERROR/VT_BOOL/VT_VARIANT/VT_UNKNOWN/VT_WBSTR) (0-14) (VT_I1/VT_UI1/VT_UI2/VT_UI4/VT_I8/VT_UI8/VT_INT/VT_UINT/VT_VOID/VT_HRESULT/VT_PTR) (16-26) (VT_LPSTR/VT_LPWSTR) (30-31) (VT_BLOB_OBJECT -- 70) (0): ", 0);
   GetARCOMValueStruct(&value->methodValue);
   GetARCOMMethodParmList(&value->parameterList);
 
}
 
/*****************************************************************************/
/*                                                                           */
/*                               GetARCOMMethodParmList                      */
/*                                                                           */
/*****************************************************************************/
 
void GetARCOMMethodParmList(value)
ARCOMMethodParmList *value;         /* OUT; loaded with value for buffer */
 
{
   int                    i;
   ARCOMMethodParmStruct *tempPtr;
 
   value->numItems = GetInt("      Number of Parameters (0): ", 0);
   if (value->numItems == 0)
      value->parameterList = NULL;
   else
   {
      value->parameterList = (ARCOMMethodParmStruct *)
            calloc((unsigned)value->numItems, sizeof(ARCOMMethodParmStruct));
      if (value->parameterList != NULL)
      {
         tempPtr = value->parameterList;
         for (i = 0; i < (int) value->numItems; i++)
         {
            GetARCOMMethodParmStruct(tempPtr, i+1);
            tempPtr++;
         }
      }
   }
}
 
 
/*****************************************************************************/
/*                                                                           */
/*                               GetARCOMMethodParmStruct                    */
/*                                                                           */
/*****************************************************************************/
 
void GetARCOMMethodParmStruct(value, order)
ARCOMMethodParmStruct   *value;   /* OUT; loaded with value for buffer */
int                      order;   /* IN; order of the parameter in the list */
 
{
   DriverPrintPrompt("      Parameter Number %d:\n", order);
   GetMallocChar("      Parameter Name (): ", "", &value->parmName);
   value->parmType = GetInt("      COM Parameter Type (VT_EMPTY/VT_NULL/VT_I2/VT_I4/VT_R4/VT_R8/VT_CY/VT_DATE/VT_BSTR/VT_DISPATCH/VT_ERROR/VT_BOOL/VT_VARIANT/VT_UNKNOWN/VT_WBSTR) (0-14) (VT_I1/VT_UI1/VT_UI2/VT_UI4/VT_I8/VT_UI8/VT_INT/VT_UINT/VT_VOID/VT_HRESULT/VT_PTR) (16-26) (VT_LPSTR/VT_LPWSTR) (30-31) (VT_BLOB_OBJECT -- 70) (0): ", 0);
   GetARCOMValueStruct(&value->parmValue);
}


/*****************************************************************************/
/*                                                                           */
/*                               GetAROpenDlgStruct                          */
/*                                                                           */
/*****************************************************************************/
 
void GetAROpenDlgStruct(value)
AROpenDlgStruct   *value;      /* OUT; loaded with value for buffer */
 
{
   DriverPrintPrompt("   Active Link Open Dialog Struct:\n");
   strcpy(value->serverName, GetChar("      Server Name: ", ""));
   strcpy(value->schemaName, GetChar("      Schema Name: ", ""));
   strcpy(value->vuiLabel, GetChar("      Vui Label: ", ""));
   value->closeBox = GetARBoolean("      Close Box? (T): ", TRUE);
   GetARFieldAssignList(&value->inputValueFieldPairs);
   GetARFieldAssignList(&value->outputValueFieldPairs);
}
  
/*****************************************************************************/
/*                                                                           */
/*                               GetARCallGuideStruct                        */
/*                                                                           */
/*****************************************************************************/
  
void GetARCallGuideStruct(value)
ARCallGuideStruct *value;
{
   DriverPrintPrompt("   Active Link Call Guide Struct:\n");
   strcpy(value->serverName, GetChar("      Server Name: " , ""));
   strcpy(value->guideName, GetChar("      Guide Name: ", ""));
   value->guideMode = GetInt("      Guide Mode: (0): ", 0);
}


/*****************************************************************************/
/*                                                                           */
/*                               GetARExitGuideStruct                        */
/*                                                                           */
/*****************************************************************************/
  
void GetARExitGuideStruct(value)
ARExitGuideStruct *value;
{
   DriverPrintPrompt("   Active Link Exit Guide Struct:\n");
   value->closeAll = GetARBoolean("      Close All? (T): ", TRUE);  
}



/*****************************************************************************/
/*                                                                           */
/*                               GetARGotoGuideLabel                         */
/*                                                                           */
/*****************************************************************************/
  
void GetARGotoGuideLabel(value)
ARGotoGuideLabelStruct  *value;
{
   DriverPrintPrompt("   Active Link Goto Guide Label Struct:\n");
   GetMallocChar("      Label (): ", "", &value->label);  
}

/*****************************************************************************/
/*                                                                           */
/*                               GetARWaitStruct                             */
/*                                                                           */
/*****************************************************************************/
  
void GetARWaitStruct(value)
ARWaitStruct *value;
{
   DriverPrintPrompt("   Active Link Wait Struct:\n");
   GetMallocChar("      Continue Button Title (): ", "Continue", &value->continueButtonTitle);
} 
 
 
/*****************************************************************************/
/*                                                                           */
/*                             GetARAssignSQLStruct                          */
/*                                                                           */
/*****************************************************************************/

void GetARAssignSQLStruct(value)
ARAssignSQLStruct *value;     /* OUT; loaded with value for buffer */

{
   (void) strcpy(value->server, GetChar("Server name (): ", ""));
   GetMallocChar("SQL command (): ", "", &value->sqlCommand);
   value->valueIndex = GetInt("Index of returned value to select (1): ", 1);
   value->noMatchOption = GetInt("No match action -- error, null (1, 2) (2): ",
                                 2);
   value->multiMatchOption = GetInt(
        "Multi match action -- error, null, first, picklist (1, 2, 3, 4) (4): ",
        4);
}


/*****************************************************************************/
/*                                                                           */
/*                               GetARAssignStruct                           */
/*                                                                           */
/*****************************************************************************/

void GetARAssignStruct(value)
ARAssignStruct *value;        /* OUT; loaded with value for buffer */

{
   value->assignType =
    GetInt("Assign type  Value, Field, Process, Arith, Function, DDE, SQL(1 - 7): ",
           0);
   switch(value->assignType)
   {
      case AR_ASSIGN_TYPE_VALUE   :
         GetARValueStruct(&value->u.value);
         break;
      case AR_ASSIGN_TYPE_FIELD   :
         value->u.field = (ARAssignFieldStruct *)
                                            malloc(sizeof(ARAssignFieldStruct));
         if (value->u.field != NULL)
            GetARAssignFieldStruct(value->u.field);
         break;
      case AR_ASSIGN_TYPE_PROCESS :
         GetMallocChar("Process to run (): ", "", &value->u.process);
         break;
      case AR_ASSIGN_TYPE_ARITH   :
         value->u.arithOp = (ARArithOpAssignStruct *)
                                          malloc(sizeof(ARArithOpAssignStruct));
         if (value->u.arithOp != NULL)
            GetARArithOpAssignStruct(value->u.arithOp);
         break;
      case AR_ASSIGN_TYPE_FUNCTION:
         value->u.function = (struct ARFunctionAssignStruct *)
                                         malloc(sizeof(ARFunctionAssignStruct));
         if (value->u.function != NULL)
            GetARFunctionAssignStruct(value->u.function);
         break;
      case AR_ASSIGN_TYPE_DDE     :
         value->u.dde = (ARDDEStruct *) malloc(sizeof(ARDDEStruct));
         if (value->u.dde != NULL)
            GetARDDEStruct(value->u.dde);
         break;
      case AR_ASSIGN_TYPE_SQL     :
         value->u.sql = (ARAssignSQLStruct *) malloc(sizeof(ARAssignSQLStruct));
         if (value->u.sql != NULL)
            GetARAssignSQLStruct(value->u.sql);
         break;
      default                     :
         break;
   }
}


/*****************************************************************************/
/*                                                                           */
/*                           GetARArithOpAssignStruct                        */
/*                                                                           */
/*****************************************************************************/

void GetARArithOpAssignStruct(value)
ARArithOpAssignStruct *value; /* OUT; loaded with value for buffer */

{
   value->operation =
            GetInt("Arithmetic op code -- +, -, *, /, %, unary - (1 - 6): ", 0);
   if (value->operation != AR_ARITH_OP_NEGATE)
   {
      DriverPrintPrompt("Left operand - ");
      GetARAssignStruct(&value->operandLeft);
   }
   DriverPrintPrompt("Right operand - ");
   GetARAssignStruct(&value->operandRight);
}


/*****************************************************************************/
/*                                                                           */
/*                           GetARFunctionAssignStruct                       */
/*                                                                           */
/*****************************************************************************/

void GetARFunctionAssignStruct(value)
ARFunctionAssignStruct *value; /* OUT; loaded with value for buffer */

{
   int     i;               /* working index */

   value->functionCode =
            GetInt("Function code -- (1 - 26): ", 0);
   value->numItems = GetInt("Number of parameters (0): ", 0);
   if (value->numItems != 0)
   {
      value->parameterList = (ARAssignStruct *) malloc((unsigned)
                                    (sizeof(ARAssignStruct) * value->numItems));
      if (value->parameterList == NULL)
         return;

      for (i = 0; i < (int)value->numItems; i++)
      {
         DriverPrintPrompt("Parameter[%d] - ", i);
         GetARAssignStruct(&value->parameterList[i]);
      }
   }
}


/*****************************************************************************/
/*                                                                           */
/*                             GetARFieldAssignStruct                        */
/*                                                                           */
/*****************************************************************************/

void GetARFieldAssignStruct(value)
ARFieldAssignStruct *value;   /* OUT; loaded with value for buffer */

{
   value->fieldId = GetLong("Field id: ", 0L);
   GetARAssignStruct(&value->assignment);
}


/*****************************************************************************/
/*                                                                           */
/*                              GetARFieldAssignList                         */
/*                                                                           */
/*****************************************************************************/

void GetARFieldAssignList(value)
ARFieldAssignList *value;     /* OUT; loaded with value for buffer */

{
   int                  i;      /* working index */
   ARFieldAssignStruct *tempPtr;/* working pointer */

   value->numItems = GetInt("   Number of field/assign pairs (0): ", 0);
   if (value->numItems == 0)
      value->fieldAssignList = NULL;
   else
   {
      value->fieldAssignList = (ARFieldAssignStruct *)
                          malloc(sizeof(ARFieldAssignStruct) * value->numItems);
      if (value->fieldAssignList != NULL)
      {
         tempPtr = value->fieldAssignList;
         for (i = 0; i < (int) value->numItems; i++)
         {
            GetARFieldAssignStruct(tempPtr);
            tempPtr++;
         }
      }
   }
}

/*****************************************************************************/
/*                                                                           */
/*                           GetARPushFieldsAssignList                       */
/*                                                                           */
/*****************************************************************************/

void GetARPushFieldsAssignList(value)
ARPushFieldsList *value;     /* OUT; loaded with value for buffer */

{
   int                  i;      /* working index */
   ARPushFieldsStruct   *tempPtr;/* working pointer */

   value->numItems = GetInt("   Number of field/assign pairs (0): ", 0);
   if (value->numItems == 0)
      value->pushFieldsList = NULL;
   else
   {
      value->pushFieldsList = (ARPushFieldsStruct *)
                          malloc(sizeof(ARPushFieldsStruct) * value->numItems);
      if (value->pushFieldsList != NULL)
      {
         tempPtr = value->pushFieldsList;
         for (i = 0; i < (int) value->numItems; i++)
         {
            GetARAssignFieldStruct(&tempPtr->field);
            GetARAssignStruct(&tempPtr->assign);
            tempPtr++;
         }
      }
   }
}

/*****************************************************************************/
/*                                                                           */
/*                           GetARFilterActionNotify                         */
/*                                                                           */
/*****************************************************************************/

void GetARFilterActionNotify(value)
ARFilterActionNotify *value;  /* OUT; loaded with value for buffer */

{
   DriverPrintPrompt("Filter Action Notify:\n");
   value->notifyMechanism = GetInt(
        "   Notifier, Email, Default, other, or XRef (1, 2, 3, 4-98, 99): ", 0);
   if (value->notifyMechanism == AR_NOTIFY_VIA_XREF)
      value->notifyMechanismXRef = GetLong("   Cross-reference field id: ", 0L);
   GetMallocChar("   User name: ", "", &value->user);
   GetMallocChar("   Notify text (): ", "", &value->notifyText);
   value->notifyPriority = GetInt("   Notify priority (1 to 10) (1): ", 1);
   GetMallocChar("   Notify subject (): ", "", &value->subjectText);
   value->fieldIdListType =
        GetInt("   None, All, specific list, or changed (0, 1, 2, 3) (0): ", 0);
   if (value->fieldIdListType > 3)
      value->fieldIdListType = 0;
   if (value->fieldIdListType == AR_FILTER_FIELD_IDS_LIST)
      GetARInternalIdList(&value->fieldIdList);
}

/*****************************************************************************/
/*                                                                           */
/*                                GetARSQLStruct                             */
/*                                                                           */
/*****************************************************************************/

void GetARSQLStruct(value)
ARSQLStruct *value;  /* OUT; loaded with value for buffer */

{
   DriverPrintPrompt("Action Direct SQL:\n");
   (void) strcpy(value->server, GetChar("      Server (): ", ""));
   GetMallocChar("      SQL command (): ", "", &value->command);
}


/*****************************************************************************/
/*                                                                           */
/*                         GetARGotoActionStruct                             */
/*                                                                           */
/*****************************************************************************/

void GetARGotoActionStruct(value)
ARGotoActionStruct *value;  /* OUT; loaded with value for buffer */

{
   DriverPrintPrompt("   Goto Action Struct:\n");
   value->tag = GetInt("      Field, Value, Forward, Backup (1-4): ", 1L);
   if (value->tag == AR_GOTO_FIELD_XREF)
      value->fieldIdOrValue = GetInt("      FieldId: ", 0L);
   if (value->tag == AR_GOTO_ABSOLUTE_ORDER)
      value->fieldIdOrValue = GetInt("      Execution Order: ", 0L);
   if (value->tag == AR_GOTO_OFFSET_FORWARD)
      value->fieldIdOrValue = GetInt("      Execution Order Advance By: ", 1L);
   if (value->tag == AR_GOTO_OFFSET_BACKWARD)
      value->fieldIdOrValue = GetInt("      Execution Order Bacup By: ", 1L);
}


/*****************************************************************************/
/*                                                                           */
/*                           GetARFilterActionStruct                         */
/*                                                                           */
/*****************************************************************************/

void GetARFilterActionStruct(value)
ARFilterActionStruct *value;  /* OUT; loaded with value for buffer */

{
   DriverPrintPrompt("Filter Action Struct:\n");
   value->action= GetInt(
               "   Notify, Message, Log, Fields, Process, Push, SQL, Goto Action (1-8): ", 0);
   switch (value->action)
   {
      case AR_FILTER_ACTION_NOTIFY  :
         GetARFilterActionNotify(&value->u.notify);
         break;
      case AR_FILTER_ACTION_MESSAGE :
         GetARFilterStatusStruct(&value->u.message);
         break;
      case AR_FILTER_ACTION_LOG     :
         GetMallocChar("   Log file name (): ", "", &value->u.logFile);
         break;
      case AR_FILTER_ACTION_FIELDS  :
         GetARFieldAssignList(&value->u.fieldList);
         break;
      case AR_FILTER_ACTION_PROCESS :
         GetMallocChar("   Process (): ", "", &value->u.process);
         break;
      case AR_FILTER_ACTION_FIELDP  :
         GetARPushFieldsAssignList(&value->u.pushFieldsList);
         break;
      case AR_FILTER_ACTION_SQL     :
         GetARSQLStruct(&value->u.sqlCommand);
         break;
      case AR_FILTER_ACTION_GOTOACTION:
         GetARGotoActionStruct(&value->u.gotoAction);
         break;
    
      default                       :
         break;
   }
}


/*****************************************************************************/
/*                                                                           */
/*                           GetARFilterActionList                           */
/*                                                                           */
/*****************************************************************************/

void GetARFilterActionList(actionFlag, value)
ARBoolean           actionFlag;/* IN; TRUE - action; FALSE - else */
ARFilterActionList *value;     /* OUT; loaded with value for buffer */

{
   int                   i;      /* working index */
   ARFilterActionStruct *tempPtr;/* working pointer */

   if (actionFlag)
      value->numItems = GetInt("Number of actions for the filter (1): ", 1);
   else
      value->numItems = GetInt("Number of elses for the filter (0): ", 0);
   if (value->numItems == 0)
      value->actionList = NULL;
   else
   {
      value->actionList = (ARFilterActionStruct *)
                         malloc(sizeof(ARFilterActionStruct) * value->numItems);
      if (value->actionList != NULL)
      {
         tempPtr = value->actionList;
         for (i = 0; i < (int) value->numItems; i++)
         {
            GetARFilterActionStruct(tempPtr);
            tempPtr++;
         }
      }
   }
}


/*****************************************************************************/
/*                                                                           */
/*                          GetARActiveLinkMacroStruct                       */
/*                                                                           */
/*****************************************************************************/

void GetARActiveLinkMacroStruct(value)
ARActiveLinkMacroStruct *value;  /* OUT; loaded with value for buffer */

{
   int   i;                   /* working index */

   DriverPrintPrompt("   Active Link Macro Struct:\n");
   (void) strcpy(value->macroName, GetChar("      Macro name: ", ""));
   GetMallocChar("      Macro text(): ", "", &value->macroText);
   value->macroParms.numItems = GetInt("      Number of macro parms (0): ", 0);
   if (value->macroParms.numItems == 0)
      value->macroParms.parms = NULL;
   else
   {
      value->macroParms.parms = (ARMacroParmStruct *)
                 malloc(sizeof(ARMacroParmStruct) * value->macroParms.numItems);
      if (value->macroParms.parms != NULL)
         for (i = 0; i < (int) value->macroParms.numItems; i++)
         {
            (void) strcpy(value->macroParms.parms[i].name,
                          GetChar("         Parm Name (): ", ""));
            GetMallocChar("         Parm Value (): ", "",
                          &value->macroParms.parms[i].value);
         }
   }
}


/*****************************************************************************/
/*                                                                           */
/*                          GetARFieldCharacteristics                        */
/*                                                                           */
/*****************************************************************************/

void GetARFieldCharacteristics(value)
ARFieldCharacteristics  *value;  /* OUT; loaded with value for buffer */

{
   DriverPrintPrompt("   Active Link Set Field Characteristics:\n");
   value->fieldId = GetLong( "      Field Id (0): ", 0L);
   if (GetARBoolean("      Change menu? (F): ", FALSE))
      GetMallocChar("      Char menu (): ", "", &value->charMenu);
   else
      value->charMenu = NULL;
   if (GetARBoolean("      Change display Properties? (F): ", FALSE))
      GetARPropList(&value->props);
   else
      value->props.numItems = 0;
   if (GetARBoolean("      Change focus to this field? (F): ", FALSE))
      value->focus = AR_FOCUS_SET_TO_FIELD;
   else
      value->focus = AR_FOCUS_UNCHANGED;
   value->accessOption = GetInt(
  "      Access option (unchanged, read-only, read/write, disable) (0-3) (0): ",
                                0);
}


/*****************************************************************************/
/*                                                                           */
/*                          GetARActiveLinkActionStruct                      */
/*                                                                           */
/*****************************************************************************/

void GetARActiveLinkActionStruct(value)
ARActiveLinkActionStruct *value;  /* OUT; loaded with value for buffer */

{
   DriverPrintPrompt("Active Link Action Struct:\n");
   value->action= GetInt(
  "   Macro, Fields, Process, Message, Field Characteristics, DDE, Push, SQL, Automation, Open Dialog, Commit Changes, Close Window, Call Guide, Exit Guide, Goto Guide, Wait, Goto Action(1-17): ", 0);
   switch (value->action)
   {
      case AR_ACTIVE_LINK_ACTION_MACRO   :
         GetARActiveLinkMacroStruct(&value->u.macro);
         break;
      case AR_ACTIVE_LINK_ACTION_FIELDS  :
         GetARFieldAssignList(&value->u.fieldList);
         break;
      case AR_ACTIVE_LINK_ACTION_PROCESS :
         GetMallocChar("   Command string (): ", "", &value->u.process);
         break;
      case AR_ACTIVE_LINK_ACTION_MESSAGE :
         GetARMessageStruct(&value->u.message);
         break;
      case AR_ACTIVE_LINK_ACTION_SET_CHAR :
         GetARFieldCharacteristics(&value->u.characteristics);
         break;
      case AR_ACTIVE_LINK_ACTION_DDE      :
         GetARDDEStruct(&value->u.dde);
         break;
      case AR_ACTIVE_LINK_ACTION_FIELDP  :
         GetARPushFieldsAssignList(&value->u.pushFieldsList);
         break;
      case AR_ACTIVE_LINK_ACTION_SQL     :
         GetARSQLStruct(&value->u.sqlCommand);
         break;
      case AR_ACTIVE_LINK_ACTION_AUTO    :
         GetARAutomationStruct(&value->u.automation);
         break;
      case AR_ACTIVE_LINK_ACTION_OPENDLG   :
         GetAROpenDlgStruct(&value->u.openDlg);
         break;
      case AR_ACTIVE_LINK_ACTION_COMMITC   :
         break;
      case AR_ACTIVE_LINK_ACTION_CLOSEWND :
         break;
      case AR_ACTIVE_LINK_ACTION_CALLGUIDE  :
         GetARCallGuideStruct(&value->u.callGuide);
         break;
      case AR_ACTIVE_LINK_ACTION_EXITGUIDE  :
         GetARExitGuideStruct(&value->u.exitGuide);
         break;
      case AR_ACTIVE_LINK_ACTION_GOTOGUIDELABEL :
         GetARGotoGuideLabel(&value->u.gotoGuide);
         break;
      case AR_ACTIVE_LINK_ACTION_WAIT:
         GetARWaitStruct(&value->u.waitAction);
         break;
      case AR_ACTIVE_LINK_ACTION_GOTOACTION:
         GetARGotoActionStruct(&value->u.gotoAction);
         break;
      default:
         break;
   }
}


/*****************************************************************************/
/*                                                                           */
/*                          GetARActiveLinkActionList                        */
/*                                                                           */
/*****************************************************************************/

void GetARActiveLinkActionList(actionFlag, value)
ARBoolean               actionFlag;/* IN; TRUE - action; FALSE - else */
ARActiveLinkActionList *value;     /* OUT; loaded with value for buffer */

{
   int                       i;      /* working index */
   ARActiveLinkActionStruct *tempPtr;/* working pointer */

   if (actionFlag)
      value->numItems = GetInt("Number of actions for the active link (1): ",1);
   else
      value->numItems = GetInt("Number of elses for the active link (0): ", 0);
   if (value->numItems == 0)
      value->actionList = NULL;
   else
   {
      value->actionList = (ARActiveLinkActionStruct *)
                     malloc(sizeof(ARActiveLinkActionStruct) * value->numItems);
      if (value->actionList)
      {
         tempPtr = value->actionList;
         for (i = 0; i < (int) value->numItems; i++)
         {
            GetARActiveLinkActionStruct(tempPtr);
            tempPtr++;
         }
      }
   }
}


/*****************************************************************************/
/*                                                                           */
/*                           GetARPermissionStruct                           */
/*                                                                           */
/*****************************************************************************/

void GetARPermissionStruct(fieldFlag, value)
ARBoolean           fieldFlag;/* IN; TRUE - field; FALSE - schema */
ARPermissionStruct *value;    /* OUT; loaded with value for buffer */

{
   DriverPrintPrompt("   Permission Struct:\n");
   value->groupId = GetInt("      Group id (3): ", 0);
   if (fieldFlag)
      value->permissions = GetInt("      None, View, or Change (0, 1, 2): ", 0);
   else
      value->permissions = GetInt("      None, Visible, or Hidden(0, 1, 2): ",
                                  0);
}


/*****************************************************************************/
/*                                                                           */
/*                           GetARPermissionList                             */
/*                                                                           */
/*****************************************************************************/

void GetARPermissionList(fieldFlag, value)
ARBoolean         fieldFlag;  /* IN; TRUE - field; FALSE - schema */
ARPermissionList *value;      /* OUT; loaded with value for buffer */

{
   int                 i;      /* working index */
   ARPermissionStruct *tempPtr;/* working pointer */

   DriverPrintPrompt("Permission List:\n");
   value->numItems = GetInt("   Number of permission defns (0): ", 0);
   if (value->numItems == 0)
      value->permissionList = NULL;
   else
   {
      value->permissionList = (ARPermissionStruct *)
                           malloc(sizeof(ARPermissionStruct) * value->numItems);
      if (value->permissionList != NULL)
      {
         tempPtr = value->permissionList;
         for (i = 0; i < (int) value->numItems; i++)
         {
            GetARPermissionStruct(fieldFlag, tempPtr);
            tempPtr++;
         }
      }
   }
}


/*****************************************************************************/
/*                                                                           */
/*                         GetARIntegerLimitsStruct                          */
/*                                                                           */
/*****************************************************************************/

void GetARIntegerLimitsStruct(value)
ARIntegerLimitsStruct *value; /* OUT; loaded with value for buffer */

{
   value->rangeLow = GetLong("   Integer range low (0): ", 0L);
   value->rangeHigh = GetLong("   Integer range high (5000): ", 5000L);
}


/*****************************************************************************/
/*                                                                           */
/*                           GetARRealLimitsStruct                           */
/*                                                                           */
/*****************************************************************************/

void GetARRealLimitsStruct(value)
ARRealLimitsStruct *value;    /* OUT; loaded with value for buffer */

{
   value->rangeLow = GetReal("   Real range low (0): ", (double) 0);
   value->rangeHigh = GetReal("   Real range high (5000): ", (double) 5000);
   value->precision = GetInt("   Precision (-1 is no precision) (-1): ", -1);
}


/*****************************************************************************/
/*                                                                           */
/*                           GetARCharLimitsStruct                           */
/*                                                                           */
/*****************************************************************************/

void GetARCharLimitsStruct(value)
ARCharLimitsStruct *value;    /* OUT; loaded with value for buffer */

{
   value->maxLength = GetInt("   Maximum length (128): ", 128);
   value->menuStyle = GetInt(
                       "   Menu style - Append or Overwrite (1 or 2) (1): ", 1);
   value->qbeMatchOperation = GetInt(
     "   Match operation - Anywhere, Leading match, or Equal (1 - 3) (1): ", 1);
   (void) strcpy(value->charMenu, GetChar("   Character menu (): ", ""));
   GetMallocChar("   Pattern (): ", "", &value->pattern);
   value->fullTextOptions = GetInt("   FullText Options(0): ", 0);
}

/*****************************************************************************/
/*                                                                           */
/*                           GetARDiaryLimitsStruct                          */
/*                                                                           */
/*****************************************************************************/

void GetARDiaryLimitsStruct(value)
ARDiaryLimitsStruct *value;    /* OUT; loaded with value for buffer */

{
   value->fullTextOptions = GetInt("   FullText Options(0): ", 0);
}


/*****************************************************************************/
/*                                                                           */
/*                           GetARAttachLimitsStruct                         */
/*                                                                           */
/*****************************************************************************/

void GetARAttachLimitsStruct(value)
ARAttachLimitsStruct *value;    /* OUT; loaded with value for buffer */

{
   value->maxSize = GetInt("   maxSize(0): ", 0);
   value->attachType = AR_ATTACH_FIELD_TYPE_EMBED;
}

/*****************************************************************************/
/*                                                                           */
/*                           GetARTableLimitsStruct                          */
/*                                                                           */
/*****************************************************************************/

void GetARTableLimitsStruct(value)
ARTableLimitsStruct *value;    /* OUT; loaded with value for buffer */

{
   (void) strcpy(value->schema, GetChar("   schema name : ", ""));
   (void) strcpy(value->server, GetChar("   server name : ", ""));
   GetARQualifierStruct(&value->qualifier);
   value->maxRetrieve = GetInt("  Max rows to retrieve (10): ", 10);
   value->numColumns = 0;
}

/*****************************************************************************/
/*                                                                           */
/*                           GetARColumnLimitsStruct                         */
/*                                                                           */
/*****************************************************************************/

void GetARColumnLimitsStruct(value)
ARColumnLimitsStruct *value;    /* OUT; loaded with value for buffer */

{
   value->parent = GetLong("   Parent Table field Id : ", 0);
   value->dataField = GetLong("   Data field Id : ", 0);
   value->colLength = GetInt("   Length to display(10) : ", 10);
}

/*****************************************************************************/
/*                                                                           */
/*                         GetARDecimalLimitsStruct                          */
/*                                                                           */
/*****************************************************************************/

void GetARDecimalLimitsStruct(value)
ARDecimalLimitsStruct *value;    /* OUT; loaded with value for buffer */

{
   GetMallocChar("   Decimal range low (0.0): ", "0.0", &value->rangeLow);
   GetMallocChar("   Decimal range high (5000.0): ", "5000.0", &value->rangeHigh);
   value->precision = GetInt("   Precision (0): ", 0);
}

/*****************************************************************************/
/*                                                                           */
/*                           GetARFieldLimitStruct                           */
/*                                                                           */
/*****************************************************************************/

void GetARFieldLimitStruct(dataType, value)
int                 dataType;/* IN; datatype to get limits for */
ARFieldLimitStruct *value;   /* OUT; loaded with value for buffer */

{
   if (GetARBoolean("Set to NO LIMITS? (F): ", FALSE))
      value->dataType = AR_FIELD_LIMIT_NONE;
   else
   {
      value->dataType = dataType;
      switch (dataType)
      {
         case AR_DATA_TYPE_INTEGER :
            GetARIntegerLimitsStruct(&value->u.intLimits);
            break;
         case AR_DATA_TYPE_REAL    :
            GetARRealLimitsStruct(&value->u.realLimits);
            break;
         case AR_DATA_TYPE_CHAR    :
            GetARCharLimitsStruct(&value->u.charLimits);
            break;
         case AR_DATA_TYPE_DIARY   :
            GetARDiaryLimitsStruct(&value->u.diaryLimits);
            break;
         case AR_DATA_TYPE_TIME    :
            break;
         case AR_DATA_TYPE_ENUM    :
            GetARNameList(&value->u.enumLimits);
            break;
         case AR_DATA_TYPE_BITMASK :
            GetARNameList(&value->u.maskLimits);
            break;
         case AR_DATA_TYPE_ATTACH :
            GetARAttachLimitsStruct(&value->u.attachLimits);
            break;
         case AR_DATA_TYPE_DECIMAL:
            GetARDecimalLimitsStruct(&value->u.decimalLimits);
            break;
         case AR_DATA_TYPE_TABLE :
            GetARTableLimitsStruct(&value->u.tableLimits);
            break;
         case AR_DATA_TYPE_COLUMN :
            GetARColumnLimitsStruct(&value->u.columnLimits);
            break;
      }
   }
}


/*****************************************************************************/
/*                                                                           */
/*                          GetARCharMenuItemStruct                          */
/*                                                                           */
/*****************************************************************************/

void GetARCharMenuItemStruct(value)
ARCharMenuItemStruct *value;  /* OUT; loaded with value for buffer */

{
   DriverPrintPrompt("   Char Menu Item Struct:\n");
   (void) strcpy(value->menuLabel, GetChar("      Menu Label (): ", ""));
   value->menuType = GetInt("      Menu type  Value or Menu (1 or 2) (1): ", 1);
   switch (value->menuType)
   {
      case AR_MENU_TYPE_VALUE :
         GetMallocChar("      Char menu text (): ", "", &value->u.menuValue);
         break;
      case AR_MENU_TYPE_MENU  :
         value->u.childMenu = (ARCharMenuStruct *)
                                               malloc(sizeof(ARCharMenuStruct));
         if (value->u.childMenu != NULL)
            GetARCharMenuStruct(value->u.childMenu);
         break;
      default                 :
         break;
   }
}


/*****************************************************************************/
/*                                                                           */
/*                         GetARCharMenuQueryStruct                          */
/*                                                                           */
/*****************************************************************************/

void GetARCharMenuQueryStruct(value)
ARCharMenuQueryStruct *value; /* OUT; loaded with value for buffer */

{
   DriverPrintPrompt("   Char Menu Query Struct:\n");
   (void) strcpy(value->schema, GetChar("      Schema (): ", ""));
   (void) strcpy(value->server, GetChar("      Server (): ", ""));
   GetARQualifierStruct(&value->qualifier);
   value->labelField = GetLong("      Id of the label field (0): ", 0);
   value->valueField = GetLong("      Id of the value field (0): ", 0);
   value->sortOnLabel = GetARBoolean("      Sort the label field (T): ", TRUE);
}


/*****************************************************************************/
/*                                                                           */
/*                          GetARCharMenuSQLStruct                           */
/*                                                                           */
/*****************************************************************************/

void GetARCharMenuSQLStruct(value)
ARCharMenuSQLStruct *value; /* OUT; loaded with value for buffer */

{
   DriverPrintPrompt("   Char Menu SQL Struct:\n");
   (void) strcpy(value->server, GetChar("      Server (): ", ""));
   GetMallocChar("      SQL command (): ", "", &value->sqlCommand);
   value->labelIndex = GetLong("      Index of the label field (1): ", 1);
   value->valueIndex = GetLong("      Index of the value field (1): ", 1);
}


/*****************************************************************************/
/*                                                                           */
/*                             GetARCharMenuList                             */
/*                                                                           */
/*****************************************************************************/

void GetARCharMenuList(value)
ARCharMenuList *value;        /* OUT; loaded with value for buffer */

{
   int                   i;       /* working index */
   ARCharMenuItemStruct *tempPtr; /* working pointer */

   value->numItems = GetInt("   Number of char menu items (0): ", 0);
   if (value->numItems == 0)
      value->charMenuList = NULL;
   else
   {
      value->charMenuList = (ARCharMenuItemStruct *)
                         malloc(sizeof(ARCharMenuItemStruct) * value->numItems);
      if (value->charMenuList != NULL)
      {
         tempPtr = value->charMenuList;
         for (i = 0; i < (int) value->numItems; i++)
         {
            GetARCharMenuItemStruct(tempPtr);
            tempPtr++;
         }
      }
   }
}


/*****************************************************************************/
/*                                                                           */
/*                            GetARCharMenuStruct                            */
/*                                                                           */
/*****************************************************************************/

void GetARCharMenuStruct(value)
ARCharMenuStruct *value;      /* OUT; loaded with value for buffer */

{
   DriverPrintPrompt("   Char Menu Struct:\n");
   value->menuType =
       GetInt("      Menu type  None, List, Query, File, SQL (0 - 4) (1): ", 1);
   switch (value->menuType)
   {
      case AR_CHAR_MENU_NONE  :
         break;
      case AR_CHAR_MENU_LIST  :
         GetARCharMenuList(&value->u.menuList);
         break;
      case AR_CHAR_MENU_QUERY :
         GetARCharMenuQueryStruct(&value->u.menuQuery);
         break;
      case AR_CHAR_MENU_FILE  :
         value->u.menuFile.fileLocation =
              GetInt("      File location  Server or Client (1 or 2) (1): ", 1);
         GetMallocChar("      Filename (): ", "", &value->u.menuFile.filename);
         break;
      case AR_CHAR_MENU_SQL   :
         GetARCharMenuSQLStruct(&value->u.menuSQL);
         break;
      default                 :
         break;
   }
}


/*****************************************************************************/
/*                                                                           */
/*                            GetARStructItemStruct                          */
/*                                                                           */
/*****************************************************************************/

void GetARStructItemStruct(value)
ARStructItemStruct *value;    /* OUT; loaded with value for buffer */

{
   value->type = GetInt(
"   Schema(full,defn,view,mail),Filter,ActLink,<7-obsolete>,CharMenu,\n\
   Escalation,Dist Map, Schema(min view), Container (1 - 6, 8 - 12): ",
                        0);
   (void) strcpy(value->name, GetChar("   Structure name (): ", ""));
}


/*****************************************************************************/
/*                                                                           */
/*                            GetARStructItemList                            */
/*                                                                           */
/*****************************************************************************/

void GetARStructItemList(value)
ARStructItemList *value;      /* OUT; loaded with value for buffer */

{
   int                 i;      /* working index */
   ARStructItemStruct *tempPtr;/* working pointer */

   DriverPrintPrompt("Struct Item List:\n");
   value->numItems = GetInt("   Number of structure items (0): ", 0);
   if (value->numItems == 0)
      value->structItemList = NULL;
   else
   {
      value->structItemList = (ARStructItemStruct *)
                           malloc(sizeof(ARStructItemStruct) * value->numItems);
      if (value->structItemList != NULL)
      {
         tempPtr = value->structItemList;
         for (i = 0; i < (int) value->numItems; i++)
         {
            GetARStructItemStruct(tempPtr);
            tempPtr++;
         }
      }
   }
}


/*****************************************************************************/
/*                                                                           */
/*                         GetARServerInfoRequestList                        */
/*                                                                           */
/*****************************************************************************/

void GetARServerInfoRequestList(value)
ARServerInfoRequestList *value;/* OUT; loaded with value for buffer */

{
   unsigned int  i;           /* working index */
   unsigned int *tempPtr;     /* working pointer */
   char          buffer[80];  /* place to put the prompt */

   value->numItems = GetInt("   Number of server info operations (0): ", 0);
   if (value->numItems == 0)
      value->requestList = NULL;
   else
   {
      value->requestList = (unsigned int *)
                                 malloc(sizeof(unsigned int) * value->numItems);
      if (value->requestList != NULL)
      {
         tempPtr = value->requestList;
         for (i = 0; i < value->numItems; i++)
         {
            sprintf(buffer, "   Operation (1-%d) (1): ",
                           AR_MAX_SERVER_INFO_USED);
            *tempPtr = GetInt(buffer, 1);
            tempPtr++;
         }
      }
   }
}

/*****************************************************************************/
/*                                                                           */
/*                         GetARFullTextInfoRequestList                      */
/*                                                                           */
/*****************************************************************************/

void GetARFullTextInfoRequestList(value)
ARFullTextInfoRequestList *value;/* OUT; loaded with value for buffer */
{
   int           i;           /* working index */
   unsigned int *tempPtr;     /* working pointer */

   value->numItems = GetInt("   Number of fulltext info types (0): ", 0);
   if (value->numItems == 0)
      value->requestList = NULL;
   else
   {
      value->requestList = (unsigned int *)
                                 malloc(sizeof(unsigned int) * value->numItems);
      if (value->requestList != NULL)
      {
         tempPtr = value->requestList;
         for (i = 0; i < (int) value->numItems; i++)
         {
            *tempPtr = GetInt("   InfoType (1-6) (1): ", 1);
            tempPtr++;
         }
      }
   }
}

/*****************************************************************************/
/*                                                                           */
/*                             GetARServerInfoStruct                         */
/*                                                                           */
/*****************************************************************************/

void GetARServerInfoStruct(value)
ARServerInfoStruct *value;     /* OUT; loaded with value for buffer */

{
   char   buffer[80];          /* place to put the prompt */

   DriverPrintPrompt("Server info list entry\n");
   sprintf(buffer, "Operation (1-%d) (1): ", AR_MAX_SERVER_INFO_USED);
   value->operation = GetInt(buffer, 1);
   GetARValueStruct(&value->value);
}


/*****************************************************************************/
/*                                                                           */
/*                             GetARServerInfoList                           */
/*                                                                           */
/*****************************************************************************/

void GetARServerInfoList(value)
ARServerInfoList  *value;    /* OUT; loaded with value for buffer */

{
   int                 i;       /* working index */
   ARServerInfoStruct *tempPtr; /* working pointer */

   value->numItems = GetInt("   Number of server info operations (0): ", 0);
   if (value->numItems == 0)
      value->serverInfoList = NULL;
   else
   {
      value->serverInfoList = (ARServerInfoStruct *)
                           malloc(sizeof(ARServerInfoStruct) * value->numItems);
      if (value->serverInfoList != NULL)
      {
         tempPtr = value->serverInfoList;
         for (i = 0; i < (int) value->numItems; i++)
         {
            GetARServerInfoStruct(tempPtr);
            tempPtr++;
         }
      }
   }
}

/*****************************************************************************/
/*                                                                           */
/*                             GetARFullTextInfoStruct                       */
/*                                                                           */
/*****************************************************************************/

void GetARFullTextInfoStruct(value)
ARFullTextInfoStruct *value;     /* OUT; loaded with value for buffer */

{
   DriverPrintPrompt("FullText info list entry\n");
   value->infoType = GetInt("InfoType (1-6) (1): ", 1);
   switch (value->infoType)
   {
      case AR_FULLTEXTINFO_COLLECTION_DIR:
         GetARValueStruct(&value->u.value);
         break;
      case AR_FULLTEXTINFO_STOPWORD:
         GetARValueList(&value->u.valueList);
         break;
      case AR_FULLTEXTINFO_REINDEX:
         GetARValueStruct(&value->u.value);
         break;
      case AR_FULLTEXTINFO_CASE_SENSITIVE_SRCH:
         GetARValueStruct(&value->u.value);
         break;
      case AR_FULLTEXTINFO_STATE:
         GetARValueStruct(&value->u.value);
         break;
      case AR_FULLTEXTINFO_FTS_MATCH_OP:
         GetARValueStruct(&value->u.value);
         break;
   }
}

/*****************************************************************************/
/*                                                                           */
/*                             GetARFullTextInfoList                         */
/*                                                                           */
/*****************************************************************************/

void GetARFullTextInfoList(value)
ARFullTextInfoList  *value;    /* OUT; loaded with value for buffer */

{
   ARFullTextInfoStruct *tempPtr; /* working pointer */
   int                   i;       /* working index */

   value->numItems = GetInt("   Number of fulltext info types (0): ", 0);
   if (value->numItems == 0)
      value->fullTextInfoList = NULL;
   else
   {
      value->fullTextInfoList = (ARFullTextInfoStruct *)
                       malloc(sizeof(ARFullTextInfoStruct) * value->numItems);
      if (value->fullTextInfoList != NULL)
      {
         tempPtr = value->fullTextInfoList;
         for (i = 0; i < (int) value->numItems; i++)
         {
            GetARFullTextInfoStruct(tempPtr);
            tempPtr++;
         }
      }
   }
}


/*****************************************************************************/
/*                                                                           */
/*                              GetARReferenceList                           */
/*                                                                           */
/*****************************************************************************/

void GetARReferenceList(
ARReferenceList      *value      /* OUT; loaded with value for buffer */
)
{
   unsigned int         i;          /* working index */
   ARReferenceStruct   *tempPtr;    /* working pointer */

   value->numItems = GetInt("   Number of references (0): ", 0);
   if (value->numItems == 0)
      value->referenceList = NULL;
   else
   {
      value->referenceList = (ARReferenceStruct *)
                                malloc(sizeof(ARReferenceStruct) * value->numItems);
      if (value->referenceList != NULL)
      {
         tempPtr = value->referenceList;
         for (i = 0; i < value->numItems; i++)
         {
            GetARReferenceStruct(tempPtr);
            tempPtr++;
         }
      }
   }
}


/*****************************************************************************/
/*                                                                           */
/*                             GetARReferenceStruct                          */
/*                                                                           */
/*****************************************************************************/

void GetARReferenceStruct(
ARReferenceStruct    *value      /* OUT; loaded with value for buffer */
)
{
   GetMallocChar("   Reference label (): ", "", &value->label);
   GetMallocChar("   Reference description (): ", "", &value->description);
   DriverPrintPrompt("   Reference type none,schema,filter,esc,actlink,\n");
   value->type = GetInt("      container,charmenu (0,2-7) (0): ", 0);

   value->reference.dataType = GetInt("   References data type ARS,external (0,1) (0): ", 0);

   if (value->reference.dataType == ARREF_DATA_ARSREF)
      strcpy(value->reference.u.name, GetChar("   ARS reference name (): ", ""));
   else if (value->reference.dataType == ARREF_DATA_EXTREF)
      GetARExtReferenceStruct(&value->reference.u.extRef);
}


/*****************************************************************************/
/*                                                                           */
/*                           GetARExtReferenceStruct                         */
/*                                                                           */
/*****************************************************************************/

void GetARExtReferenceStruct(
ARExtReferenceStruct *value      /* OUT; loaded with value for buffer */
)
{
   DriverPrintPrompt("   External reference access group ids: \n");
   GetARInternalIdList(&value->permittedGroups);
   DriverPrintPrompt("   External reference value: ");
   GetARValueStruct(&value->value);
}


/*****************************************************************************/
/*                                                                           */
/*                            GetARReferenceTypeList                         */
/*                                                                           */
/*****************************************************************************/

void GetARReferenceTypeList(
ARReferenceTypeList  *value      /* OUT; loaded with value for buffer */
)
{
   unsigned int   i;          /* working index */
   int           *tempPtr;    /* working pointer */

   value->numItems = GetInt("   Number of reference types (0): ", 0);
   if (value->numItems == 0)
      value->refType = NULL;
   else
   {
      value->refType = (int *) malloc(sizeof(unsigned int) * value->numItems);
      if (value->refType != NULL)
      {
         tempPtr = value->refType;
         for (i = 0; i < value->numItems; i++)
         {
            *tempPtr = GetInt("   Reference type none,schema,filter,esc,actlink,container (0,2-6) (0): ", 0);
            tempPtr++;
         }
      }
   }
}


/*****************************************************************************/
/*                                                                           */
/*                            GetARContainerOwnerObj                         */
/*                                                                           */
/*****************************************************************************/

void GetARContainerOwnerObj(
ARContainerOwnerObj  *value      /* OUT; loaded with value for buffer */
)
{
   value->type = GetInt("   Container owner type (unowned, all, owned) (0-2) (0): ", 0);
   if ((value->type == ARCONOWNER_NONE) || (value->type == ARCONOWNER_ALL))
      value->ownerName[0] = '\0';
   else
      strcpy(value->ownerName, GetChar("   Container owner name (): ", ""));
}


/*****************************************************************************/
/*                                                                           */
/*                            GetARContainerTypeList                         */
/*                                                                           */
/*****************************************************************************/

void GetARContainerTypeList(
ARContainerTypeList  *value      /* OUT; loaded with value for buffer */
)
{
   unsigned int   i;          /* working index */
   int           *tempPtr;    /* working pointer */

   value->numItems = GetInt("   Number of container types (0): ", 0);
   if (value->numItems == 0)
      value->type = NULL;
   else
   {
      value->type = (int *) malloc(sizeof(unsigned int) * value->numItems);
      if (value->type != NULL)
      {
         tempPtr = value->type;
         for (i = 0; i < value->numItems; i++)
         {
            *tempPtr = GetInt("   Container type (0): ", 0);
            tempPtr++;
         }
      }
   }
}

/*****************************************************************************/
/*                                                                           */
/*                              GetARSignalList                              */
/*                                                                           */
/*****************************************************************************/

void GetARSignalList(
ARSignalList  *value      /* OUT; loaded with value for buffer */
)
{
   unsigned int   i;          /* working index */
   int           *tempPtr;    /* working pointer */

   value->numItems = GetInt("   Number of Signals (0): ", 0);
   if (value->numItems == 0)
      value->signal = NULL;
   else
   {
      value->signal= (int *) malloc(sizeof(unsigned int) * value->numItems);
      if (value->signal != NULL)
      {
         tempPtr = value->signal;
         for (i = 0; i < value->numItems; i++)
         {
            *tempPtr = GetInt("   Signal (cache,group,license)(1-3): ", 0);
            tempPtr++;
         }
      }
   }
}
