/* File:  print.c */

#include <stdio.h>
#include <stdarg.h>
#include <string.h>

#include "ar.h"
#include "arstruct.h"
#include "globals.h"

#include "api.h"
#include "print.h"
#include "util.h"


/*****************************************************************************/
/*                                                                           */
/*                                DriverPrint                                */
/*                                                                           */
/*****************************************************************************/

void DriverPrint(const char *format, ...)
{
   va_list marker;

   va_start(marker, format);

   vfprintf(((ThreadControlBlock *) GetThreadControlBlockPtr())->outFile,
            format, marker);

   va_end(marker);
}


/*****************************************************************************/
/*                                                                           */
/*                               PrintARBoolean                              */
/*                                                                           */
/*****************************************************************************/

void PrintARBoolean(header, value)
char      *header;            /* IN; header string for the value */
ARBoolean  value;             /* IN; value to print */

{
   DriverPrint((value) ? "%s TRUE\n" : "%s FALSE\n", header);
}


/*****************************************************************************/
/*                                                                           */
/*                                  PrintInt                                 */
/*                                                                           */
/*****************************************************************************/

void PrintInt(header, value)
char      *header;            /* IN; header string for the value */
int        value;             /* IN; value to print */

{
   DriverPrint("%s %d\n", header, value);
}          


/*****************************************************************************/
/*                                                                           */
/*                                  PrintLong                                */
/*                                                                           */
/*****************************************************************************/

void PrintLong(header, value)
char      *header;            /* IN; header string for the value */
long       value;             /* IN; value to print */

{
   DriverPrint("%s %ld\n", header, value);
}          


/*****************************************************************************/
/*                                                                           */
/*                                 PrintReal                                 */
/*                                                                           */
/*****************************************************************************/

void PrintReal(header, value)
char      *header;            /* IN; header string for the value */
double     value;             /* IN; value to print */

{
   DriverPrint("%s %lf\n", header, value);
}          


/*****************************************************************************/
/*                                                                           */
/*                                 PrintChar                                 */
/*                                                                           */
/*****************************************************************************/

void PrintChar(header, value)
char      *header;            /* IN; header string for the value */
char      *value;             /* IN; value to print */

{
   if (value == NULL)
      DriverPrint("%s\n", header);
   else
      DriverPrint("%s %s\n", header, value);
}          


/*****************************************************************************/
/*                                                                           */
/*                             PrintARTimestamp                              */
/*                                                                           */
/*****************************************************************************/

void PrintARTimestamp(header, value)
char       *header;           /* IN; header string for the value */
ARTimestamp value;            /* IN; value to print */

{
   DriverPrint("%s %ld\n", header, value);
}          


/*****************************************************************************/
/*                                                                           */
/*                             PrintAREntryIdList                            */
/*                                                                           */
/*****************************************************************************/

void PrintAREntryIdList(header, header2, value)
char          *header;        /* IN; header string for the value */
char          *header2;       /* IN; header string for 2nd+ line */
AREntryIdList *value;         /* IN; value to print */

{
   char           buff[82];   /* buffer to build output in */
   int            i;          /* working index */
   AREntryIdType *tempPtr;    /* working pointer */

   (void) sprintf(buff, "%s (%u)  ", header, value->numItems);
   if (value->numItems != 0)
   {
      tempPtr = value->entryIdList;
      for (i = 0; i < (int) value->numItems; i++)
      {
         if (i != 0)
            (void) strcat(buff, ", ");
         if ((strlen(buff) + strlen(*tempPtr)) >= (size_t) 79)
         {
            buff[strlen(buff) - 1] = '\n'; /* over last blank */
            DriverPrint("%s", buff);
            (void) strcpy(buff, header2);
         }
         (void) strcat(buff, *tempPtr);
         tempPtr++;
      }
   }
   DriverPrint("%s\n", buff);
}


/*****************************************************************************/
/*                                                                           */
/*                               PrintARInternalId                           */
/*                                                                           */
/*****************************************************************************/

void PrintARInternalId(header, value)
char        *header;          /* IN; header string for the value */
ARInternalId value;           /* IN; value to print */

{
   DriverPrint("%s %lu\n", header, value);
}          


/*****************************************************************************/
/*                                                                           */
/*                           PrintARInternalIdList                           */
/*                                                                           */
/*****************************************************************************/

void PrintARInternalIdList(header, header2, value)
char             *header;     /* IN; header string for the value */
char             *header2;    /* IN; header string for 2nd+ line */
ARInternalIdList *value;      /* IN; value to print */

{
   char          buff[82];    /* buffer to build output in */
   char          buff2[16];   /* buffer to hold a converted number */
   int           i;           /* working index */
   ARInternalId *tempPtr;     /* working pointer */

   (void) sprintf(buff, "%s (%u)  ", header, value->numItems);
   if (value->numItems != 0)
   {
      tempPtr = value->internalIdList;
      for (i = 0; i < (int) value->numItems; i++)
      {
         if (i != 0)
            (void) strcat(buff, ", ");
         (void) sprintf(buff2, "%lu", *tempPtr);
         if ((strlen(buff) + strlen(buff2)) >= (size_t) 79)
         {
            buff[strlen(buff) - 1] = '\n'; /* over last blank */
            DriverPrint("%s", buff);
            (void) strcpy(buff, header2);
         }
         (void) strcat(buff, buff2);
         tempPtr++;
      }
   }
   DriverPrint("%s\n", buff);
}


/*****************************************************************************/
/*                                                                           */
/*                              PrintARNameList                              */
/*                                                                           */
/*****************************************************************************/

void PrintARNameList(header, header2, value)
char       *header;           /* IN; header string for the value */
char       *header2;          /* IN; header string for 2nd+ line */
ARNameList *value;            /* IN; value to print */

{
   char        buff[82];      /* buffer to build output in */
   int         i;             /* working index */
   ARNameType *tempPtr;       /* working pointer */

   (void) sprintf(buff, "%s (%u)  ", header, value->numItems);
   if (value->numItems != 0)
   {
      tempPtr = value->nameList;
      for (i = 0; i < (int) value->numItems; i++)
      {
         if (i != 0)
            (void) strcat(buff, ", ");
         if ((strlen(buff) + strlen(*tempPtr)) >= (size_t) 79)
         {
            buff[strlen(buff) - 1] = '\n'; /* over last blank */
            DriverPrint("%s", buff);
            (void) strcpy(buff, header2);
         }
         (void) strcat(buff, *tempPtr);
         tempPtr++;
      }
   }
   DriverPrint("%s\n", buff);
}


/*****************************************************************************/
/*                                                                           */
/*                            PrintARServerNameList                          */
/*                                                                           */
/*****************************************************************************/

void PrintARServerNameList(header, header2, value)
char             *header;     /* IN; header string for the value */
char             *header2;    /* IN; header string for 2nd+ line */
ARServerNameList *value;      /* IN; value to print */

{
   char              buff[82];/* buffer to build output in */
   int               i;       /* working index */
   ARServerNameType *tempPtr; /* working pointer */

   (void) sprintf(buff, "%s (%u)  ", header, value->numItems);
   if (value->numItems != 0)
   {
      tempPtr = value->nameList;
      for (i = 0; i < (int) value->numItems; i++)
      {
         if (i != 0)
            (void) strcat(buff, ", ");
         if ((strlen(buff) + strlen(*tempPtr)) >= (size_t) 79)
         {
            buff[strlen(buff) - 1] = '\n'; /* over last blank */
            DriverPrint("%s", buff);
            (void) strcpy(buff, header2);
         }
         (void) strcat(buff, *tempPtr);
         tempPtr++;
      }
   }
   DriverPrint("%s\n", buff);
}


/*****************************************************************************/
/*                                                                           */
/*                            PrintARControlStruct                           */
/*                                                                           */
/*****************************************************************************/

void PrintARControlStruct(value)
ARControlStruct *value;       /* IN; value to print */

{
   DriverPrint("Control record:\n");
   PrintChar("   User name: ", value->user);
   PrintChar("   Password : ", value->password);
   PrintChar("   Language : ", value->language);
   PrintARTimestamp("   Operation time:", value->operationTime);
   PrintChar("   Server   : ", value->server);
}


/*****************************************************************************/
/*                                                                           */
/*                            PrintARStatusStruct                            */
/*                                                                           */
/*****************************************************************************/

void PrintARStatusStruct(value)
ARStatusStruct *value;        /* IN; value to print */

{
   DriverPrint("Status Struct :\n");
   DriverPrint("   Message type : ");
   switch (value->messageType)
   {
      case AR_RETURN_OK         :
         DriverPrint("NOTE\n");
         break;
      case AR_RETURN_WARNING    :
         DriverPrint("WARNING\n");
         break;
      case AR_RETURN_ERROR      :
         DriverPrint("ERROR\n");
         break;
      default                   :
         DriverPrint("<unknown type>\n");
         break;
   }
   PrintLong("   Message number : ", value->messageNum);
   PrintChar("   Message: ", value->messageText);
   if (value->appendedText && value->appendedText[0])
      PrintChar("   Appended: ", value->appendedText);
}


/*****************************************************************************/
/*                                                                           */
/*                            PrintARStatusList                              */
/*                                                                           */
/*****************************************************************************/

void PrintARStatusList(value)
ARStatusList *value;          /* IN; value to print */

{
   int             i;         /* working index */
   ARStatusStruct *tempPtr;   /* working pointer */

   DriverPrint("Status List : %u items\n", value->numItems);
   if (value->numItems != 0)
   {
      tempPtr = value->statusList;
      for (i = 0; i < (int) value->numItems; i++)
      {
         PrintARStatusStruct(tempPtr);
         tempPtr++;
      }
   }
}


/*****************************************************************************/
/*                                                                           */
/*                            PrintARByteList                                */
/*                                                                           */
/*****************************************************************************/

void PrintARByteList(header, header2, value)
char          *header;        /* IN; header string for the value */
char          *header2;       /* IN; header string for 2nd+ line */
ARByteList    *value;         /* IN; value to print */
{
   int             i;
   int             length;
   unsigned char  *tempPtr;

   DriverPrint("%s  type - %lu with %u bytes", header, value->type,
               value->numItems);
   length = (80 - strlen(header2)) / 2;
   if (value->numItems != 0)
   {  /* Print the stream of bytes */
      tempPtr = value->bytes;
      for (i = 0; i < (int) value->numItems; i++)
      {
         if ((i % length) == 0)
            DriverPrint("\n%s", header2);
         DriverPrint("%02X", *tempPtr);
         tempPtr++;
      }
      DriverPrint("\n");
    }
}


/*****************************************************************************/
/*                                                                           */
/*                            PrintARCoordList                               */
/*                                                                           */
/*****************************************************************************/

void PrintARCoordList(header, header2, value)
char          *header;        /* IN; header string for the value */
char          *header2;       /* IN; header string for 2nd+ line */
ARCoordList   *value;         /* IN; value to print */
{
   int             i;
   ARCoordStruct  *tempPtr;

   DriverPrint("%s %u items\n", header, value->numItems);
   if (value->numItems != 0)
   {
      /* Print the pairs of coordinates */
      tempPtr = value->coords;
      for (i = 0; i < (int) value->numItems; i++)
      {
         DriverPrint("%sx: %ld, y: %ld\n", header2, tempPtr->x, tempPtr->y);
         tempPtr++;
      }
   }
}

/*****************************************************************************/
/*                                                                           */
/*                            PrintARAttachLocStruct                         */
/*                                                                           */
/*****************************************************************************/

void PrintARAttachLocStruct(header, value)
char           *header;        /* IN; header string for the value */
ARLocStruct    *value;         /* IN; value to print */
{
   DriverPrint("      %s :\n", header);

   if (value->locType == 0)
      DriverPrint("            Locator Type : (NULL)");
   else
   {
      DriverPrint("            Locator Type : %ld\n", value->locType);
      if (value->locType == AR_LOC_FILENAME)
         DriverPrint("            File Name    : %s\n", value->u.filename);
   }
}

/*****************************************************************************/
/*                                                                           */
/*                            PrintARAttachProps                             */
/*                                                                           */
/*****************************************************************************/

void PrintARAttachProps(header, value)
char              *header;        /* IN; header string for the value */
ARAttachStruct    *value;         /* IN; value to print */
{
   DriverPrint("%s\n", header);

   if (value->name[0] != '\0')
      DriverPrint("      Name            : %s\n", value->name);
   else
      DriverPrint("      Name            : <null>\n");

   DriverPrint("      Original size   : %ld\n", value->origSize);
   DriverPrint("      Compressed size : %ld\n", value->compSize);
   PrintARAttachLocStruct("Locator", &value->loc);
}


/*****************************************************************************/
/*                                                                           */
/*                            PrintARValueStruct                             */
/*                                                                           */
/*****************************************************************************/

void PrintARValueStruct(header, value)
char          *header;        /* IN; header string for the value */
ARValueStruct *value;         /* IN; value to print */

{
   char  buff[82];         /* buffer to hold second line header */

   DriverPrint("%sValue:  ", header);
   switch (value->dataType)
   {
      case AR_DATA_TYPE_NULL    :  /* NULL value (incl. no default) */
         DriverPrint("NULL\n");
         break;
      case AR_DATA_TYPE_KEYWORD :
         PrintInt("(keyword)  ", value->u.keyNum);
         break;
      case AR_DATA_TYPE_INTEGER :
         PrintLong("(integer)  ", (int) value->u.intVal);
         break;
      case AR_DATA_TYPE_REAL    :
         PrintReal("(real)  ", value->u.realVal);
         break;
      case AR_DATA_TYPE_CHAR    :
         PrintChar("(char)  ", value->u.charVal);
         break;
      case AR_DATA_TYPE_DIARY   :
         APIPrintDiary("(diary)  ", value->u.diaryVal);
         break;
      case AR_DATA_TYPE_ENUM    :
         PrintLong("(enumerated)  ", (int) value->u.enumVal);
         break;
      case AR_DATA_TYPE_TIME    :
         PrintARTimestamp("(timestamp)  ", value->u.timeVal);
         break;
      case AR_DATA_TYPE_BITMASK :
         PrintLong("(bitmask)  ", (int) value->u.maskVal);
         break;
      case AR_DATA_TYPE_BYTES :
         (void) sprintf(buff, "%s        ", header);
         PrintARByteList("(bytelist)  ", buff, value->u.byteListVal);
         break;
      case AR_DATA_TYPE_JOIN :
         DriverPrint("(join)\n");
         break;
      case AR_DATA_TYPE_TRIM :
         DriverPrint("(trim)\n");
         break;
      case AR_DATA_TYPE_CONTROL:
         DriverPrint("(control)\n");
         break;
      case AR_DATA_TYPE_ULONG :
         PrintLong("(ulong)  ", (int) value->u.ulongVal);
         break;
      case AR_DATA_TYPE_COORDS:
         (void) sprintf(buff, "%s        ", header);
         PrintARCoordList("(coordlist)  ", buff, value->u.coordListVal);
         break;
      case AR_DATA_TYPE_PAGE:
         DriverPrint("(Page)\n");
         break;
      case AR_DATA_TYPE_PAGE_HOLDER:
         DriverPrint("(PageHolder)\n");
         break;
      case AR_DATA_TYPE_ATTACH:
         PrintARAttachProps("(attachment)  ", value->u.attachVal);
         break;
      case AR_DATA_TYPE_DECIMAL :
         PrintChar("(decimal)  ", value->u.decimalVal);
         break;
      default                   :
         DriverPrint("<undefined data type tag = %u>\n", value->dataType);
         break;
   }
}


/*****************************************************************************/
/*                                                                           */
/*                             PrintARValueList                              */
/*                                                                           */
/*****************************************************************************/

void PrintARValueList(header, value)
char        *header;          /* IN; header string for the value */
ARValueList *value;           /* IN; value to print */

{
   char            buff[45];  /* buffer to build nested header in */
   int             i;         /* working index */
   ARValueStruct  *tempPtr;   /* working pointer */

   DriverPrint("%sValue List : %u items\n", header,value->numItems);
   if (value->numItems != 0)
   {
      if (strlen(header) >= (size_t) 40)
         buff[0] = '\0';
      else
         (void) sprintf(buff, "%s   ", header);

      tempPtr = value->valueList;
      for (i = 0; i < (int) value->numItems; i++)
      {
         PrintARValueStruct(buff, tempPtr);
         tempPtr++;
      }
   }
}


/*****************************************************************************/
/*                                                                           */
/*                           PrintARValueListList                            */
/*                                                                           */
/*****************************************************************************/

void PrintARValueListList(value)
ARValueListList *value;       /* IN; value to print */

{
   int             i;         /* working index */
   ARValueList    *tempPtr;   /* working pointer */

   DriverPrint("Value List List : %u items\n", value->numItems);
   if (value->numItems != 0)
   {
      tempPtr = value->valueListList;
      for (i = 0; i < (int) value->numItems; i++)
      {
         PrintARValueList("   ", tempPtr);
         tempPtr++;
      }
   }
}


/*****************************************************************************/
/*                                                                           */
/*                        PrintAREntryListFieldStruct                        */
/*                                                                           */
/*****************************************************************************/

void PrintAREntryListFieldStruct(value)
AREntryListFieldStruct *value;     /* IN; value to print */

{
   DriverPrint("   Entry List Field Struct:\n");
   PrintLong("      Field Id :", value->fieldId);
   PrintInt("      Column Width:", value->columnWidth);
   PrintChar("      Separator:", value->separator);
}


/*****************************************************************************/
/*                                                                           */
/*                        PrintAREntryListFieldList                          */
/*                                                                           */
/*****************************************************************************/

void PrintAREntryListFieldList(value)
AREntryListFieldList *value;  /* IN; value to print */

{
   int                     i;      /* working index */
   AREntryListFieldStruct *tempPtr;/* working pointer */

   DriverPrint("Entry List Field List : %u items\n", value->numItems);
   if (value->numItems != 0)
   {
      tempPtr = value->fieldsList;
      for (i = 0; i < (int) value->numItems; i++)
      {
         PrintAREntryListFieldStruct(tempPtr);
         tempPtr++;
      }
   }
}


/*****************************************************************************/
/*                                                                           */
/*                             PrintARSortList                               */
/*                                                                           */
/*****************************************************************************/

void PrintARSortList(value)
ARSortList *value;      /* IN; value to print */

{
   int            i;      /* working index */

   DriverPrint("Sort List : %u items\n", value->numItems);
   for (i = 0; i < (int) value->numItems; i++)
   {
      DriverPrint("   Sort Struct:\n");
      PrintLong("      Field Id :", value->sortList[i].fieldId);
      switch (value->sortList[i].sortOrder)
      {
         case AR_SORT_ASCENDING  :
            DriverPrint("      Sort Order: ASCENDING\n");
            break;
         case AR_SORT_DESCENDING :
            DriverPrint("      Sort Order: DESCENDING\n");
            break;
         default              :
            DriverPrint("      Sort Order: <undefined tag value = %u>\n",
                        value->sortList[i].sortOrder);
            break;
      }
   }
}


/*****************************************************************************/
/*                                                                           */
/*                           PrintAREntryListStruct                          */
/*                                                                           */
/*****************************************************************************/

void PrintAREntryListStruct(value)
AREntryListStruct *value;     /* IN; value to print */

{
   unsigned int i;

   DriverPrint("Entry List Struct:\n");
   if (value != NULL && value->entryId.numItems > 0)
   {
      DriverPrint("   Entry Id : %s", value->entryId.entryIdList[0]);
      for (i = 1; i < value->entryId.numItems; i++)
         DriverPrint("|%s", value->entryId.entryIdList[i]);
      DriverPrint("\n");
      PrintChar("   Short Description :", value->shortDesc);
   }
}


/*****************************************************************************/
/*                                                                           */
/*                           PrintAREntryListList                            */
/*                                                                           */
/*****************************************************************************/

void PrintAREntryListList(value)
AREntryListList *value;       /* IN; value to print */

{
   int                i;      /* working index */
   AREntryListStruct *tempPtr;/* working pointer */

   DriverPrint("Entry List List : %u items\n", value->numItems);
   if (value->numItems != 0)
   {
      tempPtr = value->entryList;
      for (i = 0; i < (int) value->numItems; i++)
      {
         PrintAREntryListStruct(tempPtr);
         tempPtr++;
      }
   }
}


/*****************************************************************************/
/*                                                                           */
/*                          PrintARFieldValueStruct                          */
/*                                                                           */
/*****************************************************************************/

void PrintARFieldValueStruct(value)
ARFieldValueStruct *value;    /* IN; value to print */

{
   DriverPrint("Field Value Struct:\n");
   PrintARInternalId("   Field Id :", value->fieldId);
   if ((value->fieldId == AR_CORE_STATUS_HISTORY) &&
       (value->value.dataType == AR_DATA_TYPE_CHAR))
      APIPrintStatusHistory("   Value: ", value->value.u.charVal);
   else
      PrintARValueStruct("   ", &value->value);
}


/*****************************************************************************/
/*                                                                           */
/*                          PrintARFieldValueList                            */
/*                                                                           */
/*****************************************************************************/

void PrintARFieldValueList(value)
ARFieldValueList *value;      /* IN; value to print */

{
   int                 i;      /* working index */
   ARFieldValueStruct *tempPtr;/* working pointer */

   DriverPrint("Field Value List : %u items\n", value->numItems);
   if (value->numItems != 0)
   {
      tempPtr = value->fieldValueList;
      for (i = 0; i < (int) value->numItems; i++)
      {
         PrintARFieldValueStruct(tempPtr);
         tempPtr++;
      }
   }
}

/*****************************************************************************/
/*                                                                           */
/*                      PrintARFieldValueListList                            */
/*                                                                           */
/*****************************************************************************/

void PrintARFieldValueListList(value)
ARFieldValueListList *value;      /* IN; value to print */

{
   int               i;       /* working index */
   ARFieldValueList *tempPtr; /* working pointer */

   DriverPrint("Field Value List List : %u items\n", value->numItems);
   if (value->numItems != 0)
   {
      tempPtr = value->valueListList;
      for (i = 0; i < (int) value->numItems; i++)
      {
         PrintARFieldValueList(tempPtr);
         tempPtr++;
      }
   }
}

/*****************************************************************************/
/*                                                                           */
/*                           PrintAREntryListFieldValueStruct                */
/*                                                                           */
/*****************************************************************************/

void PrintAREntryListFieldValueStruct(value)
AREntryListFieldValueStruct *value;     /* IN; value to print */

{
   unsigned int i;

   DriverPrint("Entry List Struct:\n");
   if (value != NULL && value->entryId.numItems > 0)
   {
      DriverPrint("   Entry Id : %s", value->entryId.entryIdList[0]);
      for (i = 1; i < value->entryId.numItems; i++)
         DriverPrint("|%s", value->entryId.entryIdList[i]);
      DriverPrint("\n");
      DriverPrint("Entry Field/Value List:\n");
      PrintARFieldValueList(value->entryValues);
   }
}

/*****************************************************************************/
/*                                                                           */
/*                           PrintAREntryListFieldValueList                  */
/*                                                                           */
/*****************************************************************************/

void PrintAREntryListFieldValueList(value)
AREntryListFieldValueList *value;       /* IN; value to print */

{
   int                          i;      /* working index */
   AREntryListFieldValueStruct *tempPtr;/* working pointer */

   DriverPrint("Entry List List : %u items\n", value->numItems);
   if (value->numItems != 0)
   {
      tempPtr = value->entryList;
      for (i = 0; i < (int) value->numItems; i++)
      {
         PrintAREntryListFieldValueStruct(tempPtr);
         tempPtr++;
      }
   }
}

/*****************************************************************************/
/*                                                                           */
/*                      PrintARFieldValueOrArithStruct                       */
/*                                                                           */
/*****************************************************************************/

void PrintARFieldValueOrArithStruct(header, value)
char                      *header;/* IN; header string for the value */
ARFieldValueOrArithStruct *value; /* IN; value to print */

{
   char   buff[45];           /* buffer to build nested header in */
   char   buff2[65];          /* buffer to build nested header in */

   if (strlen(header) >= (size_t) 40)
      buff[0] = '\0';
   else
      (void) sprintf(buff, "%s   ", header);

   DriverPrint("%sField, Value, or Arithmetic Struct:\n", header);
   switch (value->tag)
   {
      case AR_FIELD        :
         DriverPrint("%s   Tag: FIELD\n", header);
         (void) sprintf(buff2, "%s   Field Id :", header);
         PrintARInternalId(buff2, value->u.fieldId);
         break;
      case AR_VALUE        :
         DriverPrint("%s   Tag: VALUE\n", header);
         PrintARValueStruct(buff, &value->u.value);
         break;
      case AR_ARITHMETIC   :
         DriverPrint("%s   Tag: ARITHMETIC\n", header);
         PrintARArithOpStruct(buff, (ARArithOpStruct *) value->u.arithOp);
         break;
      case AR_STAT_HISTORY :
         DriverPrint("%s   Tag: STAT HISTORY\n", header);
         (void) sprintf(buff2, "%s   Enumerated value index :", header);
         PrintInt(buff2, (int) value->u.statHistory.enumVal);
         switch (value->u.statHistory.userOrTime)
         {
            case AR_STAT_HISTORY_USER :
               DriverPrint("%s   User or Time: USER\n", header);
               break;
            case AR_STAT_HISTORY_TIME :
               DriverPrint("%s   User or Time: TIME\n", header);
               break;
            default                   :
               DriverPrint("%s   User or Time: %s%u>\n", header,
                           "<undefined value = ",
                           value->u.statHistory.userOrTime);
               break;
         }
         break;
      case AR_VALUE_SET    :
         DriverPrint("%s   Tag: VALUE SET\n", header);
         PrintARValueList(buff, &value->u.valueSet);
         break;
      case AR_FIELD_TRAN   :
         DriverPrint("%s   Tag: FIELD TRAN\n", header);
         (void) sprintf(buff2, "%s   Field Id :", header);
         PrintARInternalId(buff2, value->u.fieldId);
         break;
      case AR_FIELD_DB     :
         DriverPrint("%s   Tag: FIELD DB\n", header);
         (void) sprintf(buff2, "%s   Field Id :", header);
         PrintARInternalId(buff2, value->u.fieldId);
         break;
      case AR_LOCAL_VARIABLE :
         DriverPrint("%s   Tag: LOCAL VARIABLE\n", header);
         (void) sprintf(buff2, "%s   Local Variable number:", header);
         PrintInt(buff2, (int) value->u.variable);
         break;
      case AR_QUERY        :
         DriverPrint("%s   Tag: QUERY\n", header);
         (void) sprintf(buff2, "%s   Schema :", header);
         PrintChar(buff2, value->u.queryValue->schema);
         (void) sprintf(buff2, "%s   Server :", header);
         PrintChar(buff2, value->u.queryValue->server);
         PrintARQualifierStruct(buff, value->u.queryValue->qualifier);
         PrintLong(buff2, (int) value->u.queryValue->valueField);
         DriverPrint("%s   Multi Value Code:", header);
         switch (value->u.queryValue->multiMatchCode)
         {
            case AR_QUERY_VALUE_MULTI_ERROR :
               DriverPrint("ERROR\n");
               break;
            case AR_QUERY_VALUE_MULTI_FIRST :
               DriverPrint("FIRST\n");
               break;
            case AR_QUERY_VALUE_MULTI_SET   :
               DriverPrint("SET\n");
               break;
            default                         :
               DriverPrint("<undefined multi-match value = %u>\n",
                           value->u.queryValue->multiMatchCode);
               break;
         }
         break;
      case AR_FIELD_CURRENT:
         DriverPrint("%s   Tag: FIELD (current screen)\n", header);
         (void) sprintf(buff2, "%s   Field Id :", header);
         PrintARInternalId(buff2, value->u.fieldId);
         break;
      default              :
         DriverPrint("%s   Tag: <undefined tag value = %u>\n", header,
                     value->tag);
         break;
   }
}


/*****************************************************************************/
/*                                                                           */
/*                           PrintARArithOpStruct                            */
/*                                                                           */
/*****************************************************************************/

void PrintARArithOpStruct(header, value)
char            *header;      /* IN; header string for the value */
ARArithOpStruct *value;       /* IN; value to print */

{
   char   buff[45];           /* buffer to build nested header in */

   if (strlen(header) >= (size_t) 40)
      buff[0] = '\0';
   else
      (void) sprintf(buff, "%s   ", header);

   DriverPrint("%sArithmetic Operation:\n", header);
   switch (value->operation)
   {
      case AR_ARITH_OP_ADD      :
         DriverPrint("%s   Operation: ADD\n", header);
         break;
      case AR_ARITH_OP_SUBTRACT :
         DriverPrint("%s   Operation: SUBTRACT\n", header);
         break;
      case AR_ARITH_OP_MULTIPLY :
         DriverPrint("%s   Operation: MULTIPLY\n", header);
         break;
      case AR_ARITH_OP_DIVIDE   :
         DriverPrint("%s   Operation: DIVIDE\n", header);
         break;
      case AR_ARITH_OP_MODULO   :
         DriverPrint("%s   Operation: MODULO\n", header);
         break;
      case AR_ARITH_OP_NEGATE   :
         DriverPrint("%s   Operation: NEGATE (unary minus)\n", header);
         break;
      default                   :
         DriverPrint("%s   Operation: <undefined operation = %u>\n", header,
                     value->operation);
         break;
   }
   if (value->operation != AR_ARITH_OP_NEGATE)
      PrintARFieldValueOrArithStruct(buff, &value->operandLeft);
   PrintARFieldValueOrArithStruct(buff, &value->operandRight);
}


/*****************************************************************************/
/*                                                                           */
/*                            PrintARRelOpStruct                             */
/*                                                                           */
/*****************************************************************************/

void PrintARRelOpStruct(header, value)
char          *header;        /* IN; header string for the value */
ARRelOpStruct *value;         /* IN; value to print */

{
   char   buff[45];           /* buffer to build nested header in */

   if (strlen(header) >= (size_t) 40)
      buff[0] = '\0';
   else
      (void) sprintf(buff, "%s   ", header);

   DriverPrint("%sRelational Operation:\n", header);
   switch (value->operation)
   {
      case AR_REL_OP_EQUAL         :
         DriverPrint("%s   Operation: EQUAL\n", header);
         break;
      case AR_REL_OP_GREATER       :
         DriverPrint("%s   Operation: GREATER\n", header);
         break;
      case AR_REL_OP_GREATER_EQUAL :
         DriverPrint("%s   Operation: GREATER_EQUAL\n", header);
         break;
      case AR_REL_OP_LESS          :
         DriverPrint("%s   Operation: LESS\n", header);
         break;
      case AR_REL_OP_LESS_EQUAL    :
         DriverPrint("%s   Operation: LESS_EQUAL\n", header);
         break;
      case AR_REL_OP_NOT_EQUAL     :
         DriverPrint("%s   Operation: NOT_EQUAL\n", header);
         break;
      case AR_REL_OP_LIKE          :
         DriverPrint("%s   Operation: LIKE\n", header);
         break;
      case AR_REL_OP_IN            :
         DriverPrint("%s   Operation: IN\n", header);
         break;
      default                      :
         DriverPrint("%s   Operation: <undefined operation = %u>\n", header,
                     value->operation);
         break;
   }
   PrintARFieldValueOrArithStruct(buff, &value->operandLeft);
   PrintARFieldValueOrArithStruct(buff, &value->operandRight);
}


/*****************************************************************************/
/*                                                                           */
/*                           PrintARQualifierStruct                          */
/*                                                                           */
/*****************************************************************************/

void PrintARQualifierStruct(header, value)
char              *header;    /* IN; header string for the value */
ARQualifierStruct *value;     /* IN; value to print */

{
   char   buff[45];           /* buffer to build nested header in */

   if (strlen(header) >= (size_t) 40)
      buff[0] = '\0';
   else
      (void) sprintf(buff, "%s   ", header);

   DriverPrint("%sQualifier Struct:\n", header);
   switch (value->operation)
   {
      case AR_COND_OP_NONE   :
         /* is a dummy with no value */
         DriverPrint("%s   Operation: <none>\n", header);
         break;
      case AR_COND_OP_AND    :
         DriverPrint("%s   Operation: AND\n", header);
         PrintARQualifierStruct(buff, value->u.andor.operandLeft);
         PrintARQualifierStruct(buff, value->u.andor.operandRight);
         break;
      case AR_COND_OP_OR     :
         DriverPrint("%s   Operation: OR\n", header);
         PrintARQualifierStruct(buff, value->u.andor.operandLeft);
         PrintARQualifierStruct(buff, value->u.andor.operandRight);
         break;
      case AR_COND_OP_NOT    :
         DriverPrint("%s   Operation: NOT\n", header);
         PrintARQualifierStruct(buff, value->u.not);
         break;
      case AR_COND_OP_REL_OP :
         DriverPrint("%s   Operation: RELATIONAL OPERATION\n", header);
         PrintARRelOpStruct(buff, value->u.relOp);
         break;
      default             :
         DriverPrint("%s   Operation: <undefined operation = %u>\n", header,
                     value->operation);
         break;
   }
}


/*****************************************************************************/
/*                                                                           */
/*                             PrintARIndexStruct                            */
/*                                                                           */
/*****************************************************************************/

void PrintARIndexStruct(value)
ARIndexStruct   *value;       /* IN; value to print */

{
   int   i;                   /* working index */

   DriverPrint("Index Struct:\n");
   PrintInt("   Number of Index Fields:", (int) value->numFields);
   for (i = 0; i < (int)value->numFields; i++)
      PrintLong("   Field Id:", value->fieldIds[i]);
   PrintARBoolean("   Unique flag:", value->unique);
}


/*****************************************************************************/
/*                                                                           */
/*                              PrintARIndexList                             */
/*                                                                           */
/*****************************************************************************/

void PrintARIndexList(value)
ARIndexList   *value;         /* IN; value to print */

{
   int              i;        /* working index */
   ARIndexStruct   *tempPtr;  /* working pointer */

   DriverPrint("Index List : %u items\n", value->numItems);
   if (value->numItems != 0)
   {
      tempPtr = value->indexList;
      for (i = 0; i < (int) value->numItems; i++)
      {
         PrintARIndexStruct(tempPtr);
         tempPtr++;
      }
   }
}


/*****************************************************************************/
/*                                                                           */
/*                         PrintARCompoundSchema                             */
/*                                                                           */
/*****************************************************************************/

void PrintARCompoundSchema(value)
ARCompoundSchema   *value;         /* IN; value to print */

{
   DriverPrint("Compound Schema Info:  ");
   switch (value->schemaType)
   {
      case AR_SCHEMA_NONE :
         DriverPrint("<none>\n");
         break;
      case AR_SCHEMA_REGULAR :
         DriverPrint("REGULAR\n");
         break;
      case AR_SCHEMA_DIALOG :
         DriverPrint("DIALOG\n");
         break;
      case AR_SCHEMA_JOIN :
         DriverPrint("JOIN\n");
         DriverPrint("  Join member A %s \n", value->u.join.memberA);
         DriverPrint("  Join member B %s \n", value->u.join.memberB);
         PrintARQualifierStruct("  Join qualification",&value->u.join.joinQual);
         DriverPrint("  Join option %d \n", value->u.join.option);
         break;
      case AR_SCHEMA_VIEW :
         DriverPrint("VIEW\n");
         DriverPrint("  View table name %s \n", value->u.view.tableName);
         DriverPrint("  View qualification %s \n", value->u.view.viewQual);
         break;
      default             :
         DriverPrint("  <undefined schemaType = %u>\n", value->schemaType);
         break;
   }
}


/*****************************************************************************/
/*                                                                           */
/*                        PrintARStatisticsResultStruct                      */
/*                                                                           */
/*****************************************************************************/

void PrintARStatisticsResultStruct(header, value)
char                     *header;  /* IN; header string for the value */
ARStatisticsResultStruct *value;   /* IN; value to print */

{
   char   buff[55];           /* buffer to build nested header in */
   int    i;                  /* working index */

   if (value->groupByValues.numItems != 0)
   {
      if (strlen(header) >= (size_t) 40)
         (void) strcpy(buff, "Grouped by ");
      else
         (void) sprintf(buff, "%sGrouped by ", header);

      for (i = 0; i < (int) value->groupByValues.numItems; i++)
      {
         PrintARValueStruct(buff, &value->groupByValues.valueList[i]);

         if (i == 0)
            if (strlen(header) >= (size_t) 40)
               (void) strcpy(buff, " then by ");
            else
               (void) sprintf(buff, "%s then by ", header);
      }
   }

   if (strlen(header) >= (size_t) 40)
      (void) strcpy(buff, "   Stat ");
   else
      (void) sprintf(buff, "%s   Stat ", header);
   PrintARValueStruct(buff, &value->result);
}


/*****************************************************************************/
/*                                                                           */
/*                        PrintARStatisticsResultList                        */
/*                                                                           */
/*****************************************************************************/

void PrintARStatisticsResultList(header, value)
char                   *header;  /* IN; header string for the value */
ARStatisticsResultList *value;   /* IN; value to print */

{
   char                      buff[45];  /* buffer to build nested header in */
   int                       i;         /* working index */
   ARStatisticsResultStruct *tempPtr;   /* working pointer */

   DriverPrint("%sStatistics Result List : %u items\n", header,
               value->numItems);
   if (value->numItems != 0)
   {
      if (strlen(header) >= (size_t) 40)
         buff[0] = '\0';
      else
         (void) sprintf(buff, "%s   ", header);

      tempPtr = value->resultList;
      for (i = 0; i < (int) value->numItems; i++)
      {
         PrintARStatisticsResultStruct(buff, tempPtr);
         tempPtr++;
      }
   }
}


/*****************************************************************************/
/*                                                                           */
/*                              PrintARPropList                              */
/*                                                                           */
/*****************************************************************************/

void PrintARPropList(header, prefix, value)
char       *header;      /* IN; header string for the value */
char       *prefix;      /* IN; prefix to Prop list on first line */
ARPropList *value;       /* IN; value to print */

{
   char          buff[45];    /* buffer to build nested header in */
   int           i;           /* working index */
   ARPropStruct *tempPtr;     /* working pointer */

   if (strlen(header) >= (size_t) 40)
      buff[0] = '\0';
   else
      (void) sprintf(buff, "%s   ", header);

   DriverPrint("%s%sProp List : %u items\n", header, prefix, value->numItems);
   if (value->numItems != 0)
   {
      tempPtr = value->props;
      for (i = 0; i < (int) value->numItems; i++)
      {
         DriverPrint("%sProp Struct:  ", header);
         switch (tempPtr->prop)
         {
            case AR_DPROP_TRIM_TYPE       :
               DriverPrint("TRIM TYPE\n");
               break;
            case AR_DPROP_CNTL_TYPE       :
               DriverPrint("CNTL TYPE\n");
               break;
            case AR_DPROP_BBOX            :
               DriverPrint("BBOX\n");
               break;
            case AR_DPROP_VISIBLE         :
               DriverPrint("VISIBLE\n");
               break;
            case AR_DPROP_ENABLE          :
               DriverPrint("ENABLE\n");
               break;
            case AR_DPROP_HELP            :
               DriverPrint("HELP\n");
               break;
            case AR_DPROP_Z_ORDER         :
               DriverPrint("Z ORDER\n");
               break;
            case AR_DPROP_COLOR_FILL      :
               DriverPrint("COLOR FILL\n");
               break;
            case AR_DPROP_DEPTH_EFFECT    :
               DriverPrint("DEPTH EFFECT\n");
               break;
            case AR_DPROP_DEPTH_AMOUNT    :
               DriverPrint("DEPTH AMOUNT\n");
               break;
            case AR_DPROP_COLOR_LINE      :
               DriverPrint("COLOR LINE\n");
               break;
            case AR_DPROP_COLOR_TEXT      :
               DriverPrint("COLOR TEXT\n");
               break;
            case AR_DPROP_PROMPT          :
               DriverPrint("PROMPT\n");
               break;
            case AR_DPROP_LABEL           :
               DriverPrint("LABEL\n");
               break;
            case AR_DPROP_LABEL_BBOX      :
               DriverPrint("LABEL BBOX\n");
               break;
            case AR_DPROP_LABEL_FONT_STYLE:
               DriverPrint("LABEL FONT STYLE\n");
               break;
            case AR_DPROP_LABEL_FONT_SIZE :
               DriverPrint("LABEL FONT SIZE\n");
               break;
            case AR_DPROP_LABEL_COLOR_TEXT:
               DriverPrint("LABEL COLOR TEXT\n");
               break;
            case AR_DPROP_LABEL_JUSTIFY   :
               DriverPrint("LABEL JUSTIFY\n");
               break;
            case AR_DPROP_LABEL_ALIGN     :
               DriverPrint("LABEL ALIGN\n");
               break;
            case AR_DPROP_LABEL_POS_SECTOR :
               DriverPrint("LABEL POS SECTOR\n");
               break;
            case AR_DPROP_LABEL_POS_JUSTIFY:
               DriverPrint("LABEL POS JUSTIFY\n");
               break;
            case AR_DPROP_LABEL_POS_ALIGN :
               DriverPrint("LABEL POS ALIGN\n");
               break;
            case AR_DPROP_LABEL_COLOR_FILL:
               DriverPrint("LABEL COLOR FILL\n");
               break;
            case AR_DPROP_LABEL_COLOR_LINE:
               DriverPrint("LABEL COLOR LINE\n");
               break;
            case AR_DPROP_COORDS          :
               DriverPrint("COORDS\n");
               break;
            case AR_DPROP_LINE_WIDTH      :
               DriverPrint("LINE WIDTH\n");
               break;
            case AR_DPROP_LINE_PATTERN    :
               DriverPrint("LINE PATTERN\n");
               break;
            case AR_DPROP_JOINT_STYLE     :
               DriverPrint("JOINT STYLE\n");
               break;
            case AR_DPROP_ENDCAP_START    :
               DriverPrint("ENDCAP START\n");
               break;
            case AR_DPROP_ENDCAP_END      :
               DriverPrint("ENDCAP END\n");
               break;
            case AR_DPROP_DATA_ROWS       :
               DriverPrint("DATA ROWS\n");
               break;
            case AR_DPROP_DATA_COLS       :
               DriverPrint("DATA COLS\n");
               break;
            case AR_DPROP_DATA_SPIN       :
               DriverPrint("DATA SPIN\n");
               break;
            case AR_DPROP_DATA_MENU       :
               DriverPrint("DATA MENU\n");
               break;
            case AR_DPROP_DATA_RADIO      :
               DriverPrint("DATA RADIO\n");
               break;
            case AR_DPROP_DATA_MENU_BBOX  :
               DriverPrint("MENU BBOX\n");
               break;
            case AR_DPROP_DATA_EXPAND_BBOX:
               DriverPrint("EXPAND BBOX\n");
               break;
            case AR_DPROP_TEXT            :
               DriverPrint("TEXT\n");
               break;
            case AR_DPROP_TEXT_FONT_STYLE :
               DriverPrint("TEXT FONT STYLE\n");
               break;
            case AR_DPROP_TEXT_FONT_SIZE  :
               DriverPrint("TEXT FONT SIZE\n");
               break;
            case AR_DPROP_JUSTIFY         :
               DriverPrint("JUSTIFY\n");
               break;
            case AR_DPROP_ALIGN           :
               DriverPrint("ALIGN\n");
               break;
            case AR_DPROP_IMAGE           :
               DriverPrint("IMAGE\n");
               break;
            case AR_DPROP_PUSH_BUTTON_IMAGE:
               DriverPrint("PUSH BUTTON IMAGE\n");
               break;
            case AR_DPROP_BUTTON_TEXT     :
               DriverPrint("BUTTON TEXT\n");
               break;
            case AR_DPROP_MENU_TEXT       :
               DriverPrint("MENU TEXT\n");
               break;
            case AR_DPROP_MENU_POS        :
               DriverPrint("MENU POS\n");
               break;
            case AR_DPROP_MENU_MODE       :
               DriverPrint("MENU MODE\n");
               break;
            case AR_DPROP_MENU_PARENT     :
               DriverPrint("MENU PARENT\n");
               break;
            case AR_DPROP_MENU_HELP       :
               DriverPrint("MENU HELP\n");
               break;
            case AR_DPROP_TOOLTIP         :
               DriverPrint("TOOLTIP\n");
               break;
            case AR_DPROP_TOOLBAR_POS     :
               DriverPrint("TOOLBAR POS\n");
               break;
            case AR_DPROP_TOOLBAR_MODE    :
               DriverPrint("TOOLBAR MODE\n");
               break;
            case AR_DPROP_TOOLBAR_TEXT    :
               DriverPrint("TOOLBAR TEXT\n");
               break;
            case AR_DPROP_TAB_MODE        :
               DriverPrint("TAB MODE\n");
               break;
            case AR_DPROP_TAB_COORD       :
               DriverPrint("TAB COORD\n");
               break;
            case AR_DPROP_TAB_TEXT        :
               DriverPrint("TAB TEXT\n");
               break;
            case AR_DPROP_TAB_NEXT        :
               DriverPrint("TAB NEXT\n");
               break;
            case AR_DPROP_VIEW_GRID_BBOX  :
               DriverPrint("VIEW GRID BBOX\n");
               break;
            case AR_DPROP_VUI_DEFAULT     :
               DriverPrint("VUI DEFAULT\n");
               break;
            default                       :
               DriverPrint("   <undefined property = %u>\n", tempPtr->prop);
               break;
         }
         PrintARValueStruct(buff, &tempPtr->value);
         tempPtr++;
      }
   }
}


/*****************************************************************************/
/*                                                                           */
/*                             PrintARDisplayInstanceList                    */
/*                                                                           */
/*****************************************************************************/

void PrintARDisplayInstanceList(value)
ARDisplayInstanceList  *value;        /* IN; value to print */

{
   int                      i;        /* working index */
   ARDisplayInstanceStruct *tempPtr;  /* working pointer */

   DriverPrint("Display Instance List : %u items\n", value->numItems);
   if (value->numItems != 0)
   {
      PrintARPropList("   ", "Common ", &value->commonProps);
      tempPtr = value->dInstanceList;
      for (i = 0; i < (int) value->numItems; i++)
      {
         DriverPrint("   Display Instance : #%d for VUI %lu\n", i,
                     tempPtr->vui);
         PrintARPropList("   ", "Instance ", &tempPtr->props);
         tempPtr++;
      }
   }
}


/*****************************************************************************/
/*                                                                           */
/*                          PrintARAssignFieldStruct                         */
/*                                                                           */
/*****************************************************************************/

void PrintARAssignFieldStruct(header, value)
char                *header;  /* IN; header string for the value */
ARAssignFieldStruct *value;   /* IN; value to print */

{
   char   buff[45];           /* buffer to build nested header in */

   if (strlen(header) >= (size_t) 40)
      buff[0] = '\0';
   else
      (void) sprintf(buff, "%s   ", header);

   DriverPrint("%s", buff);
   PrintChar("Server: ", value->server);
   DriverPrint("%s", buff);
   PrintChar("Schema: ", value->schema);

   switch (value->tag)
   {
      case AR_FIELD        :
         DriverPrint("%sTag: FIELD\n", buff);
         DriverPrint("%s", buff);
         PrintARInternalId("Field Id:", value->u.fieldId);
         break;
      case AR_STAT_HISTORY :
         DriverPrint("%sTag: STAT_HISTORY\n", buff);
         DriverPrint("%s", buff);
         PrintLong("Enumerated value index:",(int)value->u.statHistory.enumVal);
         switch (value->u.statHistory.userOrTime)
         {
            case AR_STAT_HISTORY_USER :
               DriverPrint("%sUser or Time: USER\n", buff);
               break;
            case AR_STAT_HISTORY_TIME :
               DriverPrint("%sUser or Time: TIME\n", buff);
               break;
            default                   :
               DriverPrint("%sUser or Time: %s%u>\n", buff,
                           "<undefined value = ",
                           value->u.statHistory.userOrTime);
               break;
         }
         break;
      default              :
         DriverPrint("%sTag: <undefined tag = %u>\n", buff, value->tag);
         break;
   }

   PrintARQualifierStruct(header, &value->qualifier);
   switch (value->noMatchOption)
   {
      case AR_NO_MATCH_ERROR    :
         DriverPrint("%sNo Match Option: ERROR\n", header);
         break;
      case AR_NO_MATCH_SET_NULL :
         DriverPrint("%sNo Match Option: NULL\n", header);
         break;
      case AR_NO_MATCH_NO_ACTION:
         DriverPrint("%sNo Match Option: NO ACTION\n", header);
         break;
      case AR_NO_MATCH_SUBMIT   :
         DriverPrint("%sNo Match Option: SUBMIT\n", header);
         break;
      default                   :
         DriverPrint("%sNo Match Option: %s%u>\n", header,
                     "<undefined value = ", value->noMatchOption);
         break;
   }
   switch (value->multiMatchOption)
   {
      case AR_MULTI_MATCH_ERROR    :
         DriverPrint("%sMulti Match Option: ERROR\n", header);
         break;
      case AR_MULTI_MATCH_SET_NULL :
         DriverPrint("%sMulti Match Option: NULL\n", header);
         break;
      case AR_MULTI_MATCH_USE_FIRST:
         DriverPrint("%sMulti Match Option: FIRST\n", header);
         break;
      case AR_MULTI_MATCH_PICKLIST :
         DriverPrint("%sMulti Match Option: PICKLIST\n", header);
         break;
      case AR_MULTI_MATCH_MODIFY_ALL :
         DriverPrint("%sMulti Match Option: MODIFY ALL\n", header);
         break;
      case AR_MULTI_MATCH_NO_ACTION :
         DriverPrint("%sMulti Match Option: NO ACTION\n", header);
         break;
      default                      :
         DriverPrint("%sMulti Match Option: %s%u>\n", header,
                     "<undefined value = ", value->multiMatchOption);
         break;
   }
}


/*****************************************************************************/
/*                                                                           */
/*                          PrintARAssignSQLStruct                           */
/*                                                                           */
/*****************************************************************************/

void PrintARAssignSQLStruct(header, value)
char                *header;  /* IN; header string for the value */
ARAssignSQLStruct   *value;   /* IN; value to print */

{
   char   buff[45];           /* buffer to build nested header in */

   if (strlen(header) >= (size_t) 40)
      buff[0] = '\0';
   else
      (void) sprintf(buff, "%s   ", header);

   DriverPrint("%s", buff);
   PrintChar("Server: ", value->server);
   DriverPrint("%s", buff);
   if (value->sqlCommand == NULL)
      DriverPrint("SQL Command: NULL");
   else
      PrintChar("SQL Command: ", value->sqlCommand);

   DriverPrint("%s", buff);
   PrintInt("Value Index:", value->valueIndex);

   switch (value->noMatchOption)
   {
      case AR_NO_MATCH_ERROR    :
         DriverPrint("%sNo Match Option: ERROR\n", buff);
         break;
      case AR_NO_MATCH_SET_NULL :
         DriverPrint("%sNo Match Option: NULL\n", buff);
         break;
      default                   :
         DriverPrint("%sNo Match Option: %s%u>\n", buff,
                     "<undefined value = ", value->noMatchOption);
         break;
   }
   switch (value->multiMatchOption)
   {
      case AR_MULTI_MATCH_ERROR    :
         DriverPrint("%sMulti Match Option: ERROR\n", buff);
         break;
      case AR_MULTI_MATCH_SET_NULL :
         DriverPrint("%sMulti Match Option: NULL\n", buff);
         break;
      case AR_MULTI_MATCH_USE_FIRST:
         DriverPrint("%sMulti Match Option: FIRST\n", buff);
         break;
      case AR_MULTI_MATCH_PICKLIST :
         DriverPrint("%sMulti Match Option: PICKLIST\n", buff);
         break;
      default                      :
         DriverPrint("%sMulti Match Option: %s%u>\n", buff,
                     "<undefined value = ", value->multiMatchOption);
         break;
   }
}


/*****************************************************************************/
/*                                                                           */
/*                            PrintARAssignStruct                            */
/*                                                                           */
/*****************************************************************************/

void PrintARAssignStruct(header, value)
char           *header;       /* IN; header string for the value */
ARAssignStruct *value;        /* IN; value to print */

{
   char   buff[45];           /* buffer to build nested header in */

   if (strlen(header) >= (size_t) 40)
      buff[0] = '\0';
   else
      (void) sprintf(buff, "%s   ", header);

   DriverPrint("%sAssign Value: ", buff);
   switch (value->assignType)
   {
      case AR_ASSIGN_TYPE_NONE    :
         DriverPrint("  NONE\n");
         break;
      case AR_ASSIGN_TYPE_VALUE   :
         DriverPrint("  VALUE\n");
         PrintARValueStruct(buff, &value->u.value);
         break;
      case AR_ASSIGN_TYPE_FIELD   :
         DriverPrint("  FIELD\n");
         PrintARAssignFieldStruct(buff, value->u.field);
         break;
      case AR_ASSIGN_TYPE_PROCESS :
         DriverPrint("  PROCESS\n");
         DriverPrint("%s", buff);
         PrintChar("Process: ", value->u.process);
         break;
      case AR_ASSIGN_TYPE_ARITH   :
         DriverPrint("  ARITH\n");
         PrintARArithOpAssignStruct(buff, value->u.arithOp);
         break;
      case AR_ASSIGN_TYPE_FUNCTION:
         DriverPrint("  FUNCTION\n");
         PrintARFunctionAssignStruct(buff, value->u.function);
         break;
      case AR_ASSIGN_TYPE_DDE     :
         DriverPrint("  DDE\n");
         PrintARDDEStruct(buff, value->u.dde);
         break;
      case AR_ASSIGN_TYPE_SQL     :
         DriverPrint("  SQL\n");
         PrintARAssignSQLStruct(buff, value->u.sql);
         break;
      default                   :
         DriverPrint("  <undefined assign type tag = %u>\n", value->assignType);
         break;
   }
}


/*****************************************************************************/
/*                                                                           */
/*                          PrintARFieldAssignStruct                         */
/*                                                                           */
/*****************************************************************************/

void PrintARFieldAssignStruct(value)
ARFieldAssignStruct *value;   /* IN; value to print */

{
   DriverPrint("Field Assign Struct:\n");
   PrintARInternalId("   Field Id :", value->fieldId);
   PrintARAssignStruct("", &value->assignment);
}


/*****************************************************************************/
/*                                                                           */
/*                          PrintARFieldAssignList                           */
/*                                                                           */
/*****************************************************************************/

void PrintARFieldAssignList(value)
ARFieldAssignList *value;     /* IN; value to print */

{
   int                  i;      /* working index */
   ARFieldAssignStruct *tempPtr;/* working pointer */

   DriverPrint("Field Assign List : %u items\n", value->numItems);
   if (value->numItems != 0)
   {
      tempPtr = value->fieldAssignList;
      for (i = 0; i < (int) value->numItems; i++)
      {
         PrintARFieldAssignStruct(tempPtr);
         tempPtr++;
      }
   }
}


/*****************************************************************************/
/*                                                                           */
/*                        PrintARArithOpAssignStruct                         */
/*                                                                           */
/*****************************************************************************/

void PrintARArithOpAssignStruct(header, value)
char                  *header; /* IN; header string for the value */
ARArithOpAssignStruct *value;  /* IN; value to print */

{
   char   buff[45];           /* buffer to build nested header in */

   if (strlen(header) >= (size_t) 40)
      buff[0] = '\0';
   else
      (void) sprintf(buff, "%s   ", header);

   DriverPrint("%sArithmetic Operation:\n", header);
   switch (value->operation)
   {
      case AR_ARITH_OP_ADD      :
         DriverPrint("%s   Operation: ADD\n", header);
         break;
      case AR_ARITH_OP_SUBTRACT :
         DriverPrint("%s   Operation: SUBTRACT\n", header);
         break;
      case AR_ARITH_OP_MULTIPLY :
         DriverPrint("%s   Operation: MULTIPLY\n", header);
         break;
      case AR_ARITH_OP_DIVIDE   :
         DriverPrint("%s   Operation: DIVIDE\n", header);
         break;
      case AR_ARITH_OP_MODULO   :
         DriverPrint("%s   Operation: MODULO\n", header);
         break;
      case AR_ARITH_OP_NEGATE   :
         DriverPrint("%s   Operation: NEGATE (unary minus)\n", header);
         break;
      default                   :
         DriverPrint("%s   Operation: <undefined operation = %u>\n", header,
                     value->operation);
         break;
   }
   if (value->operation != AR_ARITH_OP_NEGATE)
      PrintARAssignStruct(buff, &value->operandLeft);
   PrintARAssignStruct(buff, &value->operandRight);
}


/*****************************************************************************/
/*                                                                           */
/*                        PrintARFunctionAssignStruct                        */
/*                                                                           */
/*****************************************************************************/

void PrintARFunctionAssignStruct(header, value)
char                   *header; /* IN; header string for the value */
ARFunctionAssignStruct *value;  /* IN; value to print */

{
   char   buff[45];           /* buffer to build nested header in */
   int    i;                  /* working index */

   if (strlen(header) >= (size_t) 40)
      buff[0] = '\0';
   else
      (void) sprintf(buff, "%s   ", header);

   DriverPrint("%sFunction:\n", header);
   switch (value->functionCode)
   {
      case AR_FUNCTION_DATE     :
         DriverPrint("%s   Function code: DATE\n", header);
         break;
      case AR_FUNCTION_TIME     :
         DriverPrint("%s   Function code: TIME\n", header);
         break;
      case AR_FUNCTION_MONTH    :
         DriverPrint("%s   Function code: MONTH\n", header);
         break;
      case AR_FUNCTION_DAY      :
         DriverPrint("%s   Function code: DAY\n", header);
         break;
      case AR_FUNCTION_YEAR     :
         DriverPrint("%s   Function code: YEAR\n", header);
         break;
      case AR_FUNCTION_WEEKDAY  :
         DriverPrint("%s   Function code: WEEKDAY\n", header);
         break;
      case AR_FUNCTION_HOUR     :
         DriverPrint("%s   Function code: HOUR\n", header);
         break;
      case AR_FUNCTION_MINUTE   :
         DriverPrint("%s   Function code: MINUTE\n", header);
         break;
      case AR_FUNCTION_SECOND   :
         DriverPrint("%s   Function code: SECOND\n", header);
         break;
      case AR_FUNCTION_TRUNC    :
         DriverPrint("%s   Function code: TRUNC\n", header);
         break;
      case AR_FUNCTION_ROUND    :
         DriverPrint("%s   Function code: ROUND\n", header);
         break;
      case AR_FUNCTION_CONVERT  :
         DriverPrint("%s   Function code: CONVERT\n", header);
         break;
      case AR_FUNCTION_LENGTH   :
         DriverPrint("%s   Function code: LENGTH\n", header);
         break;
      case AR_FUNCTION_UPPER    :
         DriverPrint("%s   Function code: UPPER\n", header);
         break;
      case AR_FUNCTION_LOWER    :
         DriverPrint("%s   Function code: LOWER\n", header);
         break;
      case AR_FUNCTION_SUBSTR   :
         DriverPrint("%s   Function code: SUBSTR\n", header);
         break;
      case AR_FUNCTION_LEFT     :
         DriverPrint("%s   Function code: LEFT\n", header);
         break;
      case AR_FUNCTION_RIGHT    :
         DriverPrint("%s   Function code: RIGHT\n", header);
         break;
      case AR_FUNCTION_LTRIM    :
         DriverPrint("%s   Function code: LTRIM\n", header);
         break;
      case AR_FUNCTION_RTRIM    :
         DriverPrint("%s   Function code: RTRIM\n", header);
         break;
      case AR_FUNCTION_LPAD     :
         DriverPrint("%s   Function code: LPAD\n", header);
         break;
      case AR_FUNCTION_RPAD     :
         DriverPrint("%s   Function code: RPAD\n", header);
         break;
      case AR_FUNCTION_REPLACE  :
         DriverPrint("%s   Function code: REPLACE\n", header);
         break;
      case AR_FUNCTION_STRSTR   :
         DriverPrint("%s   Function code: STRSTR\n", header);
         break;
      case AR_FUNCTION_MIN      :
         DriverPrint("%s   Function code: MIN\n", header);
         break;
      case AR_FUNCTION_MAX      :
         DriverPrint("%s   Function code: MAX\n", header);
         break;
      default                   :
         DriverPrint("%s   Function code: <undefined function = %u>\n",
                     header, value->functionCode);
         break;
   }
   DriverPrint("%s   Number parameters: %u\n", header, value->numItems);

   if ((value->numItems != 0) && (value->parameterList != NULL))
      for (i = 0; i < (int) value->numItems; i++)
         PrintARAssignStruct(buff, &value->parameterList[i]);
}


/*****************************************************************************/
/*                                                                           */
/*                          PrintARFilterActionNotify                        */
/*                                                                           */
/*****************************************************************************/

void PrintARFilterActionNotify(value)
ARFilterActionNotify *value;  /* IN; value to print */

{
   DriverPrint("NOTIFY\n");
   PrintChar("   User:", value->user);
   PrintInt("   Priority:", value->notifyPriority);
   PrintChar("   Text:", value->notifyText);
   PrintChar("   Subject:", value->subjectText);
   switch (value->notifyMechanism)
   {
      case AR_NOTIFY_VIA_NOTIFIER :
         DriverPrint("   Mechanism: NOTIFIER\n");
         break;
      case AR_NOTIFY_VIA_EMAIL    :
         DriverPrint("   Mechanism: E-MAIL\n");
         break;
      case AR_NOTIFY_VIA_DEFAULT  :
         DriverPrint("   Mechanism: DEFAULT\n");
         break;
      case AR_NOTIFY_VIA_XREF     :
         DriverPrint("   Mechanism: CROSS-REFERENCE\n");
         DriverPrint("   Cross reference field id: %ld",
                     value->notifyMechanismXRef);
         break;
      default                     :
         DriverPrint("   Mechanism: %u\n", value->notifyMechanism);
         break;
   }
   switch (value->fieldIdListType)
   {
      case AR_FILTER_FIELD_IDS_NONE :
         DriverPrint("   Field Id List: NONE\n");
         break;
      case AR_FILTER_FIELD_IDS_ALL  :
         DriverPrint("   Field Id List: ALL\n");
         break;
      case AR_FILTER_FIELD_IDS_LIST :
         DriverPrint("   Field Id List: LIST\n");
         PrintARInternalIdList("      Field List:", "         ",
                               &value->fieldIdList);
         break;
      case AR_FILTER_FIELD_IDS_CHANGED :
         DriverPrint("   Field Id List: CHANGED\n");
         break;
      default                     :
         DriverPrint("   Field Id List: <undefined type = %u>\n",
                     value->fieldIdListType);
         break;
   }
}

/*****************************************************************************/
/*                                                                           */
/*                           PrintARPushFieldsStruct                         */
/*                                                                           */
/*****************************************************************************/

void PrintARPushFieldsStruct(value)
ARPushFieldsStruct *value;  /* IN; value to print */

{
   DriverPrint("Push Fields Struct:\n");
   PrintARAssignFieldStruct("", &value->field);
   PrintARAssignStruct("", &value->assign);
}

/*****************************************************************************/
/*                                                                           */
/*                            PrintARPushFieldsList                          */
/*                                                                           */
/*****************************************************************************/

void PrintARPushFieldsList(value)
ARPushFieldsList *value;  /* IN; value to print */

{
   unsigned int    i;    /* working index */

   if ((value->numItems != 0) && (value->pushFieldsList != NULL))
   {
      DriverPrint("Push Fields List : %u items\n", value->numItems);
      for (i = 0; i < value->numItems; i++)
         PrintARPushFieldsStruct(&value->pushFieldsList[i]);
   }
}

/*****************************************************************************/
/*                                                                           */
/*                               PrintARSQLStruct                            */
/*                                                                           */
/*****************************************************************************/

void PrintARSQLStruct(value)
ARSQLStruct *value;  /* IN; value to print */

{
   DriverPrint("SQL Struct:\n");
   PrintChar("Server: ", value->server);
   if (value->command == NULL)
      DriverPrint("SQL Command: NULL");
   else
      PrintChar("SQL Command: ", value->command);
}

/*****************************************************************************/
/*                                                                           */
/*                       PrintARGotoActionStruct                             */
/*                                                                           */
/*****************************************************************************/
  
void PrintARGotoActionStruct(value)
ARGotoActionStruct *value;
{
   DriverPrint("   Goto Action Struct:\n");
   PrintInt("      Tag (Field Or Value): ", value->tag);
   if (value->tag == AR_FIELD)
      PrintInt("      FieldId: ", value->fieldIdOrValue);
   else if (value->tag == AR_VALUE)
      PrintInt("      Execution Order: ", value->fieldIdOrValue);
   else
      DriverPrint("<unknown type>\n");
}

/*****************************************************************************/
/*                                                                           */
/*                         PrintARFilterActionStruct                         */
/*                                                                           */
/*****************************************************************************/

void PrintARFilterActionStruct(value)
ARFilterActionStruct *value;  /* IN; value to print */

{
   DriverPrint("Filter Action Struct:  ");
   switch (value->action)
   {
      case AR_FILTER_ACTION_NOTIFY  :
         PrintARFilterActionNotify(&value->u.notify);
         break;
      case AR_FILTER_ACTION_MESSAGE :
         DriverPrint("MESSAGE\n");
         PrintARStatusStruct(&value->u.message);
         break;
      case AR_FILTER_ACTION_LOG     :
         DriverPrint("LOG FILE\n");
         PrintChar("   Filename:", value->u.logFile);
         break;
      case AR_FILTER_ACTION_FIELDS  :
         DriverPrint("SET FIELDS\n");
         PrintARFieldAssignList(&(value->u.fieldList));
         break;
      case AR_FILTER_ACTION_PROCESS :
         DriverPrint("PROCESS\n");
         PrintChar("   Filename:", value->u.process);
         break;
      case AR_FILTER_ACTION_FIELDP  :
         DriverPrint("PUSH FIELDS\n");
         PrintARPushFieldsList(&(value->u.pushFieldsList));
         break;
      case AR_FILTER_ACTION_SQL     :
         DriverPrint("DIRECT SQL\n");
         PrintARSQLStruct(&(value->u.sqlCommand));
         break;
      case AR_FILTER_ACTION_GOTOACTION     :
         DriverPrint("GOTO ACTION\n");
         PrintARGotoActionStruct(&(value->u.gotoAction));
         break;
      default                       :
         DriverPrint("   <undefined action = %u>\n", value->action);
         break;
   }
}


/*****************************************************************************/
/*                                                                           */
/*                         PrintARFilterActionList                           */
/*                                                                           */
/*****************************************************************************/

void PrintARFilterActionList(value, actionFlag)
ARFilterActionList *value;     /* IN; value to print */
ARBoolean           actionFlag;/* IN; TRUE - action; FALSE - else */

{
   int                   i;       /* working index */
   ARFilterActionStruct *tempPtr;/* working pointer */

   if (actionFlag)
      DriverPrint("Filter Action List : %u items\n", value->numItems);
   else
      DriverPrint("Filter Else List : %u items\n", value->numItems);
   if (value->numItems != 0)
   {
      tempPtr = value->actionList;
      for (i = 0; i < (int) value->numItems; i++)
      {
         PrintARFilterActionStruct(tempPtr);
         tempPtr++;
      }
   }
}


/*****************************************************************************/
/*                                                                           */
/*                         PrintARActiveLinkMacroStruct                      */
/*                                                                           */
/*****************************************************************************/

void PrintARActiveLinkMacroStruct(value)
ARActiveLinkMacroStruct *value;  /* IN; value to print */

{
   int   i;                   /* working index */

   DriverPrint("   Active Link Macro Struct:\n");
   PrintChar("      Macro name:", value->macroName);
   PrintChar("      Macro text:", value->macroText);
   DriverPrint("      Macro Parms:  %u\n", value->macroParms.numItems);
   for (i = 0; i < (int) value->macroParms.numItems; i++)
   {
      DriverPrint("      Macro parm (%d): %s", (i + 1),
                  value->macroParms.parms[i].name);
      if (value->macroParms.parms[i].value == NULL)
         DriverPrint("\n");
      else
         DriverPrint("  =  %s\n", value->macroParms.parms[i].value);
   }
}

/*****************************************************************************/
/*                                                                           */
/*                            PrintARMessageStruct                           */
/*                                                                           */
/*****************************************************************************/

void PrintARMessageStruct(value)
ARMessageStruct *value;        /* IN; value to print */

{
   DriverPrint("Message Struct :\n");
   DriverPrint("   Message type : ");
   switch (value->messageType)
   {
      case AR_RETURN_OK         :
         DriverPrint("NOTE\n");
         break;
      case AR_RETURN_WARNING    :
         DriverPrint("WARNING\n");
         break;
      case AR_RETURN_ERROR      :
         DriverPrint("ERROR\n");
         break;
      case AR_RETURN_PROMPT     :
         DriverPrint("PROMPT\n");
         break;
      default                   :
         DriverPrint("<unknown type>\n");
         break;
   }
   PrintLong("   Message number : ", value->messageNum);
   PrintChar("   Message: ", value->messageText);
   PrintARBoolean("   Use Prompting Pane : ", value->usePromptingPane);
}


/*****************************************************************************/
/*                                                                           */
/*                         PrintARFieldCharacteristics                       */
/*                                                                           */
/*****************************************************************************/

void PrintARFieldCharacteristics(value)
ARFieldCharacteristics *value;  /* IN; value to print */

{
   DriverPrint("   Active Link Field Characteristics Struct:\n");
   PrintLong("      Field Id:", value->fieldId);
   if (value->charMenu == NULL)
      DriverPrint("      Char menu: NULL\n");
   else
      PrintChar("      Char menu:", value->charMenu);
   PrintARPropList("      ", "", &value->props);
   PrintInt("      Reset Focus:", value->focus);
   PrintInt("      Access option:", value->accessOption);
}


/*****************************************************************************/
/*                                                                           */
/*                               PrintARDDEStruct                            */
/*                                                                           */
/*****************************************************************************/

void PrintARDDEStruct(header, value)
char          *header;        /* IN; header string for the value */
ARDDEStruct   *value;         /* IN; value to print */

{
   DriverPrint("   Active Link DDE Struct:\n");
   PrintChar("      Service name:", value->serviceName);
   PrintChar("      Topic:", value->topic);
   PrintInt("      Action:", value->action);
   PrintChar("      Path to Program:", value->pathToProgram);
   PrintChar("      Command:", value->command);
   if (value->item != NULL)
      PrintChar("      Item:", value->item);
}


/*****************************************************************************/
/*                                                                           */
/*                               PrintARAutomationStruct                     */
/*                                                                           */
/*****************************************************************************/
 
void PrintARAutomationStruct(header, value)
char                 *header;        /* IN; header string for the value */
ARAutomationStruct   *value;         /* IN; value to print */
 
{
   DriverPrint("   Active Link Outbound Automation Struct:\n");
   PrintChar("      Auto Server name:", value->autoServerName);
   PrintARBoolean("      Automation Server Showing flag :", value->isVisible);
   PrintChar("      Class Id:", value->clsId);
   PrintChar("      Automation Action:", value->action);
   PrintARCOMMethodList(&value->methodList);
}
 
/*****************************************************************************/
/*                                                                           */
/*                               PrintARCOMValueStruct                       */
/*                                                                           */
/*****************************************************************************/
 
void PrintARCOMValueStruct(header, value, indent)
char             *header;       /* IN; header string for the value */
ARCOMValueStruct *value;        /* IN; value to print */
char             *indent;       /* IN; for indentation */
 
{
   char buffer[512];

   DriverPrint("      %sCOM %s Value Struct:\n", indent, header);
   sprintf(buffer, "         %sValue IId:", indent);
   PrintChar(buffer, ((value->valueIId)? value->valueIId : "(NULL)"));
   sprintf(buffer, "         %sTransient Id:", indent);
   PrintLong(buffer, value->transId);

   switch (value->valueType)
   {
      case AR_COM_PARM_NULL:
         sprintf(buffer, "         %sValue: <NULL>", indent);
         PrintChar(buffer, "");
         break;
      case AR_COM_PARM_FIELDID: /* AR_COM_METHOD_FIELDID has same type */
         sprintf(buffer, "         %sFieldId:", indent);
         PrintLong(buffer, value->u.fieldId);
         break;
      case AR_COM_PARM_VALUE:
         sprintf(buffer, "         %s", indent);
         PrintARValueStruct(buffer, &value->u.value);
         break;
      default:
         break;
   }
}


/*****************************************************************************/
/*                                                                           */
/*                               PrintARCOMMethodList                        */
/*                                                                           */
/*****************************************************************************/
 
void PrintARCOMMethodList(value)
ARCOMMethodList     *value;         /* IN; value to print */
 
{
   int                   i;
   ARCOMMethodStruct    *tempPtr;
 
   DriverPrint("    COM Automation Method List:\n");
   if (value->numItems > 0)
   {
      tempPtr = value->methodList;
      for (i = 0; i < (int) value->numItems; i++)
      {
         PrintARCOMMethodStruct(tempPtr);
         tempPtr++;
      }
   }
}
 
 
/*****************************************************************************/
/*                                                                           */
/*                               PrintARCOMMethodStruct                      */
/*                                                                           */
/*****************************************************************************/
 
void PrintARCOMMethodStruct(value)
ARCOMMethodStruct       *value;         /* IN; value to print */
 
{
   DriverPrint("      COM Method Struct:\n");
   PrintChar("      Method Name:", value->methodName);
   PrintChar("      Method Interface Id:", value->methodIId);
   PrintInt("      Method Type:", value->methodType);
   PrintARCOMValueStruct("Method", &value->methodValue, "");
   PrintARCOMMethodParmList(&(value->parameterList));
}
 
 
/*****************************************************************************/
/*                                                                           */
/*                        PrintARCOMMethodParmList                           */
/*                                                                           */
/*****************************************************************************/
 
void PrintARCOMMethodParmList(value)
ARCOMMethodParmList     *value;         /* IN; value to print */
 
{
   int                           i;
   ARCOMMethodParmStruct        *tempPtr;
 
   DriverPrint("      COM Parameter List:\n");
   if (value->numItems > 0)
   {
      tempPtr = value->parameterList;
      for (i = 0; i < (int) value->numItems; i++)
      {
         PrintARCOMMethodParmStruct(tempPtr);
         tempPtr++;
      }
   }
}
 
 
/*****************************************************************************/
/*                                                                           */
/*                               PrintARCOMMethodParmStruct                  */
/*                                                                           */
/*****************************************************************************/
 
void PrintARCOMMethodParmStruct(value)
ARCOMMethodParmStruct   *value;         /* IN; value to print */
 
{
   DriverPrint("         COM Parameter Struct:\n");
   PrintChar("         Parameter Name:", value->parmName);
   PrintInt("         Parameter Type:", value->parmType);
   PrintARCOMValueStruct("Parameter", &value->parmValue, "   ");
}


/*****************************************************************************/
/*                                                                           */
/*                             PrintAROpenDlgStruct                          */
/*                                                                           */
/*****************************************************************************/

void PrintAROpenDlgStruct(value)
AROpenDlgStruct   *value;      /* OUT; loaded with value for buffer */
 
{
   DriverPrint("   Active Link Open Dialogue Struct:\n");
   PrintChar("      Server Name: ", value->serverName);
   PrintChar("      Schema Name: ", value->schemaName);
   PrintChar("      Vui Label: ", value->vuiLabel);
   PrintARBoolean("      Close Box Showing flag : ", value->closeBox);
   PrintARFieldAssignList(&value->inputValueFieldPairs);
   PrintARFieldAssignList(&value->outputValueFieldPairs);
}

  
/*****************************************************************************/
/*                                                                           */
/*                             PrintARCallGuideStruct                        */
/*                                                                           */
/*****************************************************************************/
  
void PrintARCallGuideStruct(value)
ARCallGuideStruct *value;

{
   DriverPrint("   Active Link Call Guide Struct:\n");
   PrintChar("      Server Name: ", value->serverName);
   PrintChar("      Guide Name: ", value->guideName);
   PrintInt("      Guide Mode: ", value->guideMode);
}


/*****************************************************************************/
/*                                                                           */
/*                             PrintARExitGuideStruct                        */
/*                                                                           */
/*****************************************************************************/
  
void PrintARExitGuideStruct(value)
ARExitGuideStruct *value;

{
   DriverPrint("   Active Link Exit Guide Struct:\n");
   PrintARBoolean("      Close All Windows : ", value->closeAll);
}


/*****************************************************************************/
/*                                                                           */
/*                             PrintARGotoGuideLabel                         */
/*                                                                           */
/*****************************************************************************/
  
void PrintARGotoGuideLabel(value)
ARGotoGuideLabelStruct *value;

{
   DriverPrint("   Active Link Goto Guide Label Struct:\n");
   PrintChar("      Label: ", value->label);
}


/*****************************************************************************/
/*                                                                           */
/*                             PrintARWaitStruct                             */
/*                                                                           */
/*****************************************************************************/
  
void PrintARWaitStruct(value)
ARWaitStruct *value;

{
   DriverPrint("   Active Link Wait Struct:\n");
   PrintChar("      Continue Button Title: ", value->continueButtonTitle);
}


/*****************************************************************************/
/*                                                                           */
/*                        PrintARActiveLinkActionStruct                      */
/*                                                                           */
/*****************************************************************************/

void PrintARActiveLinkActionStruct(value)
ARActiveLinkActionStruct *value;  /* IN; value to print */

{
   DriverPrint("Active Link Action Struct:  ");
   switch (value->action)
   {
      case AR_ACTIVE_LINK_ACTION_MACRO   :
         DriverPrint("MACRO\n");
         PrintARActiveLinkMacroStruct(&value->u.macro);
         break;
      case AR_ACTIVE_LINK_ACTION_FIELDS  :
         DriverPrint("FIELDS\n");
         PrintARFieldAssignList(&value->u.fieldList);
         break;
      case AR_ACTIVE_LINK_ACTION_PROCESS :
         DriverPrint("PROCESS\n");
         PrintChar("   Process:", value->u.process);
         break;
      case AR_ACTIVE_LINK_ACTION_MESSAGE :
         DriverPrint("MESSAGE\n");
         PrintARMessageStruct(&value->u.message);
         break;
      case AR_ACTIVE_LINK_ACTION_SET_CHAR:
         DriverPrint("SET FIELD CHARACTERISTICS\n");
         PrintARFieldCharacteristics(&value->u.characteristics);
         break;
      case AR_ACTIVE_LINK_ACTION_DDE     :
         DriverPrint("DDE\n");
         PrintARDDEStruct("      ", &value->u.dde);
         break;
      case AR_ACTIVE_LINK_ACTION_FIELDP  :
         DriverPrint("PUSH FIELDS\n");
         PrintARPushFieldsList(&(value->u.pushFieldsList));
         break;
      case AR_ACTIVE_LINK_ACTION_SQL     :
         DriverPrint("DIRECT SQL\n");
         PrintARSQLStruct(&(value->u.sqlCommand));
         break;
      case AR_ACTIVE_LINK_ACTION_AUTO    :
         DriverPrint("Automation\n");
         PrintARAutomationStruct("      ", &value->u.automation);
         break;
      case AR_ACTIVE_LINK_ACTION_OPENDLG   :
         DriverPrint("Open Dialogue\n");
         PrintAROpenDlgStruct(&value->u.openDlg);
         break;
      case AR_ACTIVE_LINK_ACTION_COMMITC   :
         DriverPrint("Commit Changes\n");
         break;
      case AR_ACTIVE_LINK_ACTION_CLOSEWND :
         DriverPrint("Close Window\n");
         break;
      case AR_ACTIVE_LINK_ACTION_CALLGUIDE  :
         DriverPrint("Call Guide\n");
         PrintARCallGuideStruct(&value->u.callGuide);
         break;
      case AR_ACTIVE_LINK_ACTION_EXITGUIDE  :
         DriverPrint("Exit Guide\n");
         PrintARExitGuideStruct(&value->u.exitGuide);
         break;
      case AR_ACTIVE_LINK_ACTION_GOTOGUIDELABEL :
         DriverPrint("Goto Guide\n");
         PrintARGotoGuideLabel(&value->u.gotoGuide);
         break;
      case AR_ACTIVE_LINK_ACTION_WAIT:
         DriverPrint("Wait Guide\n");
         PrintARWaitStruct(&value->u.waitAction);
         break;
      case AR_ACTIVE_LINK_ACTION_GOTOACTION:
         DriverPrint("Goto Action\n");
         PrintARGotoActionStruct(&value->u.gotoAction);
         break;
      default          :
         DriverPrint("   <undefined action = %u>\n", value->action);
         break;
   }
}


/*****************************************************************************/
/*                                                                           */
/*                         PrintARActiveLinkActionList                       */
/*                                                                           */
/*****************************************************************************/

void PrintARActiveLinkActionList(value, actionFlag)
ARActiveLinkActionList *value;     /* IN; value to print */
ARBoolean               actionFlag;/* IN; TRUE - action; FALSE - else */

{
   int                       i;      /* working index */
   ARActiveLinkActionStruct *tempPtr;/* working pointer */

   if (actionFlag)
      DriverPrint("Active Link Action List : %u items\n", value->numItems);
   else
      DriverPrint("Active Link Else List : %u items\n", value->numItems);
   if (value->numItems != 0)
   {
      tempPtr = value->actionList;
      for (i = 0; i < (int) value->numItems; i++)
      {
         PrintARActiveLinkActionStruct(tempPtr);
         tempPtr++;
      }
   }
}


/*****************************************************************************/
/*                                                                           */
/*                         PrintARPermissionStruct                           */
/*                                                                           */
/*****************************************************************************/

void PrintARPermissionStruct(value, fieldFlag)
ARPermissionStruct *value;    /* IN; value to print */
ARBoolean           fieldFlag;/* IN; TRUE - field; FALSE - schema */

{
   DriverPrint("Permission Struct:\n");
   PrintARInternalId("   Group id:", value->groupId);
   switch (value->permissions)
   {
      case AR_PERMISSIONS_NONE   :
         DriverPrint("   Permissions:  NONE\n");
         break;
      case AR_PERMISSIONS_VIEW   :
         if (fieldFlag)
            DriverPrint("   Permissions:  VIEW\n");
         else
            DriverPrint("   Permissions:  VISIBLE\n");
         break;
      case AR_PERMISSIONS_CHANGE :
         if (fieldFlag)
            DriverPrint("   Permissions:  CHANGE\n");
         else
            DriverPrint("   Permissions:  HIDDEN\n");
         break;
      default                    :
         DriverPrint("   Permissions: <undefined permission = %u>\n",
                     value->permissions);
         break;
   }
}


/*****************************************************************************/
/*                                                                           */
/*                         PrintARPermissionList                             */
/*                                                                           */
/*****************************************************************************/

void PrintARPermissionList(value, fieldFlag)
ARPermissionList *value;      /* IN; value to print */
ARBoolean         fieldFlag;  /* IN; TRUE - field; FALSE - schema */

{
   int                 i;      /* working index */
   ARPermissionStruct *tempPtr;/* working pointer */

   DriverPrint("Permission List : %u items\n", value->numItems);
   if (value->numItems != 0)
   {
      tempPtr = value->permissionList;
      for (i = 0; i < (int) value->numItems; i++)
      {
         PrintARPermissionStruct(tempPtr, fieldFlag);
         tempPtr++;
      }
   }
}


/*****************************************************************************/
/*                                                                           */
/*                         PrintARGroupInfoStruct                            */
/*                                                                           */
/*****************************************************************************/

void PrintARGroupInfoStruct(value)
ARGroupInfoStruct  *value;    /* IN; value to print */

{
   DriverPrint("   Group Info Struct:\n");
   PrintARInternalId("      Group id:", value->groupId);
   switch (value->groupType)
   {
      case 0                    :
         DriverPrint("      Group type:  NONE\n");
         break;
      case AR_GROUP_TYPE_VIEW   :
         DriverPrint("      Group type:  VIEW\n");
         break;
      case AR_GROUP_TYPE_CHANGE :
         DriverPrint("      Group type:  CHANGE\n");
         break;
      default                    :
         DriverPrint("      Group type:  <undefined group type = %u>\n",
                     value->groupType);
         break;
   }
   PrintARNameList("      Group name(s): ", "      ", &value->groupName);
}


/*****************************************************************************/
/*                                                                           */
/*                          PrintARGroupInfoList                             */
/*                                                                           */
/*****************************************************************************/

void PrintARGroupInfoList(value)
ARGroupInfoList *value;       /* IN; value to print */

{
   int                i;       /* working index */
   ARGroupInfoStruct *tempPtr; /* working pointer */

   DriverPrint("Group Info List : %u items\n", value->numItems);
   if (value->numItems != 0)
   {
      tempPtr = value->groupList;
      for (i = 0; i < (int) value->numItems; i++)
      {
         PrintARGroupInfoStruct(tempPtr);
         tempPtr++;
      }
   }
}


/*****************************************************************************/
/*                                                                           */
/*                           PrintARUserLicenseStruct                        */
/*                                                                           */
/*****************************************************************************/

void PrintARUserLicenseStruct(value)
ARUserLicenseStruct *value;   /* IN; value to print */

{
   char *tempPtr;             /* working pointer */
   char *tempPtr2;            /* working pointer */

   DriverPrint("         User License Struct:  ");
   switch (value->licenseTag)
   {
      case AR_LICENSE_TAG_WRITE          :
         DriverPrint("WRITE\n");
         switch (value->licenseType)
         {
            case AR_LICENSE_TYPE_NONE :
               tempPtr = "READ";
               break;
            case AR_LICENSE_TYPE_FIXED :
               tempPtr = "FIXED";
               break;
            case AR_LICENSE_TYPE_FLOATING :
               tempPtr = "FLOATING";
               break;
            default                    :
               tempPtr = "<undefined>";
               break;
         }
         switch (value->currentLicenseType)
         {
            case AR_LICENSE_TYPE_NONE :
               tempPtr2 = "READ";
               break;
            case AR_LICENSE_TYPE_FIXED :
               tempPtr2 = "FIXED";
               break;
            case AR_LICENSE_TYPE_FLOATING :
               tempPtr2 = "FLOATING";
               break;
            default                    :
               tempPtr2 = "<undefined>";
               break;
         }
         break;
      case AR_LICENSE_TAG_FULL_TEXT      :
         DriverPrint("FULL TEXT\n");
         switch (value->licenseType)
         {
            case AR_LICENSE_TYPE_NONE :
               tempPtr = "NONE";
               break;
            case AR_LICENSE_TYPE_FIXED :
               tempPtr = "FIXED";
               break;
            case AR_LICENSE_TYPE_FLOATING :
               tempPtr = "FLOATING";
               break;
            default                    :
               tempPtr = "<undefined>";
               break;
         }
         switch (value->currentLicenseType)
         {
            case AR_LICENSE_TYPE_NONE :
               tempPtr2 = "NONE";
               break;
            case AR_LICENSE_TYPE_FIXED :
               tempPtr2 = "FIXED";
               break;
            case AR_LICENSE_TYPE_FLOATING :
               tempPtr2 = "FLOATING";
               break;
            default                    :
               tempPtr2 = "<undefined>";
               break;
         }
         break;
      case AR_LICENSE_TAG_RESERVED1     :
         DriverPrint("RESERVED\n");
         switch (value->licenseType)
         {
            case AR_LICENSE_TYPE_NONE :
               tempPtr = "NONE";
               break;
            case AR_LICENSE_TYPE_FIXED :
               tempPtr = "FIXED";
               break;
            case AR_LICENSE_TYPE_FLOATING :
               tempPtr = "FLOATING";
               break;
            default                    :
               tempPtr = "<undefined>";
               break;
         }
         switch (value->currentLicenseType)
         {
            case AR_LICENSE_TYPE_NONE :
               tempPtr2 = "NONE";
               break;
            case AR_LICENSE_TYPE_FIXED :
               tempPtr2 = "FIXED";
               break;
            case AR_LICENSE_TYPE_FLOATING :
               tempPtr2 = "FLOATING";
               break;
            default                    :
               tempPtr2 = "<undefined>";
               break;
         }
         break;
      default                       :
         DriverPrint("   <undefined operation = %u>\n", value->licenseTag);
         tempPtr = "";
         tempPtr2 = "";
         break;
   }
   PrintChar("            Assigned License Type:", tempPtr);
   PrintChar("            Current License Type:", tempPtr2);
}


/*****************************************************************************/
/*                                                                           */
/*                         PrintARUserLicenseList                            */
/*                                                                           */
/*****************************************************************************/

void PrintARUserLicenseList(value)
ARUserLicenseList *value;       /* IN; value to print */

{
   int                  i;       /* working index */
   ARUserLicenseStruct *tempPtr; /* working pointer */

   DriverPrint("      User License List : %u items\n", value->numItems);
   if (value->numItems != 0)
   {
      tempPtr = value->licenseList;
      for (i = 0; i < (int) value->numItems; i++)
      {
         PrintARUserLicenseStruct(tempPtr);
         tempPtr++;
      }
   }
}


/*****************************************************************************/
/*                                                                           */
/*                          PrintARUserInfoStruct                            */
/*                                                                           */
/*****************************************************************************/

void PrintARUserInfoStruct(value)
ARUserInfoStruct  *value;    /* IN; value to print */

{
   DriverPrint("   User Info Struct:\n");
   PrintChar("      User name:", value->userName);
   PrintARUserLicenseList(&value->licenseInfo);
   PrintARTimestamp("      Connect time:", value->connectTime);
   PrintARTimestamp("      Last access:", value->lastAccess);
   PrintChar("      Email address:", value->emailAddr);
   PrintInt("      Default notify mechanism:", value->defaultNotifyMech);
}


/*****************************************************************************/
/*                                                                           */
/*                           PrintARUserInfoList                             */
/*                                                                           */
/*****************************************************************************/

void PrintARUserInfoList(value)
ARUserInfoList *value;       /* IN; value to print */

{
   int               i;       /* working index */
   ARUserInfoStruct *tempPtr; /* working pointer */

   DriverPrint("User Info List : %u items\n", value->numItems);
   if (value->numItems != 0)
   {
      tempPtr = value->userList;
      for (i = 0; i < (int) value->numItems; i++)
      {
         PrintARUserInfoStruct(tempPtr);
         tempPtr++;
      }
   }
}


/*****************************************************************************/
/*                                                                           */
/*                        PrintARIntegerLimitsStruct                         */
/*                                                                           */
/*****************************************************************************/

void PrintARIntegerLimitsStruct(value)
ARIntegerLimitsStruct *value; /* IN; value to print */

{
   DriverPrint("INTEGER\n");
   PrintLong("   Low range :", (int)value->rangeLow);
   PrintLong("   High range:", (int)value->rangeHigh);
}


/*****************************************************************************/
/*                                                                           */
/*                          PrintARRealLimitsStruct                          */
/*                                                                           */
/*****************************************************************************/

void PrintARRealLimitsStruct(value)
ARRealLimitsStruct *value;    /* IN; value to print */

{
   DriverPrint("REAL\n");
   PrintReal("   Low range :", value->rangeLow);
   PrintReal("   High range:", value->rangeHigh);
   PrintInt("   Precision:", value->precision);
}


/*****************************************************************************/
/*                                                                           */
/*                          PrintARCharLimitsStruct                          */
/*                                                                           */
/*****************************************************************************/

void PrintARCharLimitsStruct(value)
ARCharLimitsStruct *value;    /* IN; value to print */

{
   DriverPrint("CHARACTER\n");
   PrintInt("   Maximum length:", value->maxLength);
   PrintInt("   Menu style:", value->menuStyle);
   PrintInt("   QBE match operation:", value->qbeMatchOperation);
   PrintChar("   Character menu:", value->charMenu);
   PrintChar("   Pattern:", value->pattern);
   PrintInt("   FullText Options:", value->fullTextOptions);
}

/*****************************************************************************/
/*                                                                           */
/*                          PrintARDiaryLimitsStruct                         */
/*                                                                           */
/*****************************************************************************/

void PrintARDiaryLimitsStruct(value)
ARDiaryLimitsStruct *value;    /* IN; value to print */

{
   DriverPrint("DIARY\n");
   PrintInt("   FullText Options:", value->fullTextOptions);
}

/*****************************************************************************/
/*                                                                           */
/*                          PrintARAttachLimitsStruct                        */
/*                                                                           */
/*****************************************************************************/

void PrintARAttachLimitsStruct(value)
ARAttachLimitsStruct *value;    /* IN; value to print */

{
   DriverPrint("ATTACHMENT\n");
   PrintInt("   Maximum size:", value->maxSize);
   PrintInt("   Attachment Type:", value->attachType);
}

/*****************************************************************************/
/*                                                                           */
/*                          PrintARTableLimitsStruct                         */
/*                                                                           */
/*****************************************************************************/

void PrintARTableLimitsStruct(header, value)
char                *header;   /* IN; header of the value */
ARTableLimitsStruct *value;    /* IN; value to print */
{
   DriverPrint("%s\n", header);
   PrintChar("   Schema:", value->schema);
   PrintChar("   Server:", value->server);
   PrintARQualifierStruct("   ", &value->qualifier);
   PrintInt("   numColumns:", value->numColumns);
   PrintInt("   maxRetrieve:", value->maxRetrieve);
}

/*****************************************************************************/
/*                                                                           */
/*                          PrintARColumnLimitsStruct                        */
/*                                                                           */
/*****************************************************************************/

void PrintARColumnLimitsStruct(header, value)
char                *header;   /* IN; header of the value */
ARColumnLimitsStruct *value;    /* IN; value to print */

{
   DriverPrint("%s\n", header);
   PrintInt("   parent:", value->parent);
   PrintInt("   dataField:", value->dataField);
   PrintInt("   colLength:", value->colLength);
}

/*****************************************************************************/
/*                                                                           */
/*                         PrintARDecimalLimitsStruct                        */
/*                                                                           */
/*****************************************************************************/

void PrintARDecimalLimitsStruct(value)
ARDecimalLimitsStruct *value;    /* IN; value to print */

{
   DriverPrint("DECIMAL\n");
   PrintChar("   Low range :", value->rangeLow);
   PrintChar("   High range:", value->rangeHigh);
   PrintInt("   Precision:", value->precision);
}

/*****************************************************************************/
/*                                                                           */
/*                         PrintARFieldLimitsStruct                          */
/*                                                                           */
/*****************************************************************************/

void PrintARFieldLimitStruct(value)
ARFieldLimitStruct *value;    /* IN; value to print */

{
   DriverPrint("Field Limit Struct:  ");
   switch (value->dataType)
   {
      case AR_FIELD_LIMIT_NONE  :
         DriverPrint("   NO LIMITS\n");
         break;
      case AR_DATA_TYPE_INTEGER :
         PrintARIntegerLimitsStruct(&value->u.intLimits);
         break;
      case AR_DATA_TYPE_REAL    :
         PrintARRealLimitsStruct(&value->u.realLimits);
         break;
      case AR_DATA_TYPE_CHAR    :
         PrintARCharLimitsStruct(&value->u.charLimits);
         break;
      case AR_DATA_TYPE_DIARY   :
         PrintARDiaryLimitsStruct(&value->u.diaryLimits);
         break;
      case AR_DATA_TYPE_ENUM    :
         DriverPrint("ENUMERATED\n");
         PrintARNameList("   Enum values:", "      ", &value->u.enumLimits);
         break;
      case AR_DATA_TYPE_TIME    :
         DriverPrint("TIMESTAMP\n");
         break;
      case AR_DATA_TYPE_BITMASK :
         DriverPrint("BITMASK\n");
         PrintARNameList("   Bitmask values:", "      ", &value->u.maskLimits);
         break;
      case AR_DATA_TYPE_ATTACH :
         PrintARAttachLimitsStruct(&value->u.attachLimits);
         break;
      case AR_DATA_TYPE_DECIMAL:
         PrintARDecimalLimitsStruct(&value->u.decimalLimits);
         break;
      case AR_DATA_TYPE_TABLE   :
         PrintARTableLimitsStruct("   Table values:", &value->u.tableLimits);
         break;
      case AR_DATA_TYPE_COLUMN  :
         PrintARColumnLimitsStruct("   Column values:", &value->u.columnLimits);
         break;
      default                   :
         DriverPrint("   <undefined datatype = %u>\n", value->dataType);
         break;
   }
}


/*****************************************************************************/
/*                                                                           */
/*                         PrintARCharMenuItemStruct                         */
/*                                                                           */
/*****************************************************************************/

void PrintARCharMenuItemStruct(indent, value)
char                 *indent;     /* IN; string of blanks for indenting */
ARCharMenuItemStruct *value;      /* IN; value to print */

{
   char  buff[55];              /* temporary buffer */

   DriverPrint("%sChar Menu Item Struct:\n", indent);
   DriverPrint("%s", indent);
   PrintChar("   Label:", value->menuLabel);
   DriverPrint("%s", indent);
   PrintInt("   Type:", value->menuType);
   if (value->menuType == AR_MENU_TYPE_VALUE)
   {
      DriverPrint("%s", indent);
      PrintChar("   Value:", value->u.menuValue);
   }
   else if (value->menuType == AR_MENU_TYPE_MENU)
   {
      if (strlen(indent) < (size_t) 50)
      {
         (void) strcpy(buff, indent);
         (void) strcat(buff, "   ");
      }
      else  /* reset to 0 spaces */
         buff[0] = '\0';
      PrintARCharMenuStruct(buff, value->u.childMenu);
   }
}


/*****************************************************************************/
/*                                                                           */
/*                        PrintARCharMenuQueryStruct                         */
/*                                                                           */
/*****************************************************************************/

void PrintARCharMenuQueryStruct(indent, value)
char                  *indent;     /* IN; string of blanks for indenting */
ARCharMenuQueryStruct *value;      /* IN; value to print */

{
   DriverPrint("%sChar Menu Query Struct:\n", indent);
   DriverPrint("%s", indent);
   PrintChar("   Schema name:", value->schema);
   DriverPrint("%s", indent);
   PrintChar("   Server name:", value->server);
   PrintARQualifierStruct(indent, &value->qualifier);
   DriverPrint("%s", indent);
   PrintLong("   Label Field Id:", value->labelField);
   DriverPrint("%s", indent);
   PrintLong("   Value Field Id:", value->valueField);
   DriverPrint("%s", indent);
   PrintARBoolean("   Sort on Label:", value->sortOnLabel);
}


/*****************************************************************************/
/*                                                                           */
/*                         PrintARCharMenuSQLStruct                          */
/*                                                                           */
/*****************************************************************************/

void PrintARCharMenuSQLStruct(indent, value)
char                  *indent;     /* IN; string of blanks for indenting */
ARCharMenuSQLStruct   *value;      /* IN; value to print */

{
   DriverPrint("%sChar Menu SQL Struct:\n", indent);
   DriverPrint("%s", indent);
   PrintChar("   Server name:", value->server);
   DriverPrint("%s", indent);
   PrintChar("   SQL Command:", value->sqlCommand);
   DriverPrint("%s", indent);
   PrintLong("   Label Index:", value->labelIndex);
   DriverPrint("%s", indent);
   PrintLong("   Value Index:", value->valueIndex);
}


/*****************************************************************************/
/*                                                                           */
/*                           PrintARCharMenuList                             */
/*                                                                           */
/*****************************************************************************/

void PrintARCharMenuList(indent, value)
char           *indent;       /* IN; string of blanks for indenting */
ARCharMenuList *value;        /* IN; value to print */

{
   int                i;      /* working index */

   DriverPrint("%sCharacter Menu List : %u items\n", indent, value->numItems);
   if (value->numItems != 0)
      for (i = 0; i < (int) value->numItems; i++)
         PrintARCharMenuItemStruct(indent, &value->charMenuList[i]);
}


/*****************************************************************************/
/*                                                                           */
/*                           PrintARCharMenuStruct                           */
/*                                                                           */
/*****************************************************************************/

void PrintARCharMenuStruct(indent, value)
char             *indent;     /* IN; string of blanks for indenting */
ARCharMenuStruct *value;      /* IN; value to print */

{
   char  buff[55];            /* temporary buffer */

   if (strlen(indent) < (size_t) 50)
   {
      (void) strcpy(buff, indent);
      (void) strcat(buff, "   ");
   }
   else  /* reset to 0 spaces */
      buff[0] = '\0';

   DriverPrint("%sChar Menu Struct:\n", indent);
   DriverPrint("%sType of Menu:  ", buff);
   switch (value->menuType)
   {
      case AR_CHAR_MENU_NONE  :
         DriverPrint("NONE\n");
         break;
      case AR_CHAR_MENU_LIST  :
         DriverPrint("LIST\n");
         PrintARCharMenuList(buff, &value->u.menuList);
         break;
      case AR_CHAR_MENU_QUERY :
         DriverPrint("QUERY\n");
         PrintARCharMenuQueryStruct(buff, &value->u.menuQuery);
         break;
      case AR_CHAR_MENU_FILE  :
         DriverPrint("FILE\n");
         DriverPrint("%s   File Location:  ", buff);
         switch (value->u.menuFile.fileLocation)
         {
            case AR_MENU_FILE_SERVER :
               DriverPrint("SERVER\n");
               break;
            case AR_MENU_FILE_CLIENT :
               DriverPrint("CLIENT\n");
               break;
            default                   :
               DriverPrint("   <undefined file location = %u>\n",
                           value->u.menuFile.fileLocation);
               break;
         }
         DriverPrint("%s   Filename:  %s\n", buff, value->u.menuFile.filename);
         break;
      case AR_CHAR_MENU_SQL   :
         DriverPrint("SQL\n");
         PrintARCharMenuSQLStruct(buff, &value->u.menuSQL);
         break;
      default                   :
         DriverPrint("   <undefined menuType = %u>\n", value->menuType);
         break;
   }
}


/*****************************************************************************/
/*                                                                           */
/*                           PrintARStructItemStruct                         */
/*                                                                           */
/*****************************************************************************/

void PrintARStructItemStruct(value)
ARStructItemStruct *value;    /* IN; value to print */

{
   DriverPrint("   Structure Item Struct:  ");
   switch (value->type)
   {
      case AR_STRUCT_ITEM_SCHEMA      :
         DriverPrint("SCHEMA\n");
         break;
      case AR_STRUCT_ITEM_SCHEMA_DEFN :
         DriverPrint("SCHEMA DEFN\n");
         break;
      case AR_STRUCT_ITEM_SCHEMA_VIEW :
         DriverPrint("SCHEMA VIEW\n");
         break;
      case AR_STRUCT_ITEM_SCHEMA_MAIL :
         DriverPrint("SCHEMA MAIL\n");
         break;
      case AR_STRUCT_ITEM_FILTER      :
         DriverPrint("FILTER\n");
         break;
      case AR_STRUCT_ITEM_ACTIVE_LINK :
         DriverPrint("ACTIVE LINK\n");
         break;
      case AR_STRUCT_ITEM_ADMIN_EXT   :
         DriverPrint("ADMINISTRATOR EXTENSION\n");
         break;
      case AR_STRUCT_ITEM_CHAR_MENU   :
         DriverPrint("CHARACTER MENU\n");
         break;
      case AR_STRUCT_ITEM_ESCALATION   :
         DriverPrint("ESCALATION\n");
         break;
      case AR_STRUCT_ITEM_DIST_MAP     :
         DriverPrint("DISTRIBUTED MAPPING\n");
         break;
      default                         :
         DriverPrint("   <undefined type = %u>\n", value->type);
         break;
   }
   PrintChar("      Name:", value->name);
}


/*****************************************************************************/
/*                                                                           */
/*                            PrintARStructItemList                          */
/*                                                                           */
/*****************************************************************************/

void PrintARStructItemList(value)
ARStructItemList *value;      /* IN; value to print */

{
   int                 i;      /* working index */
   ARStructItemStruct *tempPtr;/* working pointer */

   DriverPrint("Structure Item List : %u items\n", value->numItems);
   if (value->numItems != 0)
   {
      tempPtr = value->structItemList;
      for (i = 0; i < (int) value->numItems; i++)
      {
         PrintARStructItemStruct(tempPtr);
         tempPtr++;
      }
   }
}


/*****************************************************************************/
/*                                                                           */
/*                        PrintARServerInfoRequestList                       */
/*                                                                           */
/*****************************************************************************/

void PrintARServerInfoRequestList(header2, value)
char                    *header2;/* IN; header string for 2nd+ line */
ARServerInfoRequestList *value;  /* IN; value to print */

{
   char          buff[82];    /* buffer to build output in */
   char          buff2[16];   /* buffer to hold a converted number */
   int           i;           /* working index */
   unsigned int *tempPtr;     /* working pointer */

   (void) sprintf(buff, "Server Info Request List:  (%u)", value->numItems);
   if (value->numItems != 0)
   {
      tempPtr = value->requestList;
      for (i = 0; i < (int) value->numItems; i++)
      {
         if (i != 0)
            (void) strcat(buff, ", ");
         (void) sprintf(buff2, "%u", *tempPtr);
         if ((strlen(buff) + strlen(buff2)) >= (size_t) 79)
         {
            buff[strlen(buff) - 1] = '\n'; /* over last blank */
            DriverPrint("%s", buff);
            (void) strcpy(buff, header2);
         }
         (void) strcat(buff, buff2);
         tempPtr++;
      }
   }
   DriverPrint("%s\n", buff);
}


/*****************************************************************************/
/*                                                                           */
/*                           PrintARServerInfoStruct                         */
/*                                                                           */
/*****************************************************************************/

void PrintARServerInfoStruct(value)
ARServerInfoStruct *value;    /* IN; value to print */

{
   DriverPrint("   Server Info Struct:  ");
   switch (value->operation)
   {
      case AR_SERVER_INFO_DB_TYPE        :
         DriverPrint("DB TYPE\n");
         break;
      case AR_SERVER_INFO_SERVER_LICENSE :
         DriverPrint("SERVER LICENSE\n");
         break;
      case AR_SERVER_INFO_FIXED_LICENSE  :
         DriverPrint("FIXED USER LICENSE\n");
         break;
      case AR_SERVER_INFO_VERSION        :
         DriverPrint("VERSION\n");
         break;
      case AR_SERVER_INFO_ALLOW_GUESTS   :
         DriverPrint("ALLOW GUESTS\n");
         break;
      case AR_SERVER_INFO_USE_ETC_PASSWD :
         DriverPrint("USE /etc/passwd\n");
         break;
      case AR_SERVER_INFO_XREF_PASSWORDS :
         DriverPrint("CROSSREF PASSWORDS\n");
         break;
      case AR_SERVER_INFO_DEBUG_MODE     :
         DriverPrint("DEBUG MODE\n");
         break;
      case AR_SERVER_INFO_DB_NAME        :
         DriverPrint("DB NAME\n");
         break;
      case AR_SERVER_INFO_DB_PASSWORD    :
         DriverPrint("DB PASSWORD\n");
         break;
      case AR_SERVER_INFO_HARDWARE       :
         DriverPrint("HARDWARE\n");
         break;
      case AR_SERVER_INFO_OS             :
         DriverPrint("OS\n");
         break;
      case AR_SERVER_INFO_SERVER_DIR     :
         DriverPrint("SERVER DIRECTORY\n");
         break;
      case AR_SERVER_INFO_DBHOME_DIR     :
         DriverPrint("DBHOME DIRECTORY\n");
         break;
      case AR_SERVER_INFO_SET_PROC_TIME  :
         DriverPrint("SET FILTER PROCESS TIMEOUT\n");
         break;
      case AR_SERVER_INFO_EMAIL_FROM     :
         DriverPrint("EMAIL FROM ADDRESS FOR NOTIFICATIONS\n");
         break;
      case AR_SERVER_INFO_SQL_LOG_FILE   :
         DriverPrint("SQL LOG FILENAME\n");
         break;
      case AR_SERVER_INFO_FLOAT_LICENSE  :
         DriverPrint("FLOATING USER LICENSE\n");
         break;
      case AR_SERVER_INFO_FLOAT_TIMEOUT  :
         DriverPrint("FLOATING LICENSE TIMEOUT\n");
         break;
      case AR_SERVER_INFO_UNQUAL_QUERIES :
         DriverPrint("ALLOW UNQUALIFIED QUERIES\n");
         break;
      case AR_SERVER_INFO_FILTER_LOG_FILE:
         DriverPrint("FILTER LOG FILENAME\n");
         break;
      case AR_SERVER_INFO_USER_LOG_FILE  :
         DriverPrint("USER LOG FILENAME\n");
         break;
      case AR_SERVER_INFO_REM_SERV_ID    :
         DriverPrint("REMEDY SERVER ID\n");
         break;
      case AR_SERVER_INFO_MULTI_SERVER   :
         DriverPrint("MULTI SERVER FLAG\n");
         break;
      case AR_SERVER_INFO_EMBEDDED_SQL   :
         DriverPrint("EMBEDDED SQL FLAG\n");
         break;
      case AR_SERVER_INFO_MAX_SCHEMAS    :
         DriverPrint("MAX SCHEMAS LICENSED\n");
         break;
      case AR_SERVER_INFO_DB_VERSION     :
         DriverPrint("DB VERSION\n");
         break;
      case AR_SERVER_INFO_MAX_ENTRIES    :
         DriverPrint("MAX ENTRIES PER QUERY\n");
         break;
      case AR_SERVER_INFO_MAX_F_DAEMONS  :
         DriverPrint("MAX FAST PROCESSES\n");
         break;
      case AR_SERVER_INFO_MAX_L_DAEMONS  :
         DriverPrint("MAX LIST PROCESSES\n");
         break;
      case AR_SERVER_INFO_ESCALATION_LOG_FILE:
         DriverPrint("ESCALATION LOG FILENAME\n");
         break;
      case AR_SERVER_INFO_ESCL_DAEMON    :
         DriverPrint("ESCALATION DAEMON\n");
         break;
      case AR_SERVER_INFO_SUBMITTER_MODE :
         DriverPrint("SUBMITTER MODE\n");
         break;
      case AR_SERVER_INFO_API_LOG_FILE   :
         DriverPrint("API LOG FILE\n");
         break;
      case AR_SERVER_INFO_FTEXT_FIXED    :
         DriverPrint("FULL TEXT FIXED LICENSES\n");
         break;
      case AR_SERVER_INFO_FTEXT_FLOAT    :
         DriverPrint("FULL TEXT FLOATING LICENSES\n");
         break;
      case AR_SERVER_INFO_FTEXT_TIMEOUT  :
         DriverPrint("FULL TEXT FLOATING LICENSE TIMEOUT\n");
         break;
      case AR_SERVER_INFO_RESERV1_A     :
         DriverPrint("RESERVED FIXED LICENSES\n");
         break;
      case AR_SERVER_INFO_RESERV1_B     :
         DriverPrint("RESERVED FLOATING LICENSES\n");
         break;
      case AR_SERVER_INFO_RESERV1_C     :
         DriverPrint("RESERVED BUILDER LICENSES\n");
         break;
      case AR_SERVER_INFO_SERVER_IDENT  :
         DriverPrint("UNIQUE SERVER IDENTIFIER\n");
         break;
      case AR_SERVER_INFO_DS_SVR_LICENSE :
         DriverPrint("DISTRIBUTED SERVER LICENSE\n");
         break;
      case AR_SERVER_INFO_DS_MAPPING :
         DriverPrint("DISTRIBUTED SERVER MAPPING SCHEMA\n");
         break;
      case AR_SERVER_INFO_DS_PENDING :
         DriverPrint("DISTRIBUTED SERVER PENDING SCHEMA\n");
         break;
      case AR_SERVER_INFO_DS_RPC_SOCKET :
         DriverPrint("DISTRIBUTED SERVER RPC SOCKET\n");
         break;
      case AR_SERVER_INFO_DS_LOG_FILE   :
         DriverPrint("DISTRIBUTED SERVER LOG FILE\n");
         break;
      case AR_SERVER_INFO_SUPPRESS_WARN :
         DriverPrint("SUPPRESSED WARNINGS\n");
         break;
      case AR_SERVER_INFO_HOSTNAME      :
         DriverPrint("SERVER HOSTNAME\n");
         break;
      case AR_SERVER_INFO_FULL_HOSTNAME :
         DriverPrint("SERVER FULL (DNS) HOSTNAME\n");
         break;
      case AR_SERVER_INFO_SAVE_LOGIN    :
         DriverPrint("SAVE LOGIN\n");
         break;
      case AR_SERVER_INFO_U_CACHE_CHANGE:
         DriverPrint("USER CACHE CHANGE TIME\n");
         break;
      case AR_SERVER_INFO_G_CACHE_CHANGE:
         DriverPrint("GROUP CACHE CHANGE TIME\n");
         break;
      case AR_SERVER_INFO_STRUCT_CHANGE :
         DriverPrint("STRUCTURE CHANGE TIME\n");
         break;
      case AR_SERVER_INFO_CASE_SENSITIVE:
         DriverPrint("CASE SENSITIVITY\n");
         break;
      case AR_SERVER_INFO_SERVER_LANG   :
         DriverPrint("SERVER LANGUAGE\n");
         break;
      case AR_SERVER_INFO_ADMIN_ONLY    :
         DriverPrint("ADMIN ONLY MODE\n");
         break;
      case AR_SERVER_INFO_CACHE_LOG_FILE :
         DriverPrint("SHARED CACHE LOG FILE\n");
         break;
      case AR_SERVER_INFO_FLASH_DAEMON   :
         DriverPrint("FLASHBOARDS DAEMON\n");
         break;
      case AR_SERVER_INFO_THREAD_LOG_FILE:
         DriverPrint("THREAD LOG FILE\n");
         break;
      case AR_SERVER_INFO_ADMIN_TCP_PORT :
         DriverPrint("ADMIN SERVER TCP PORT\n");
         break;
      case AR_SERVER_INFO_ESCL_TCP_PORT  :
         DriverPrint("ESCALATION SERVER TCP PORT\n");
         break;
      case AR_SERVER_INFO_FAST_TCP_PORT  :
         DriverPrint("FAST SERVERS TCP PORT (initial port)\n");
         break;
      case AR_SERVER_INFO_LIST_TCP_PORT  :
         DriverPrint("LIST SERVERS TCP PORT (initial port)\n");
         break;
      case AR_SERVER_INFO_FLASH_TCP_PORT :
         DriverPrint("FLASHBOARDS SERVER TCP PORT\n");
         break;
      case AR_SERVER_INFO_TCD_TCP_PORT   :
         DriverPrint("TCD SERVER TCP PORT\n");
         break;
      case AR_SERVER_INFO_DSO_DEST_PORT  :
         DriverPrint("DSO SERVER DESTINATION TCP PORT\n");
         break;
      case AR_SERVER_INFO_INFORMIX_DBN   :
         DriverPrint("INFORMIX DB NAME\n");
         break;
      case AR_SERVER_INFO_INFORMIX_TBC   :
         DriverPrint("INFORMIX ONCONFIG\n");
         break;
      case AR_SERVER_INFO_INGRES_VNODE   :
         DriverPrint("INGRES VNODE\n");
         break;
      case AR_SERVER_INFO_ORACLE_SID     :
         DriverPrint("ORACLE SID\n");
         break;
      case AR_SERVER_INFO_ORACLE_TWO_T   :
         DriverPrint("ORACLE TWO TASK\n");
         break;
      case AR_SERVER_INFO_SYBASE_CHARSET :
         DriverPrint("SYBASE CHARSET\n");
         break;
      case AR_SERVER_INFO_SYBASE_SERV    :
         DriverPrint("SYBASE DB SERVER NAME\n");
         break;
      case AR_SERVER_INFO_SHARED_MEM     :
         DriverPrint("SHARED MEMORY\n");
         break;
      case AR_SERVER_INFO_SHARED_CACHE   :
         DriverPrint("SHARED CACHE\n");
         break;
      case AR_SERVER_INFO_CACHE_SEG_SIZE :
         DriverPrint("SHARED CACHE SEGMENT SIZE\n");
         break;
      case AR_SERVER_INFO_DB_USER        :
         DriverPrint("DB USER\n");
         break;
      case AR_SERVER_INFO_NFY_TCP_PORT :
         DriverPrint("NOTIFIER SERVER TCP PORT\n");
         break;
      case AR_SERVER_INFO_FILT_MAX_TOTAL :
         DriverPrint("FILTER MAX TOTAL NUMBER\n");
         break;
      case AR_SERVER_INFO_FILT_MAX_STACK :
         DriverPrint("FILTER MAX STACK SIZE\n");
         break;
      case AR_SERVER_INFO_DEFAULT_ORDER_BY :
         DriverPrint("DEFAULT ORDER BY ENTRY ID\n");
         break;
      case AR_SERVER_INFO_DELAYED_CACHE :
         DriverPrint("DELAYED CACHE\n");
         break;
      case AR_SERVER_INFO_DSO_MERGE_STYLE :
         DriverPrint("DSO MERGE STYLE\n");
         break;
      case AR_SERVER_INFO_EMAIL_LINE_LEN :
         DriverPrint("EMAIL LINE LENGTH\n");
         break;
      case AR_SERVER_INFO_EMAIL_SYSTEM :
         DriverPrint("EMAIL SYSTEM\n");
         break;
      case AR_SERVER_INFO_INFORMIX_RELAY_MOD :
         DriverPrint("INFORMIX RELAY MODULE\n");
         break;
      case AR_SERVER_INFO_PS_RPC_SOCKET :
         DriverPrint("PRIVATE SERVER RPC SOCKETS / PORTS\n");
         break;
      case AR_SERVER_INFO_REGISTER_PORTMAPPER :
         DriverPrint("REGISTER PORTMAPPER\n");
         break;
      case AR_SERVER_INFO_SERVER_NAME :
         DriverPrint("SERVER NAME\n");
         break;
      case AR_SERVER_INFO_DBCONF :
         DriverPrint("DBCONF\n");
         break;
      case AR_SERVER_INFO_APPL_PENDING :
         DriverPrint("APPLICATION PENDING SCHEMA\n");
         break;
      case AR_SERVER_INFO_AP_RPC_SOCKET :
         DriverPrint("APPROVAL SERVER RPC SOCKET\n");
         break;
      case AR_SERVER_INFO_AP_LOG_FILE   :
         DriverPrint("APPROVAL SERVER LOG FILE\n");
         break;
      case AR_SERVER_INFO_AP_DEFN_CHECK :
         DriverPrint("APPLICATION DEFINITION CHECK INTERVAL\n");
         break;
      case AR_SERVER_INFO_MAX_LOG_FILE_SIZE:
         DriverPrint("MAXIMUM SIZE OF LOG FILES\n");
         break;
      case AR_SERVER_INFO_EMAIL_TIMEOUT :
         DriverPrint("EMAIL TIMEOUT\n");
         break;
      case AR_SERVER_INFO_EXPORT_VERSION :
         DriverPrint("SERVER EXPORT VERSION\n");
         break;
      case AR_SERVER_INFO_ENCRYPT_AL_SQL :
         DriverPrint("ENCRYPT AL DIRECT SQL\n");
         break;
      default                            :
         DriverPrint("   <undefined operation = %u>\n", value->operation);
         break;
   }
   PrintARValueStruct("      ", &value->value);
}


/*****************************************************************************/
/*                                                                           */
/*                            PrintARServerInfoList                          */
/*                                                                           */
/*****************************************************************************/

void PrintARServerInfoList(value)
ARServerInfoList *value;      /* IN; value to print */

{
   int                 i;      /* working index */
   ARServerInfoStruct *tempPtr;/* working pointer */

   DriverPrint("Server Info List : %u items\n", value->numItems);
   if (value->numItems != 0)
   {
      tempPtr = value->serverInfoList;
      for (i = 0; i < (int) value->numItems; i++)
      {
         PrintARServerInfoStruct(tempPtr);
         tempPtr++;
      }
   }
}


/*****************************************************************************/
/*                                                                           */
/*                           PrintARServerInfoStruct2                        */
/*                                                                           */
/*****************************************************************************/

void PrintARServerInfoStruct2(value)
ARServerInfoStruct *value;    /* IN; value to print */

{
   DriverPrint("   Server Info Struct:  ");
   switch (value->operation)
   {
      case AR_SERVER_STAT_START_TIME     :
         DriverPrint("START TIME\n");
         break;
      case AR_SERVER_STAT_BAD_PASSWORD   :
         DriverPrint("NUMBER BAD PASSWORD LOGIN ATTEMPTS\n");
         break;
      case AR_SERVER_STAT_NO_WRITE_TOKEN :
         DriverPrint("FLOATING WRITE TOKEN UNAVAILABLE\n");
         break;
      case AR_SERVER_STAT_NO_FULL_TOKEN  :
         DriverPrint("FLOATING FULLTEXT TOKEN UNAVAILABLE\n");
         break;
      case AR_SERVER_STAT_CURRENT_USERS  :
         DriverPrint("NUMBER CURRENT USERS\n");
         break;
      case AR_SERVER_STAT_WRITE_FIXED    :
         DriverPrint("NUMBER CURRENT WRITE FIXED USERS\n");
         break;
      case AR_SERVER_STAT_WRITE_FLOATING :
         DriverPrint("NUMBER CURRENT WRITE FLOATING USERS\n");
         break;
      case AR_SERVER_STAT_WRITE_READ     :
         DriverPrint("NUMBER CURRENT WRITE READ-ONLY USERS\n");
         break;
      case AR_SERVER_STAT_FULL_FIXED     :
         DriverPrint("NUMBER CURRENT FULLTEXT FIXED USERS\n");
         break;
      case AR_SERVER_STAT_FULL_FLOATING  :
         DriverPrint("NUMBER CURRENT FULLTEXT FLOATING USERS\n");
         break;
      case AR_SERVER_STAT_FULL_NONE      :
         DriverPrint("NUMBER CURRENT FULLTEXT NONE USERS\n");
         break;
      case AR_SERVER_STAT_API_REQUESTS   :
         DriverPrint("NUMBER API REQUESTS\n");
         break;
      case AR_SERVER_STAT_API_TIME       :
         DriverPrint("API SECONDS\n");
         break;
      case AR_SERVER_STAT_ENTRY_TIME     :
         DriverPrint("ARxxxEntry SECONDS\n");
         break;
      case AR_SERVER_STAT_RESTRUCT_TIME  :
         DriverPrint("RESTRUCTURE SECONDS\n");
         break;
      case AR_SERVER_STAT_OTHER_TIME     :
         DriverPrint("OTHER SECONDS\n");
         break;
      case AR_SERVER_STAT_CACHE_TIME     :
         DriverPrint("CACHE SECONDS\n");
         break;
      case AR_SERVER_STAT_GET_E_COUNT    :
         DriverPrint("NUMBER GET ENTRY\n");
         break;
      case AR_SERVER_STAT_GET_E_TIME     :
         DriverPrint("GET ENTRY SECONDS\n");
         break;
      case AR_SERVER_STAT_SET_E_COUNT    :
         DriverPrint("NUMBER SET ENTRY\n");
         break;
      case AR_SERVER_STAT_SET_E_TIME     :
         DriverPrint("SET ENTRY SECONDS\n");
         break;
      case AR_SERVER_STAT_CREATE_E_COUNT :
         DriverPrint("NUMBER CREATE ENTRY\n");
         break;
      case AR_SERVER_STAT_CREATE_E_TIME  :
         DriverPrint("CREATE ENTRY SECONDS\n");
         break;
      case AR_SERVER_STAT_DELETE_E_COUNT :
         DriverPrint("NUMBER DELETE ENTRY\n");
         break;
      case AR_SERVER_STAT_DELETE_E_TIME  :
         DriverPrint("DELETE ENTRY SECONDS\n");
         break;
      case AR_SERVER_STAT_MERGE_E_COUNT  :
         DriverPrint("NUMBER MERGE ENTRY\n");
         break;
      case AR_SERVER_STAT_MERGE_E_TIME   :
         DriverPrint("MERGE ENTRY SECONDS\n");
         break;
      case AR_SERVER_STAT_GETLIST_E_COUNT:
         DriverPrint("NUMBER GETLIST ENTRY\n");
         break;
      case AR_SERVER_STAT_GETLIST_E_TIME :
         DriverPrint("GETLIST ENTRY SECONDS\n");
         break;
      case AR_SERVER_STAT_E_STATS_COUNT  :
         DriverPrint("NUMBER ENTRY STATISTICS\n");
         break;
      case AR_SERVER_STAT_E_STATS_TIME   :
         DriverPrint("ENTRY STATISTICS SECONDS\n");
         break;
      case AR_SERVER_STAT_FILTER_PASSED  :
         DriverPrint("NUMBER FILTERS PASSED QUAL\n");
         break;
      case AR_SERVER_STAT_FILTER_FAILED  :
         DriverPrint("NUMBER FILTERS FAILED QUAL\n");
         break;
      case AR_SERVER_STAT_FILTER_DISABLE :
         DriverPrint("NUMBER FILTERS DISABLED\n");
         break;
      case AR_SERVER_STAT_FILTER_NOTIFY  :
         DriverPrint("NUMBER FILTER NOTIFY ACTIONS\n");
         break;
      case AR_SERVER_STAT_FILTER_MESSAGE :
         DriverPrint("NUMBER FILTER MESSAGE ACTIONS\n");
         break;
      case AR_SERVER_STAT_FILTER_LOG     :
         DriverPrint("NUMBER FILTER LOG ACTIONS\n");
         break;
      case AR_SERVER_STAT_FILTER_FIELDS  :
         DriverPrint("NUMBER FILTER FIELDS ACTIONS\n");
         break;
      case AR_SERVER_STAT_FILTER_PROCESS :
         DriverPrint("NUMBER FILTER PROCESS ACTIONS\n");
         break;
      case AR_SERVER_STAT_FILTER_TIME    :
         DriverPrint("FILTER SECONDS\n");
         break;
      case AR_SERVER_STAT_ESCL_PASSED    :
         DriverPrint("NUMBER ESCALATIONS PASSED QUAL\n");
         break;
      case AR_SERVER_STAT_ESCL_FAILED    :
         DriverPrint("NUMBER ESCALATIONS FAILED QUAL\n");
         break;
      case AR_SERVER_STAT_ESCL_DISABLE   :
         DriverPrint("NUMBER ESCALATIONS DISABLED\n");
         break;
      case AR_SERVER_STAT_ESCL_NOTIFY    :
         DriverPrint("NUMBER ESCALATION NOTIFY ACTIONS\n");
         break;
      case AR_SERVER_STAT_ESCL_LOG       :
         DriverPrint("NUMBER ESCALATION LOG ACTIONS\n");
         break;
      case AR_SERVER_STAT_ESCL_FIELDS    :
         DriverPrint("NUMBER ESCALATION FIELDS ACTIONS\n");
         break;
      case AR_SERVER_STAT_ESCL_PROCESS   :
         DriverPrint("NUMBER ESCALATION PROCESS ACTIONS\n");
         break;
      case AR_SERVER_STAT_ESCL_TIME      :
         DriverPrint("ESCALATION SECONDS\n");
         break;
      case AR_SERVER_STAT_TIMES_BLOCKED  :
         DriverPrint("TIMES AT LEAST ONE REQUEST IN QUEUE\n");
         break;
      case AR_SERVER_STAT_NUMBER_BLOCKED :
         DriverPrint("TOTAL NUMBER IN QUEUE DURING RUN\n");
         break;
      case AR_SERVER_STAT_CPU            :
         DriverPrint("CPU TIME FOR SERVER PROCESS\n");
         break;
      case AR_SERVER_STAT_SQL_DB_COUNT   :
         DriverPrint("NUMBER SQL DB OPERTIONS\n");
         break;
      case AR_SERVER_STAT_SQL_DB_TIME    :
         DriverPrint("SQL DB TIME\n");
         break;
      case AR_SERVER_STAT_FTS_SRCH_COUNT :
         DriverPrint("NUMBER FULLTEXT SEARCH OPERTIONS\n");
         break;
      case AR_SERVER_STAT_FTS_SRCH_TIME  :
         DriverPrint("SQL FULLTEXT SEARCH TIME\n");
         break;
      case AR_SERVER_STAT_SINCE_START    :
         DriverPrint("TIME SINCE START TIME\n");
         break;
      case AR_SERVER_STAT_IDLE_TIME      :
         DriverPrint("IDLE TIME BETWEEN CALLS\n");
         break;
      case AR_SERVER_STAT_NET_RESP_TIME  :
         DriverPrint("NETWORK TIME SHIPPING RESPONSES\n");
         break;
      case AR_SERVER_STAT_FILTER_FIELDP  :
         DriverPrint("NUMBER FILTER PUSH FIELDS ACTIONS\n");
         break;
      case AR_SERVER_STAT_ESCL_FIELDP    :
         DriverPrint("NUMBER ESCALATION PUSH FIELDS ACTIONS\n");
         break;
      case AR_SERVER_STAT_FILTER_SQL     :
         DriverPrint("NUMBER FILTER SQL ACTIONS\n");
         break;
      case AR_SERVER_STAT_ESCL_SQL       :
         DriverPrint("NUMBER ESCALATION SQL ACTIONS\n");
         break;
      default                       :
         DriverPrint("   <undefined operation = %u>\n", value->operation);
         break;
   }
   PrintARValueStruct("      ", &value->value);
}


/*****************************************************************************/
/*                                                                           */
/*                            PrintARServerInfoList2                         */
/*                                                                           */
/*****************************************************************************/

void PrintARServerInfoList2(value)
ARServerInfoList *value;      /* IN; value to print */

{
   int                 i;      /* working index */
   ARServerInfoStruct *tempPtr;/* working pointer */

   DriverPrint("Server Info List : %u items\n", value->numItems);
   if (value->numItems != 0)
   {
      tempPtr = value->serverInfoList;
      for (i = 0; i < (int) value->numItems; i++)
      {
         PrintARServerInfoStruct2(tempPtr);
         tempPtr++;
      }
   }
}

/*****************************************************************************/
/*                                                                           */
/*                           PrintARFullTextInfoStruct                       */
/*                                                                           */
/*****************************************************************************/

void PrintARFullTextInfoStruct(value)
ARFullTextInfoStruct *value;    /* IN; value to print */

{
   DriverPrint("   FullText Info Struct:  ");
   switch (value->infoType)
   {
      case AR_FULLTEXTINFO_COLLECTION_DIR :
         DriverPrint("COLLECTION DIRECTORY\n");
         PrintARValueStruct("      ", &value->u.value);
         break;
      case AR_FULLTEXTINFO_STOPWORD:
         DriverPrint("STOPWORD LIST\n");
         PrintARValueList("      ", &value->u.valueList);
         break;
      case AR_FULLTEXTINFO_REINDEX:
         DriverPrint("COLLECTION REINDEX\n");
         PrintARValueStruct("      ", &value->u.value);
         break;
      case AR_FULLTEXTINFO_CASE_SENSITIVE_SRCH:
         DriverPrint("CASE INSENSITIVE SEARCH\n");
         PrintARValueStruct("      ", &value->u.value);
         break;
      case AR_FULLTEXTINFO_STATE:
         DriverPrint("FULLTEXT OPERATIONS STATE\n");
         PrintARValueStruct("      ", &value->u.value);
         break;
      case AR_FULLTEXTINFO_FTS_MATCH_OP:
         DriverPrint("FULLTEXT SEARCH MATCH OPERATOR\n");
         PrintARValueStruct("      ", &value->u.value);
         break;
      default                       :
         DriverPrint("   <undefined infoType = %u>\n", value->infoType);
         break;
   }
}


/*****************************************************************************/
/*                                                                           */
/*                            PrintARFullTextInfoList                        */
/*                                                                           */
/*****************************************************************************/

void PrintARFullTextInfoList(value)
ARFullTextInfoList *value;      /* IN; value to print */

{
   int                 i;      /* working index */
   ARFullTextInfoStruct *tempPtr;/* working pointer */

   DriverPrint("FullText Info List : %u items\n", value->numItems);
   if (value->numItems != 0)
   {
      tempPtr = value->fullTextInfoList;
      for (i = 0; i < (int) value->numItems; i++)
      {
         PrintARFullTextInfoStruct(tempPtr);
         tempPtr++;
      }
   }
}


/*****************************************************************************/
/*                                                                           */
/*                             PrintARStatusHistory                          */
/*                                                                           */
/*****************************************************************************/

void PrintARStatusHistoryList(header, value)
char                *header;     /* IN; header string for the value */
ARStatusHistoryList *value;      /* IN; value to print */

{
   char                   buff[55];   /* temporary buffer */
   int                    i;          /* working index */
   int                    leadSpaces; /* number of lead spaces in header */
   ARStatusHistoryStruct *tempPtr;    /* working pointer */

   leadSpaces = 0;
   while (header[leadSpaces] == ' ')
      leadSpaces++;
   leadSpaces += 3;

   if (leadSpaces < (size_t) 50)
   {
      (void) memset(buff, ' ', leadSpaces);
      buff[leadSpaces] = '\0';
   }
   else  /* reset to 0 spaces */
      buff[0] = '\0';

   DriverPrint("%s %u stat hist items\n", header, value->numItems);
   if (value->numItems != 0)
   {
      tempPtr = value->statHistList;
      for (i = 0; i < (int) value->numItems; i++)
      {
         if (tempPtr->user[0] == '\0')
            DriverPrint("%s %d - \n", buff, i);
         else
            DriverPrint("%s %d - %ld   %s\n", buff, i, tempPtr->timeVal,
                        tempPtr->user);
         tempPtr++;
      }
   }
}


/*****************************************************************************/
/*                                                                           */
/*                               PrintARDiaryList                            */
/*                                                                           */
/*****************************************************************************/

void PrintARDiaryList(header, value)
char             *header;     /* IN; header string for the value */
ARDiaryList      *value;      /* IN; value to print */

{
   char            buff[55];   /* temporary buffer */
   char            buff2[55];  /* temporary buffer */
   int             i;          /* working index */
   int             leadSpaces; /* number of lead spaces in header */
   ARDiaryStruct  *tempPtr;    /* working pointer */

   leadSpaces = 0;
   while (header[leadSpaces] == ' ')
      leadSpaces++;
   leadSpaces += 3;

   if (leadSpaces < (size_t) 50)
   {
      (void) memset(buff, ' ', leadSpaces);
      buff[leadSpaces] = '\0';
      (void) memset(buff2, ' ', (leadSpaces + 3));
      buff2[leadSpaces + 3] = '\0';
   }
   else  /* reset to 0 spaces */
   {
      buff[0] = '\0';
      (void) memset(buff2, ' ', 3);
      buff2[3] = '\0';
   }

   DriverPrint("%s %u diary entries\n", header, value->numItems);
   if (value->numItems != 0)
   {
      tempPtr = value->diaryList;
      for (i = 0; i < (int) value->numItems; i++)
      {
         DriverPrint("%s %ld   %s\n", buff, tempPtr->timeVal, tempPtr->user);
         PrintChar(buff2, tempPtr->value);

         tempPtr++;
      }
   }
}


/*****************************************************************************/
/*                                                                           */
/*                           PrintARFieldMapping                             */
/*                                                                           */
/*****************************************************************************/

void PrintARFieldMapping(header, fieldMap)
char                 *header;
ARFieldMappingStruct *fieldMap;               /* IN; value to print */

{
   DriverPrint("%s\n", header);
   DriverPrint("   Field type:  ");
   switch (fieldMap->fieldType)
   {
      case AR_FIELD_NONE :
         DriverPrint("<none>\n");
         break;
      case AR_FIELD_REGULAR :
         DriverPrint("REGULAR\n");
         break;
      case AR_FIELD_JOIN :
         DriverPrint("JOIN\n");
         PrintInt("  Schema Index: ", fieldMap->u.join.schemaIndex);
         PrintARInternalId("  Field Id :", fieldMap->u.join.realId);
         break;
      case AR_FIELD_VIEW :
         DriverPrint("VIEW\n");
         PrintChar("  External Field Name: ", fieldMap->u.view.fieldName);
         break;
      default             :
         DriverPrint("  <undefined fieldType = %u>\n", fieldMap->fieldType);
         break;
   }
}


/*****************************************************************************/
/*                                                                           */
/*                            PrintARBooleanList                             */
/*                                                                           */
/*****************************************************************************/

void PrintARBooleanList(header, value)
char          *header;        /* IN; header string for the value */
ARBooleanList *value;         /* IN; value to print */

{
   unsigned int   i;                   /* working index */

   DriverPrint("%s : %u items\n", header, value->numItems);
   for (i = 0; i < value->numItems; i++)
   {
      if (value->booleanList[i])
         DriverPrint("   Item %3d : TRUE\n", i);
      else
         DriverPrint("   Item %3d : FALSE\n", i);
   }
}


/*****************************************************************************/
/*                                                                           */
/*                           PrintARContainerInfo                            */
/*                                                                           */
/*****************************************************************************/

void PrintARContainerInfo(header, value)
char                *header;        /* IN; header string for the value */
ARContainerInfo     *value;         /* IN; value to print */
{
   DriverPrint("%sContainer Info Name: %s\n", header, value->name);
   DriverPrint("%s   Type: %u\n", header, value->type);
   DriverPrint("%s   Owner Object Type: %u\n", header, value->ownerType);
   if (value->ownerType != ARCONOWNER_NONE &&
       value->ownerType != ARCONOWNER_ALL)
      DriverPrint("%s   Owner Object Name: %s\n", header, value->ownerName);
}


/*****************************************************************************/
/*                                                                           */
/*                         PrintARContainerInfoList                          */
/*                                                                           */
/*****************************************************************************/

void PrintARContainerInfoList(header, header2, value)
char                *header;        /* IN; header string for the value */
char                *header2;       /* IN; header string for 2nd+ line */
ARContainerInfoList *value;         /* IN; value to print */
{
   unsigned int   i;          /* working index */

   DriverPrint("%s (%u)\n", header, value->numItems);
   for (i = 0; i < value->numItems; i++)
      PrintARContainerInfo(header2, &value->conInfoList[i]);
}


/*****************************************************************************/
/*                                                                           */
/*                         PrintARContainerOwnerObj                          */
/*                                                                           */
/*****************************************************************************/
void PrintARContainerOwnerObj(value)
ARContainerOwnerObj *value;         /* IN; value to print */
{
   DriverPrint("Owner Object Type: %d\n", value->type);
   if (value->type != ARCONOWNER_NONE && value->type != ARCONOWNER_ALL)
      DriverPrint("Owner Object Name: %s\n", value->ownerName);
}


/*****************************************************************************/
/*                                                                           */
/*                             PrintARReference                              */
/*                                                                           */
/*****************************************************************************/

void PrintARReferenceStruct(header, value)
char                *header;        /* IN; header string for the value */
ARReferenceStruct   *value;         /* IN; value to print */
{
   char  buf[80];

   sprintf(buf, "%s   ", header);
   DriverPrint("%sReference Info:\n", header);
   DriverPrint("%s   Label: %s\n", header, value->label);
   DriverPrint("%s   Description: %s\n", header, value->description);
   DriverPrint("%s   Type: %u\n", header, value->type);
   DriverPrint("%s   Data Type: %u\n", header, value->reference.dataType);
   if (value->reference.dataType == ARREF_DATA_ARSREF)
   {
      DriverPrint("%s   ARS Reference: %s\n", header, value->reference.u.name);
   }
   else if (value->reference.dataType == ARREF_DATA_EXTREF)
   {
      PrintARExtReferenceStruct(buf, &value->reference.u.extRef);
   }
}

/*****************************************************************************/
/*                                                                           */
/*                           PrintARExtReference                             */
/*                                                                           */
/*****************************************************************************/

void PrintARExtReferenceStruct(header, value)
char                 *header;        /* IN; header string for the value */
ARExtReferenceStruct *value;         /* IN; value to print */
{
   char  buf[80];

   DriverPrint("%sExternal Reference Info:\n", header);
   sprintf(buf, "%sGroup List:", header);
   PrintARInternalIdList(buf, header, &value->permittedGroups);
   PrintARValueStruct(header, &value->value);
}


/*****************************************************************************/
/*                                                                           */
/*                           PrintARReferenceList                            */
/*                                                                           */
/*****************************************************************************/

void PrintARReferenceList(value)
ARReferenceList *value;         /* IN; value to print */
{
   unsigned int   i;          /* working index */

   DriverPrint("Reference List (%u)\n", value->numItems);
   for (i = 0; i < value->numItems; i++)
      PrintARReferenceStruct("   ", &value->referenceList[i]);
}


/*****************************************************************************/
/*                                                                           */
/*                               PrintReturnCode                             */
/*                                                                           */
/*****************************************************************************/

void PrintReturnCode(returnCode)
int returnCode;               /* IN; value to print */

{
   DriverPrint("ReturnCode:  ");
   switch (returnCode)
   {
      case AR_RETURN_OK         :
         DriverPrint("OK\n");
         break;
      case AR_RETURN_WARNING    :
         DriverPrint("WARNING\n");
         break;
      case AR_RETURN_ERROR      :
         DriverPrint("ERROR\n");
         break;
      case AR_RETURN_FATAL      :
         DriverPrint("FATAL\n");
         break;
      case AR_RETURN_BAD_STATUS :
         DriverPrint("BAD STATUS\n");
         break;
      default                   :
         DriverPrint("<unknown return>\n");
         break;
   }
}
