/* File: main.c */

#define MAIN

#ifdef _WIN32
#include <windows.h>
#include <io.h>
#include <process.h>
#endif /* _WIN32 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "ar.h"
#include "api.h"
#include "get.h"
#include "globals.h"
#include "main.h"
#include "util.h"

void LaunchThread(ARBoolean waitFlag);

#ifdef _WIN32
/*****************************************************************************/
/*                                                                           */
/*                             SetConsoleSize                                */
/*                                                                           */
/*****************************************************************************/
void SetConsoleSize()
{
   STARTUPINFO startInfo;      /* attributes with which we were started */
   COORD coordinates;          /* screen buffer coordinates */
   SMALL_RECT rect;            /* window attributes */
   DWORD err;                  /* error return from window routines */

   GetStartupInfo(&startInfo); /* get the handle to our console */
   if ((startInfo.dwFlags & STARTF_USESTDHANDLES) == 0)
   {                           /* if we don't use handles from startInfo */
                               /* get what we do use */
      startInfo.hStdOutput = (HANDLE) _get_osfhandle(fileno(stdout));
   }

   coordinates.X = 81;         /* set the new screen buffer size */
   coordinates.Y = 999;
   if (SetConsoleScreenBufferSize(startInfo.hStdOutput, coordinates) != TRUE)
      err = GetLastError();

   rect.Left = 0;              /* set the new window size */
   rect.Top = 0;
   rect.Right = 80;
   rect.Bottom = 40;
   if (SetConsoleWindowInfo(startInfo.hStdOutput, TRUE, &rect) != TRUE)
      err = GetLastError(); 
}
#endif /* _WIN32 */


/*****************************************************************************/
/*                                                                           */
/*                         EstablishThreadEnvironment                        */
/*                                                                           */
/*****************************************************************************/

ARBoolean EstablishThreadEnvironment()
{

#ifdef SINGLE_THREADED
   gTCBPtr = NULL;
#else
#ifdef _WIN32
   if ((gTLSIndex = TlsAlloc()) == 0xFFFFFFFF)
      return FALSE;
#else
#ifdef HPUX10
   if (pthread_keycreate(&gTLSKey, NULL))
#else
   if (pthread_key_create(&gTLSKey, NULL))
#endif /* HPUX10 */
      return FALSE;

#endif /* _WIN32 */
#endif /* SINGLE_THREADED */

   return TRUE;
}


/*****************************************************************************/
/*                                                                           */
/*                          CleanupThreadEnvironment                         */
/*                                                                           */
/*****************************************************************************/

void CleanupThreadEnvironment()
{

#ifndef SINGLE_THREADED
#ifdef _WIN32
   TlsFree(gTLSIndex);
#else
#ifndef HPUX10
   pthread_key_delete(gTLSKey);
#endif /* !HPUX10 */
#endif /* _WIN32 */
#endif /* !SINGLE_THREADED */
}


/*****************************************************************************/
/*                                                                           */
/*                             ProcessCommandLine                            */
/*                                                                           */
/*****************************************************************************/

ARBoolean ProcessCommandLine(argc, argv)
int     argc;        /* IN; number of items in command line */
char   *argv[];      /* IN; array of command line arguments */

{
   int                 i;                     /* working index */
   unsigned int        maxArgumentLen;        /* argument length maximum */
   char                option;                /* option specified */
   char               *tempPtr;               /* working pointer */
   ThreadControlBlock *threadControlBlockPtr; /* control block pointer */

   threadControlBlockPtr = (ThreadControlBlock *) GetThreadControlBlockPtr();

   /* the control structure is zero'd out when created in the thread */
   /* control block.  A zero idType is invalid, so set to something  */
   /* valid here.                                                    */

   for (i = 1; i < argc; i++)
   {                              /* process each entry on the command line */
      if ((argv[i][0] == '-') &&
          (argv[i][1] == 'u' || argv[i][1] == 'p' || argv[i][1] == 'l' ||
           argv[i][1] == 's' || argv[i][1] == 'x'))
         option = argv[i][1];
      
      else
      {                           /* unrecognized option */
         (void) printf("Unrecognized option %s\n", argv[i]);
         return FALSE;
      }
                                  /* have option, load the value */
      if (strlen(argv[i]) > (size_t) 2)
         tempPtr = &(argv[i][2]);
      else
      {                           /* check next argument for name */
         i++;
         if (i < argc)
            tempPtr = argv[i];
         else
         {                        /* no next argument so error */
            (void) printf("Missing value for -%c option\n", option);
            return FALSE;
         }
      }
                                  /* get max argument length */
      switch (option)
      {
         case 'u' :
         case 'p' :
         case 'l' :
         case 's' :
            maxArgumentLen = AR_MAX_NAME_SIZE;
            break;
         case 'x' :
            maxArgumentLen = AR_MAX_FULL_FILENAME;
            break;
      }

      if (strlen(tempPtr) > maxArgumentLen)
      {                           /* argument too long so error */
         (void) printf("Value for -%c option is too long: %s\n", option,
                       tempPtr);
         return FALSE;
      }
                                  /* take appropriate action */
      switch (option)
      {
         case 'u' :
            (void) strcpy(threadControlBlockPtr->control.user, tempPtr);
            break;
         case 'p' :
            (void) strcpy(threadControlBlockPtr->control.password, tempPtr);
            break;
         case 'l' :
            (void) strcpy(threadControlBlockPtr->control.language, tempPtr);
            break;
         case 's' :
            (void) strcpy(threadControlBlockPtr->control.server, tempPtr);
            break;
         case 'x' :
            (void) OpenInputFile(tempPtr);
            break;
      }
   }

   return TRUE;
}


/*****************************************************************************/
/*                                                                           */
/*                                PrintMainMenu                              */
/*                                                                           */
/*****************************************************************************/

void PrintMainMenu()

{
                                     /* print main command menu */
   (void) printf("\n");
   (void) printf(" Container     Schema          Char Menu     Schema Field    <mult>\n");
   (void) printf(" ----------    ------          ---------     ------------    ------\n");
   (void) printf(" get    (gco)  get    (gs)     get    (gc)   get      (gsf)\n");
   (void) printf(" set    (sco)  set    (ss)     set    (sc)   set      (ssf)\n");
   (void) printf(" create (cco)  create (cs)     create (cc)   create   (csf)\n");
   (void) printf(" delete (dco)  delete (ds)     delete (dc)   delete   (dsf)  (dmsf)\n");
   (void) printf(" getlist(glco) getlist(gls)    getlist(glc)  getlist  (glsf)\n");
   (void) printf("                               expand (ec)\n");
   (void) printf("\n");
   (void) printf(" Active Link   Filter          Escalation    Entry           <mult>\n");
   (void) printf(" -----------   ------          ----------    -----           ------\n");
   (void) printf(" get    (gal)  get    (gf)     get    (ges)  get       (ge)  (gme)\n");
   (void) printf(" set    (sal)  set    (sf)     set    (ses)  set       (se)\n");
   (void) printf(" create (cal)  create (cf)     create (ces)  create    (ce)\n");
   (void) printf(" delete (dal)  delete (df)     delete (des)  delete    (de)\n");
   (void) printf(" getlist(glal) getlist(glf)    getlist(gles) getlist   (gle)\n");
   (void) printf("                                             getlistw/f(glewf)\n");
   (void) printf("                                             merge     (me)\n");
   (void) printf(" VUI           Misc Lists      Thread        stats     (stat)\n");
   (void) printf(" ---           ----------      ------\n");
   (void) printf(" get    (gv)   server(svr)     launch          (lt)\n");
   (void) printf(" set    (sv)   group (glg)     launch waiting  (lwt)\n");
   (void) printf(" create (cv)   user  (glu)     release waiting (rwt)\n");
   (void) printf(" delete (dv)   sql   (glsql)   sleep timer     (st)\n");
   (void) printf(" getlist(glsv)\n");
   (void) printf("\n");
   (void) printf(" Init/Term     Control/Logging Info          Misc            Misc\n");
   (void) printf(" ---------     --------------- ----          ----            ----\n");
   (void) printf(" init   (init) record   (rec)  get svr (gsi) ver user (ver)  get file  (gfl)\n");
   (void) printf(" term   (term) stop rec (srec) set svr (ssi) export   (exp)  set file  (sfl)\n");
   (void) printf(" help   (h, ?) open out (oout) get FT  (gft) import   (imp)  get BLOB  (geb)\n");
   (void) printf(" exit   (e, q) close out(cout) set FT  (sft) exec proc(proc) get errmsg(gem)\n");
   (void) printf(" login  (log)  execute  (ex)   get stat(gss) load qual(lqs)\n");
   (void) printf("                                             set port (ssp)\n");
}


/*****************************************************************************/
/*                                                                           */
/*                                GetNextCommand                             */
/*                                                                           */
/*****************************************************************************/

int GetNextCommand(
char **args                 /* pointer to arguments */
)

{
   int  commandCode;        /* code for the command requested */
   int  i;                  /* working index */
   ThreadControlBlock *threadControlBlockPtr; /* control block pointer */

   threadControlBlockPtr = (ThreadControlBlock *) GetThreadControlBlockPtr();
   threadControlBlockPtr->args[0] = '\0';
   *args = NULL;

   (void) printf("\nCommand: ");

   commandCode = -99;
   while (commandCode == -99)
   {                               /* read and validate the command */
      char *blank;

      GetInputLine();
      blank = strchr(threadControlBlockPtr->buffer, ' ');
      if (blank)
      {
         *blank++ = '\0';
         strcpy(threadControlBlockPtr->args, blank);
         *args = threadControlBlockPtr->args;
      }
      if ((strlen(threadControlBlockPtr->buffer) == 0) ||
          (threadControlBlockPtr->buffer[0] == '#'))
         (void) printf("\nCommand: "); /* blank line or comment */
      else if (strlen(threadControlBlockPtr->buffer) > (size_t) 5)
         (void) printf(" *** Command too long, unrecognized ***\nCommand: ");
      else if ((strcmp(threadControlBlockPtr->buffer, "h") == 0) ||
               (strcmp(threadControlBlockPtr->buffer, "?") == 0))
      {
         PrintMainMenu();
         (void) printf("\nCommand: ");
      }
      else if ((strcmp(threadControlBlockPtr->buffer, "e") == 0) ||
               (strcmp(threadControlBlockPtr->buffer, "q") == 0) ||
               (strcmp(threadControlBlockPtr->buffer, "x") == 0))
         commandCode = COMMAND_EXIT;
      else
      {
         for (i = 0; i <= MAX_COMMAND; i++)
            if (strcmp(threadControlBlockPtr->buffer, commands[i]) == 0)
               break;             /* found a match so have a command */
         if (i <= MAX_COMMAND)
            commandCode = i;
         else
            (void) printf(" *** Command not recognized ***\nCommand: ");
      }
   }

   return commandCode;
}


/*****************************************************************************/
/*                                                                           */
/*                               ProcessCommands                             */
/*                                                                           */
/*****************************************************************************/

void ProcessCommands()

{
   int  commandCode;        /* code for the command requested */
   char *args;              /* pointer to arguments */

   /* process commands until exit specified */

   while ((commandCode = GetNextCommand(&args)) != COMMAND_EXIT)
   {                              
      switch (commandCode)
      {
         case COMMAND_LOGIN             :
            GetARControlStruct();
            break;
         case COMMAND_GET_ENTRY         :
            APIARGetEntry();
            break;
         case COMMAND_SET_ENTRY         :
            APIARSetEntry();
            break;
         case COMMAND_CREATE_ENTRY      :
            APIARCreateEntry();
            break;
         case COMMAND_DELETE_ENTRY      :
            APIARDeleteEntry();
            break;
         case COMMAND_GETLIST_ENTRY     :
            APIARGetListEntry();
            break;
         case COMMAND_GETLIST_ENTRY_WITH_FIELDS     :
            APIARGetListEntryWithFields();
            break;
         case COMMAND_GET_FILTER        :
            APIARGetFilter();
            break;
         case COMMAND_SET_FILTER        :
            APIARSetFilter();
            break;
         case COMMAND_CREATE_FILTER     :
            APIARCreateFilter();
            break;
         case COMMAND_DELETE_FILTER     :
            APIARDeleteFilter();
            break;
         case COMMAND_GETLIST_FILTER    :
            APIARGetListFilter();
            break;
         case COMMAND_GET_ESCALATION        :
            APIARGetEscalation();
            break;
         case COMMAND_SET_ESCALATION        :
            APIARSetEscalation();
            break;
         case COMMAND_CREATE_ESCALATION     :
            APIARCreateEscalation();
            break;
         case COMMAND_DELETE_ESCALATION     :
            APIARDeleteEscalation();
            break;
         case COMMAND_GETLIST_ESCALATION    :
            APIARGetListEscalation();
            break;
         case COMMAND_GETLIST_GROUP     :
            APIARGetListGroup();
            break;
         case COMMAND_GET_SCHEMA        :
            APIARGetSchema();
            break;
         case COMMAND_SET_SCHEMA        :
            APIARSetSchema();
            break;
         case COMMAND_CREATE_SCHEMA     :
            APIARCreateSchema();
            break;
         case COMMAND_DELETE_SCHEMA     :
            APIARDeleteSchema();
            break;
         case COMMAND_GETLIST_SCHEMA    :
            APIARGetListSchema();
            break;
         case COMMAND_GET_SCH_FIELD     :
            APIARGetField();
            break;
         case COMMAND_SET_SCH_FIELD     :
            APIARSetField();
            break;
         case COMMAND_CREATE_SCH_FIELD  :
            APIARCreateField();
            break;
         case COMMAND_DELETE_SCH_FIELD  :
            APIARDeleteField();
            break;
         case COMMAND_GETLIST_SCH_FIELD :
            APIARGetListField();
            break;
         case COMMAND_GET_CHAR_MENU     :
            APIARGetCharMenu();
            break;
         case COMMAND_SET_CHAR_MENU     :
            APIARSetCharMenu();
            break;
         case COMMAND_CREATE_CHAR_MENU  :
            APIARCreateCharMenu();
            break;
         case COMMAND_DELETE_CHAR_MENU  :
            APIARDeleteCharMenu();
            break;
         case COMMAND_GETLIST_CHAR_MENU : 
            APIARGetListCharMenu();
            break;
         case COMMAND_GET_VUI           :
            APIARGetVUI();
            break;
         case COMMAND_SET_VUI           :
            APIARSetVUI();
            break;
         case COMMAND_CREATE_VUI        :
            APIARCreateVUI();
            break;
         case COMMAND_DELETE_VUI        :
            APIARDeleteVUI();
            break;
         case COMMAND_GETLIST_VUI       :
            APIARGetListVUI();
            break;
         case COMMAND_EXPORT            :
            APIARExport();
            break;
         case COMMAND_IMPORT            :
            APIARImport();
            break;
         case COMMAND_GET_SERVER_INFO   :
            APIARGetServerInfo();
            break;
         case COMMAND_VERIFY_USER       :
            APIARVerifyUser();
            break;
         case COMMAND_EXECUTE           :
            OpenInputFile(args);
            break;
         case COMMAND_OPEN_OUT          :
            OpenOutputFile();
            break;
         case COMMAND_CLOSE_OUT         :
            CloseOutputFile();
            break;
         case COMMAND_RECORD            :
            StartRecording();
            break;
         case COMMAND_STOP_RECORD       :
            StopRecording();
            break;
         case COMMAND_LAUNCH_THREAD :
            LaunchThread(FALSE);
            break;
         case COMMAND_LAUNCH_WAITING_THREAD :
            LaunchThread(TRUE);
            break;
         case COMMAND_RELEASE_WAITING_THREADS :
            ReleaseWaitingThreads();
            break;
         case COMMAND_SLEEP_TIMER       :
            SleepTimer();
            break;
         case COMMAND_GETLIST_SERVER    :
            APIARGetListServer();
            break;
         case COMMAND_INITIALIZATION    :
            APIARInitialization();
            break;
         case COMMAND_TERMINATION       :
            APIARTermination();
            break;
         case COMMAND_GET_ACTIVE_LINK   :
            APIARGetActiveLink();
            break;
         case COMMAND_SET_ACTIVE_LINK   :
            APIARSetActiveLink();
            break;
         case COMMAND_CREATE_ACTIVE_LINK:
            APIARCreateActiveLink();
            break;
         case COMMAND_DELETE_ACTIVE_LINK:
            APIARDeleteActiveLink();
            break;
         case COMMAND_GETLIST_ACTIVE_LINK:
            APIARGetListActiveLink();
            break;
         case COMMAND_MERGE_ENTRY       :
            APIARMergeEntry();
            break;
         case COMMAND_LOAD_AR_QUAL_STRUCT:
            APIARLoadARQualifierStruct();
            break;
         case COMMAND_EXPAND_CHAR_MENU   :
            APIARExpandCharMenu();
            break;
         case COMMAND_SET_SERVER_INFO    :
            APIARSetServerInfo();
            break;
         case COMMAND_GETLIST_USER      :
            APIARGetListUser();
            break;
         case COMMAND_ENTRY_STATISTICS  :
            APIARGetEntryStatistics();
            break;
         case COMMAND_SETFULLTEXT_INFO   :
            APIARSetFullTextInfo ();
            break;
         case COMMAND_GETFULLTEXT_INFO   :
            APIARGetFullTextInfo ();
            break;
         case COMMAND_GET_SERVER_STAT   :
            APIARGetServerStatistics();
            break;
         case COMMAND_GETLIST_SQL       :
            APIARGetListSQL();
            break;
         case COMMAND_DELETE_MULTI_FIELD:
            APIARDeleteMultipleFields();
            break;
         case COMMAND_EXECUTE_PROCESS   :
            APIARExecuteProcess();
            break;
         case COMMAND_SET_SERVER_PORT   :
            APIARSetServerPort();
            break;
         case COMMAND_GET_MULTIPLE_ENTRY :
            APIARGetMultipleEntries();
            break;
         case COMMAND_GET_SUPPORT_FILE :
            APIARGetSupportFile();
            break;
         case COMMAND_SET_SUPPORT_FILE :
            APIARSetSupportFile();
            break;
         case COMMAND_CREATE_SUPPORT_FILE :
            APIARCreateSupportFile();
            break;
         case COMMAND_DELETE_SUPPORT_FILE :
            APIARDeleteSupportFile();
            break;
         case COMMAND_GETLIST_SUPPORT_FILE :
            APIARGetListSupportFile();
            break;
         case COMMAND_GETENTRY_BLOB :
            APIARGetEntryBLOB();
            break;
         case COMMAND_GET_CONTAINER :
            APIARGetContainer();
            break;
         case COMMAND_SET_CONTAINER :
            APIARSetContainer();
            break;
         case COMMAND_CREATE_CONTAINER :
            APIARCreateContainer();
            break;
         case COMMAND_DELETE_CONTAINER :
            APIARDeleteContainer();
            break;
         case COMMAND_GETLIST_CONTAINER :
            APIARGetListContainer();
            break;
         case COMMAND_GET_ERROR_MESSAGE :
            APIARGetTextForErrorMessage();
            break;
         default                        :
            (void) printf(" **** No support for this command (%d) in driver \n",
                          commandCode);
            break;
      }
   }
}


/*****************************************************************************/
/*                                                                           */
/*                           ThreadStartFunction                             */
/*                                                                           */
/*****************************************************************************/

#ifdef _WIN32
void __stdcall ThreadStartFunction(void * threadStartInfoArg)
#else
void *ThreadStartFunction(void * threadStartInfoArg)
#endif /* _WIN32 */

{
#ifndef SINGLE_THREADED
   ThreadControlBlock * threadControlBlockPtr; /* control block pointer */
   ThreadStartInfo    * threadStartInfoPtr;    /* start info pointer */

   threadStartInfoPtr = (ThreadStartInfo *) threadStartInfoArg;

   /* create a control block for this newly created thread */

   if ((threadControlBlockPtr = CreateThreadControlBlock()) == NULL)
      return;

   /* transfer the input and output file pointers to the control block */

   threadControlBlockPtr->inFile = threadStartInfoPtr->inFile;
   threadControlBlockPtr->outFile = threadStartInfoPtr->outFile;

   /* if appropriate notify the parent thread that we are ready to start */
   /* processing commands and then wait until we are released to do so   */

   if (threadStartInfoPtr->waitFlag)
   {
#ifdef _WIN32
      SetEvent(threadStartInfoPtr->launchWaitEvent);
      WaitForSingleObject(threadStartInfoPtr->releaseWaitEvent, INFINITE);
#else
      pthread_mutex_lock(threadStartInfoPtr->launchWaitMutexPtr);
      pthread_cond_signal(threadStartInfoPtr->launchWaitCondPtr);
      pthread_mutex_unlock(threadStartInfoPtr->launchWaitMutexPtr);

      pthread_mutex_lock(threadStartInfoPtr->releaseWaitMutexPtr);
      pthread_cond_wait(threadStartInfoPtr->releaseWaitCondPtr,
                        threadStartInfoPtr->releaseWaitMutexPtr);
      pthread_mutex_unlock(threadStartInfoPtr->releaseWaitMutexPtr);
#endif /* _WIN32 */
   }
   
   /* free the memory used to facilitate the transfer of start information */

   free((char *) threadStartInfoPtr);

   /* process the commands contained in the input file */

   ProcessCommands();

   /* remove the control block for this thread that is about to terminate */

   DestroyThreadControlBlock();
#endif /* !SINGLE_THREADED */
}


/*****************************************************************************/
/*                                                                           */
/*                               LaunchThread                                */
/*                                                                           */
/*****************************************************************************/

void LaunchThread(ARBoolean waitFlag)

{
#ifndef SINGLE_THREADED
   char               *filename; /* pointer to filename for record file */
   FILE               *tempInFile; /* temporary pointer to new input file */
   FILE               *tempOutFile; /* temporary pointer to new output file */
   ThreadControlBlock *threadControlBlockPtr; /* control block pointer */
#ifdef _WIN32
   HANDLE              threadHandle;
   unsigned            threadId;
#else
#ifdef HPUX10
   pthread_attr_t      nullAttr = pthread_attr_default;
#else
   pthread_attr_t      *nullAttr = NULL;
#endif /* HPUX10 */
   pthread_t           threadHandle;
#endif /* !_WIN32 */
   ThreadStartInfo    *threadStartInfoPtr; /* start information pointer */

   /* get the filename of the required input file */

   filename = GetChar("Filename of input file (): ", "");
   if (strlen(filename) == 0)
   {
      (void) printf(" **** An input file is required to launch a thread\n");
      return;
   }

   /* open the input file for reading */

   tempInFile = fopen(filename, "r");
   if (tempInFile == NULL)
   {
      (void) printf(" **** File error during open; input file is invalid\n");
      return;
   }

   /* get the filename of the optional output file */

   filename = GetChar("Filename of output file (): ", "");
   if (strlen(filename) == 0)
      tempOutFile = stdout;
   else
   {
      /* open the output file for writing */

      tempOutFile = fopen(filename, "w");
      if (tempOutFile == NULL)
      {
         (void) printf(
                    " **** File error during open; output file is invalid\n");
         (void) fclose(tempInFile);

         return;
      }
   }

   /* allocate a thread start info structure that will be used to facilitate */
   /* the transfer of the file pointers to the newly create thread           */

   threadStartInfoPtr = (ThreadStartInfo *) malloc (sizeof(ThreadStartInfo));
   if (threadStartInfoPtr == NULL)
   {
      (void) printf(" **** malloc error creating thread start info\n");
      (void) fclose(tempInFile);
      if (tempOutFile != stdout)
         (void) fclose(tempOutFile);
      return;
   }

   /* place the file pointers in the thread start info structure */

   threadStartInfoPtr->inFile = tempInFile;
   threadStartInfoPtr->outFile = tempOutFile;

   /* get a pointer to the current thread's control block */

   threadControlBlockPtr = (ThreadControlBlock *) GetThreadControlBlockPtr();

   /* place the synchronization info in the thread start info structure if */
   /* the launched thread is expected to wait                              */

   threadStartInfoPtr->waitFlag = waitFlag;

   if (waitFlag)
   {
#ifdef _WIN32
      threadStartInfoPtr->launchWaitEvent =
         threadControlBlockPtr->launchWaitEvent;
      threadStartInfoPtr->releaseWaitEvent =
         threadControlBlockPtr->releaseWaitEvent;
#else
      threadStartInfoPtr->launchWaitMutexPtr =
         &threadControlBlockPtr->launchWaitMutex;
      threadStartInfoPtr->launchWaitCondPtr =
         &threadControlBlockPtr->launchWaitCond;
      threadStartInfoPtr->releaseWaitMutexPtr =
         &threadControlBlockPtr->releaseWaitMutex;
      threadStartInfoPtr->releaseWaitCondPtr =
         &threadControlBlockPtr->releaseWaitCond;
#endif /* _WIN32 */
   }

   /* start the new thread passing the information to the start function */

#ifdef _WIN32
   if ((threadHandle = (HANDLE) _beginthreadex(NULL, 0,
                                               (unsigned int 
                                                (__stdcall *)(void *))
                                               ThreadStartFunction,
                                               (void *) threadStartInfoPtr, 0,
                                               &threadId)) == 0)
#else
   if (pthread_create(&threadHandle, nullAttr, ThreadStartFunction,
                      (void *) threadStartInfoPtr) != 0)
#endif /* _WIN32 */
   {
      (void) printf(" **** unable to start a new thread\n");
      free((char *) threadStartInfoPtr);
      (void) fclose(tempInFile);
      if (tempOutFile != stdout)
         (void) fclose(tempOutFile);
   }

   /* if appropriate wait until the launched thread has communicated that it */
   /* is poised and ready to start reading its command file upon release     */

   if (waitFlag)
   {
#ifdef _WIN32
      WaitForSingleObject(threadControlBlockPtr->launchWaitEvent, INFINITE);
#else
      pthread_mutex_lock(&threadControlBlockPtr->launchWaitMutex);
      pthread_cond_wait(&threadControlBlockPtr->launchWaitCond,
                        &threadControlBlockPtr->launchWaitMutex);
      pthread_mutex_unlock(&threadControlBlockPtr->launchWaitMutex);
#endif /* _WIN32 */
   }

   /* save the thread handle for the newly created thread and increment */
   /* the number of handles unless there is insufficient space          */

   if (threadControlBlockPtr->numHandles < MAX_THREAD_HANDLES)
      threadControlBlockPtr->threadHandles[threadControlBlockPtr->
                                            numHandles++] = threadHandle;

#else
   (void) printf(" **** Command not allowed in single threaded driver\n");
#endif /* !SINGLE_THREADED */
}


/*****************************************************************************/
/*                                                                           */
/*                                     main                                  */
/*                                                                           */
/*****************************************************************************/

int main(argc, argv)
int     argc;        /* IN; number of items in command line */
char   *argv[];      /* IN; array of command line arguments */

{
   ThreadControlBlock * threadControlBlockPtr; /* control block pointer */

#ifdef _WIN32
   SetConsoleSize();        /* make console big enough for the help screen */
#endif /* _WIN32 */

   if (!EstablishThreadEnvironment())
      return -1;

   if ((threadControlBlockPtr = CreateThreadControlBlock()) == NULL)
      return -1;

   threadControlBlockPtr->inFile = stdin;
   threadControlBlockPtr->outFile = stdout;

   if (!ProcessCommandLine(argc, argv))
      return -1;

   (void) printf("\n\n     AR System API Driver\n");
   PrintMainMenu();

   ProcessCommands();

   DestroyThreadControlBlock();

   CleanupThreadEnvironment();

   return 0;
}
